import { ormDB, usefulLink } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog, LogType } from "./userOptLog.service";


let _ = require('lodash');
let { logger } = require('@madex/ex-js-public');


export interface UsefulLinkVO {
    id?: number;

    name?: string | any;

    url?: string | any;

    type?: number | any;

    weight?: number;

    is_active?: number;

    status?: number;

    time_per_day?: number;

    comment?: string | any;

    createdAt?: Date | any;

    updatedAt?: Date | any;

}

export interface UsefulLinkPageVO extends UsefulLinkVO {
    page?: number,

    size?: number
}

export async function list(usefulLinkPageVO: UsefulLinkPageVO) {
    let where = Object.create(null);
    if (usefulLinkPageVO.name) {
        where.name = { [ormDB.Op.like]: `${usefulLinkPageVO.name}%` };
    }

    if (usefulLinkPageVO.url) {
        where.url = { [ormDB.Op.like]: `${usefulLinkPageVO.url}%` };
    }
    if (usefulLinkPageVO.type) {
        if (!isNaN(Number(usefulLinkPageVO.type))){
            where.type = usefulLinkPageVO.type;
        }else {
            let typeList = usefulLinkPageVO.type.split(",");
            where.type = { [ormDB.Op.in]: typeList };
        }
    }
    if (!isNaN(Number(usefulLinkPageVO.is_active))) {
        where.is_active = usefulLinkPageVO.is_active;
    }


    let resList = await usefulLink.prototype.findAndCount({
        where: where,
        limit: usefulLinkPageVO.size,
        offset: (Number(usefulLinkPageVO.page) - 1) * Number(usefulLinkPageVO.size),
        order: [["weight", "desc"]],
        raw: true
    });
    return resList;
}

export async function add(usefulLinkVO: UsefulLinkVO, currentUserId: any, ip: string | undefined) {
    usefulLinkVO.createdAt = new Date();
    usefulLinkVO.updatedAt = new Date();

    let dbInfo = await usefulLink.prototype.create(usefulLinkVO);

    //日志
    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},添加链接记录:${JSON.stringify(dbInfo.id)}`, LogType.ADD);
    return 'success'
}


export async function update(usefulLinkVO: UsefulLinkVO, currentUserId: any, ip: string | undefined) {

    let dbInfo = await usefulLink.prototype.findOne({
        where: {
            id: usefulLinkVO.id
        },
        raw: true
    });

    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let updateInfo = {
        updatedAt: new Date()
    };
    if (usefulLinkVO.name) {
        updateInfo["name"] = usefulLinkVO.name
    }

    if (usefulLinkVO.url) {
        updateInfo["url"] = usefulLinkVO.url;
    }

    if (usefulLinkVO.type) {
        updateInfo["type"] = usefulLinkVO.type;
    }

    if (usefulLinkVO.weight) {
        updateInfo["weight"] = usefulLinkVO.weight;
    }

    if (!isNaN(Number(usefulLinkVO.status))) {
        updateInfo["status"] = usefulLinkVO.status;
    }

    if (!isNaN(Number(usefulLinkVO.time_per_day))) {
        updateInfo["time_per_day"] = usefulLinkVO.time_per_day;
    }

    if (!isNaN(Number(usefulLinkVO.is_active))) {
        updateInfo["is_active"] = usefulLinkVO.is_active;
    }
    if (usefulLinkVO.comment) {
        updateInfo["comment"] = usefulLinkVO.comment;
    }

    await usefulLink.prototype.update(updateInfo, {
        where: {
            id: Number(dbInfo.id)
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},修改链接记录:原记录:${JSON.stringify(dbInfo.id)}`, LogType.UPDATE);
    return 'success'
}


export async function del(id: number, currentUserId: any, ip: string | undefined) {

    let dbInfo = await usefulLink.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });

    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST;
    }

    await usefulLink.prototype.destroy({
        where: {
            id: Number(id)
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},删除链接记录:${JSON.stringify(dbInfo.id)}`, LogType.DEL);
    return 'success'
}

export async function detail(id: number) {

    let dbInfo = await usefulLink.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    return dbInfo;
}



