import { madAdminOrmDB, coinAddress, coinType, abkFeeStatistics, ormDB, coinWithdraw } from "@madex/ex-ts-dao";
import BigNumber from "bignumber.js";
import { any } from "async";


const Otplib = require('otplib');

let { logger, apiAssertUtils: ApiAssert, BigNumberUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, tickerUtils } = require('@madex/ex-js-common');

let _ = require('lodash');

/**
 * 首页数据概览
 * @param from_time
 * @param to_time
 */
export async function gatherData(from_time: any, to_time: any) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }]
    }

    let condition: any = {
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: to_time },
            }],
        status: 3//发币完成
    }

    //手续费和返佣
    let task1 = abkFeeStatistics.prototype.findAll({
        attributes: ['user_type',
            ormDB.literal('sum(spot_fee_usdt) as spot_fee_usdt'),
            ormDB.literal('sum(lpc_fee_usdt) as lpc_fee_usdt'),
            ormDB.literal('sum(ipc_fee_usdt) as ipc_fee_usdt'),
            ormDB.literal('sum(spot_back_usdt) as spot_back_usdt'),
            ormDB.literal('sum(lpc_back_usdt) as lpc_back_usdt'),
            ormDB.literal('sum(ipc_back_usdt) as ipc_back_usdt')
        ],
        where: where,
        group: ['user_type'],
        order: [['user_type', 'asc']],
        raw: true
    });

    let [dbFeeList, { total_withdraw_fee, tickerMap }] = await Promise.all([task1, getTotalWithdrawFeeByCondition(condition)]);

    //普通用户数据
    let commonData = dbFeeList[0] ? dbFeeList[0] : {};
    //代理用户（crm）用户数据
    let crmData = dbFeeList[1] ? dbFeeList[1] : {};

    let spot_data = buildOneResult(commonData.spot_fee_usdt, commonData.spot_back_usdt, crmData.spot_fee_usdt, crmData.spot_back_usdt);
    let lpc_data = buildOneResult(commonData.lpc_fee_usdt, commonData.lpc_back_usdt, crmData.lpc_fee_usdt, crmData.lpc_back_usdt);
    let ipc_data = buildOneResult(commonData.ipc_fee_usdt, commonData.ipc_back_usdt, crmData.ipc_fee_usdt, crmData.ipc_back_usdt);

    let all_common_fee = new BigNumber(commonData.spot_fee_usdt).add(new BigNumber(commonData.lpc_fee_usdt)).add(new BigNumber(commonData.ipc_fee_usdt));
    let all_common_back = new BigNumber(commonData.spot_back_usdt).add(new BigNumber(commonData.lpc_back_usdt)).add(new BigNumber(commonData.ipc_back_usdt));

    let all_crm_fee = new BigNumber(crmData.spot_fee_usdt).add(new BigNumber(crmData.lpc_fee_usdt)).add(new BigNumber(crmData.ipc_fee_usdt));
    let all_crm_back = new BigNumber(crmData.spot_back_usdt).add(new BigNumber(crmData.ipc_back_usdt)).add(new BigNumber(crmData.ipc_back_usdt));

    let all_data = buildOneResult(all_common_fee, all_common_back, all_crm_fee, all_crm_back);
    //总手续费+提现手续费
    all_data.total_fee = all_data.total_fee.add(total_withdraw_fee);
    return {
        all_data,
        spot_data,
        lpc_data,
        ipc_data,
        withdraw_data: total_withdraw_fee
    }

}

/**
 * 手续费数据
 * @param from_time
 * @param to_time
 * @param page
 * @param size
 * @param type  1 现货 2 U本位合约 3 币本位合约 4 全部
 */
export async function getFeeData(from_time: any, to_time: any, page: any, size: any, type: number) {
    let [total_obj, pageData] = await Promise.all([getTotalFeeDataOfPeriodByType(from_time, to_time, type),
        getTotalFeePageDataOfPeriodByType(page, size, from_time, to_time, type)]);
    pageData["total_data"] = total_obj;
    return pageData;
}

/**
 * 提现手续费数据
 * @param from_time
 * @param to_time
 * @param page
 * @param size
 */
export async function withdrawData(from_time: any, to_time: any, page: any, size: any) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: to_time },
            }],
        status: 3
    }


    let task2 = coinWithdraw.prototype.findAndCount({
        attributes: ['coin_symbol', 'amount', 'fee', 'status'],
        where: where,
        limit: Number(size),
        offset: (Number(page) - 1) * Number(size),
        order: [['updatedAt', 'desc']],
        raw: true
    });

    let [{ total_withdraw_fee, tickerMap }, pageData] = await Promise.all([getTotalWithdrawFeeByCondition(where), task2]);

    for (let item of pageData.rows) {
        item.amount_usdt = new BigNumber(item.amount).mul(new BigNumber(tickerMap[item.coin_symbol] || 0));
        item.fee_usdt = new BigNumber(item.fee).mul(new BigNumber(tickerMap[item.coin_symbol] || 0));
    }
    pageData['total_withdraw_fee'] = total_withdraw_fee;

    return pageData;
}

function buildOneResult(common_fee: any, common_back: any, crm_fee: any, crm_back: any) {
    return {
        total_fee: new BigNumber(common_fee || 0).add(new BigNumber(crm_fee || 0)),
        total_back: new BigNumber(common_back || 0).add(new BigNumber(crm_back || 0)),
        common_user: {
            fee: new BigNumber(common_fee || 0),
            back: new BigNumber(common_back || 0)
        },
        crm_user: {
            fee: new BigNumber(crm_fee || 0),
            back: new BigNumber(crm_back || 0),
        }
    }
}

/**
 * 根据类型查询一段时间内的总手续费
 * @param from_time
 * @param to_time
 * @param type 1 现货 2 U本位合约 3 币本位合约 4 全部
 */
async function getTotalFeeDataOfPeriodByType(from_time: any, to_time: any, type: number) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }]
    }
    let { fee_field, back_field } = getFields(type);

    let dbList = await abkFeeStatistics.prototype.findAll({
        attributes: ['user_type',
            ormDB.literal(`sum(\`${fee_field}\`) as total_fee_usdt`),
            ormDB.literal(`sum(\`${back_field}\`) as total_back_usdt`),
        ],
        where: where,
        group: ['user_type'],
        order: [['user_type', 'asc']],
        raw: true
    });

    let commonUserData = dbList[0] ? dbList[0] : {};
    let crmUserData = dbList[1] ? dbList[1] : {};

    return buildOneResult(commonUserData.total_fee_usdt, commonUserData.total_back_usdt, crmUserData.total_fee_usdt, crmUserData.total_back_usdt);
}


/**
 * 根据类型 查询一段时间内的手续费 分页数据
 * @param page
 * @param size
 * @param from_time
 * @param to_time
 * @param type 1 现货 2 U本位合约 3 币本位合约 4 全部
 */
async function getTotalFeePageDataOfPeriodByType(page: any, size: any, from_time: any, to_time: any, type: number) {
    let res: any;
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }]
    }
    let { fee_field, back_field } = getFields(type);


    res = await abkFeeStatistics.prototype.findAndCount({
        attributes: ['trade_date', 'user_type',
            ormDB.literal(`sum(\`${fee_field}\`) as total_fee_usdt`),
            ormDB.literal(`sum(\`${back_field}\`) as total_back_usdt`),
        ],
        where: where,
        limit: Number(size),
        offset: (Number(page) - 1) * Number(size),
        group: ['trade_date', 'user_type'],
        order: [['trade_date', 'desc'], ['user_type', 'asc']],
        raw: true
    });
    res.count = res.count.length;
    return res;
}

/**
 * 根据类型获取要查询的字段
 * @param type
 */
function getFields(type: number) {
    let fee_field = "";
    let back_field = "";
    switch (type) {
        case 1:
            fee_field = "spot_fee_usdt";
            back_field = "spot_back_usdt";
            break;
        case 2:
            fee_field = "lpc_fee_usdt";
            back_field = "lpc_back_usdt";
            break;
        case 3:
            fee_field = "ipc_fee_usdt";
            back_field = "ipc_back_usdt";
            break;
        default: // 这里拼接字符串 注意 ` 符号
            fee_field = "spot_fee_usdt` + `lpc_fee_usdt` + `ipc_fee_usdt";
            back_field = "spot_back_usdt` + `lpc_back_usdt` + `ipc_back_usdt";
            break;
    }

    return {
        fee_field, back_field
    }
}


/**
 * 根据条件查询总提现手续费
 * @param where
 */
async function getTotalWithdrawFeeByCondition(where: any) {
    //提现手续费
    let dbList = await coinWithdraw.prototype.findAll({
        attributes: ['coin_symbol',
            ormDB.literal('sum(fee) as fee')
        ],
        where: where,
        group: ['coin_symbol'],
        raw: true
    });
    let tickerMap = {};
    let total_withdraw_fee = new BigNumber(0);
    for (let item of dbList) {
        let usdt = await tickerUtils.rateCoin2USDT(item.coin_symbol);
        let fee_usdt = new BigNumber(item.fee).mul(new BigNumber(usdt));
        total_withdraw_fee = total_withdraw_fee.add(fee_usdt);
        tickerMap[item.coin_symbol] = usdt;
    }
    return {
        total_withdraw_fee, tickerMap
    }
}



