import * as systemTriggerService from "../service/systemTrigger.service";
import { SystemTriggerVO, SystemTriggerPageVO } from "../service/systemTrigger.service";


let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert, datetimeUtils } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUserId } from "../../../utils/aclUserUtils";
import { coinType, spotPairs } from "@madex/ex-ts-dao";

let isIp = require('is-ip');
/**
 * 触发配置列表
 * @param req
 * @param infoVO
 */
export const list = async (req: any, pageVO: SystemTriggerPageVO) => {
    let func_name = "systemTrigger.control.list";
    try {
        pageVO.page = Optional.opt(pageVO, 'page', 1);
        pageVO.size = Optional.opt(pageVO, 'size', 20);
        let res = await systemTriggerService.list(pageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 添加新的触发配置
 * @param req
 * @param authConfigVO
 */
export const add = async (req: any, systemTriggerVO: SystemTriggerVO) => {
    let func_name = "systemTrigger.control.add";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await paramValid(systemTriggerVO);

        let res = await systemTriggerService.add(systemTriggerVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 修改触发配置
 * @param req
 * @param authConfigVO
 */
export const update = async (req: any, systemTriggerVO: SystemTriggerVO) => {
    let func_name = "systemTrigger.control.update";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!systemTriggerVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        await paramValid(systemTriggerVO);

        let res = await systemTriggerService.update(systemTriggerVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 删除触发配置
 * @param req
 * @param authConfigVO
 */
export const del = async (req: any, systemTriggerVO: SystemTriggerVO) => {
    let func_name = "systemTrigger.control.del";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!systemTriggerVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await systemTriggerService.del(systemTriggerVO.id, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

async function paramValid(systemTriggerVO: SystemTriggerVO) {

    let symbolOrPair = systemTriggerVO.trigger_symbol;
    let triggerType = systemTriggerVO.trigger_type;
    if (!symbolOrPair || !triggerType || !systemTriggerVO.trigger_time) {
        throw ErrorCode.PARAM_MISS
    }
    if (triggerType == 1) {
        let dbInfo = await coinType.prototype.findOne({
            where: {
                symbol: symbolOrPair
            },
            raw: true
        });
        if (!dbInfo) {
            throw ErrorCode.COIN_TYPE_NOT_EXIST
        }
    }
    else {
        //TODO:之前查的  ex_pair  现在查询的是  spot_pairs 是否需要补充别的查询合约交易对的逻辑？？？
        let dbInfo = await spotPairs.prototype.findOne({
            where: {
                symbol: symbolOrPair
            },
            raw: true
        });
        if (!dbInfo) {
            throw ErrorCode.PAIR_NOT_EXIST
        }
    }
}

