import { exPairApply, coinType, ormDB, systemTrigger, spotPairs, exTradeArea, exBusinessAreaRouter, contractPairs } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { addCoin2Core, addPairToCore } from "../../../utils/coreSystemUtils";
import { PAIR_APPLY_STATUS } from "../../../constant/pairApplyConst";
import { checkPairInCoinType } from "../../../utils/coinTypeUtils";
import { pairs } from "rxjs";
import * as RobotUtil from "../../../utils/robotUtils";


let _ = require('lodash');
let { logger, datetimeUtils } = require('@madex/ex-js-public');


export interface PairApplyVO {
    id?: number;

    type?: number;

    symbol?: string;

    name?: string;

    business_area_id?: number;

    price_scale?: number;

    quantity_scale?: number;

    maker_fee?: string;

    taker_fee?: string;

    price_increment?: number;

    quantity_increment?: number;

    view_precision?: number

    min_order_size?: number

    max_order_size?: number

    min_order_value?: number

    max_order_value?: number

    settlement_asset?: string;

    funding_times_perday?: number;

    default_initial_margin?: number; //"0.01",

    base_initial_margin?: number;//"0.01",

    base_maintenance_margin?: number;//"0.01",

    base_interest: number; //"0.0003",

    quote_interest?: number; //"0.0006",

    impact_notional?: number; // "600000",

    base_risk_limit?: number; // "1000000",

    risk_limit_step?: number; //"1000000",

    tm_active?: Date | any;

    applyer?: number;

    checker?: number;

    status?: number;

    reason?: string;

    is_active?: number

    is_hide?: number

    createdAt?: Date | any,

    updatedAt?: Date | any,

}


export interface PairApplyPageVO extends PairApplyVO {

    page?: number,

    size?: number,

    symbol?: string;

}


export async function list(pageVO: PairApplyPageVO) {

    let resList = await exPairApply.prototype.findAndCount({
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [['status', 'asc'], ["id", "desc"]],
        raw: true
    });
    return resList;
}

export async function listed(pageVO: PairApplyPageVO) {
    let where = {};
    if (pageVO.symbol) {
        where['symbol'] = pageVO.symbol;
    }
    //TODO:是否查询此表 还是ex_pair ???
    let resList = await spotPairs.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}



export async function apply(pairApplyVO: PairApplyVO, currentUser: any, ip: string | undefined) {

    //检查交易对中的币种是否存在于coin_type 中
    await checkPairInCoinType(String(pairApplyVO.symbol));

    pairApplyVO.applyer = currentUser.account;
    pairApplyVO.status = PAIR_APPLY_STATUS.WAIT_VIEW;
    pairApplyVO.createdAt = new Date();
    pairApplyVO.updatedAt = new Date();
    if (Number(pairApplyVO.id) > 0) {//update
        let dbApply = await exPairApply.prototype.findOne({
            where: { id: pairApplyVO.id },
            raw: true
        });
        if (!dbApply) {
            throw ErrorCode.DATA_NOT_EXIST;
        }
        if (dbApply.status != PAIR_APPLY_STATUS.RE_WRITE) {
            throw ErrorCode.NOT_EDIT
        }
        await exPairApply.prototype.update(pairApplyVO, {
            where: {
                id: Number(pairApplyVO.id)
            }
        });
    }
    else {
        await exPairApply.prototype.create(pairApplyVO);
    }
    sendMsg(pairApplyVO.status, String(pairApplyVO.symbol));
//管理后台操作日志
    addOptLog(currentUser.userId, 0, '新增上新交易对申请', ip, JSON.stringify(pairApplyVO), '交易上下线管理');
    return 'success';
}


export async function edit(pairApplyVO: PairApplyVO, currentUser: any, ip: string | undefined) {

    let dbApply = await exPairApply.prototype.findOne({
        where: { id: pairApplyVO.id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let dbApply2 = await exPairApply.prototype.findOne({
        where: { symbol: pairApplyVO.symbol, id: { [ormDB.Op.ne]: pairApplyVO.id } },
        raw: true
    });
    if (dbApply2) {
        throw ErrorCode.DATA_EXIST;
    }

    //检查交易对中的币种是否存在于coin_type 中
    await checkPairInCoinType(String(pairApplyVO.symbol));

    pairApplyVO.applyer = dbApply.applyer;
    pairApplyVO.updatedAt = new Date();

    await exPairApply.prototype.update(pairApplyVO, {
        where: {
            id: Number(pairApplyVO.id)
        }
    });
    sendMsg(Number(pairApplyVO.status), String(pairApplyVO.symbol));
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '上新交易对申请编辑', ip, JSON.stringify(pairApplyVO), '交易上下线管理');
    return 'success';
}

export async function cancelSelf(id: any, reason: any, currentUser: any, ip: string | undefined) {

    let dbApply = await exPairApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    if (dbApply.status >= PAIR_APPLY_STATUS.PASS) {
        throw ErrorCode.NO_CANCEL;
    }
    await updateApply(Number(id), PAIR_APPLY_STATUS.CANCEL, currentUser.account, String(reason), dbApply.symbol);
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '撤销上新交易对申请', ip, JSON.stringify(dbApply), '交易上下线管理');
    return 'success';
}

export async function rewrite(id: any, reason: any, currentUser: any, ip: string | undefined) {

    let dbApply = await exPairApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    await updateApply(Number(id), PAIR_APPLY_STATUS.RE_WRITE, currentUser.account, String(reason), dbApply.symbol);
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '驳回上新交易对申请', ip, JSON.stringify(dbApply), '交易上下线管理');
    return 'success';
}

export async function cancel(id: any, reason: any, currentUser: any, ip: string | undefined) {

    let dbApply = await exPairApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    await updateApply(Number(id), PAIR_APPLY_STATUS.CANCEL, currentUser.account, String(reason), dbApply.symbol);
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '管理员取消上新交易对申请', ip, JSON.stringify(dbApply), '交易上下线管理');
    return 'success';
}

export async function review(id: any, currentUser: any, ip: string | undefined) {

    let dbApply = await exPairApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let reason = "";
    //1被驳回 2申请待审批 3审批通过 4交易对创建完成 5交易对增加到撮合完成 6等待撮合系统重启 7发布交易对到撮合完成 8交易对激活定时器完成 9取消
    let status = Number(dbApply.status)
    let pair = dbApply.symbol;
    let base = pair.split('_')[0];
    let quote = pair.split('_')[1];
    //1 现货 2 合约
    let type = dbApply.type;
    if (status == PAIR_APPLY_STATUS.WAIT_VIEW) {//2-3
        reason = "审核通过";
        await updateApply(Number(id), PAIR_APPLY_STATUS.PASS, currentUser.account, reason, pair);
    }
    else if (status == PAIR_APPLY_STATUS.PASS) {//3-4
        //检查交易对中的币种是否存在于coin_type 中
        await checkPairInCoinType(pair);
        //检查交易对是否存在
        await checkPairExistBeforeCreate(pair, type);

        let insertInfo = {
            symbol: pair,
            name: pair,
            price_scale: dbApply.price_scale,
            quantity_scale: dbApply.quantity_scale,
            maker_fee: dbApply.maker_fee,
            taker_fee: dbApply.taker_fee,
            price_increment: dbApply.price_increment,
            quantity_increment: dbApply.quantity_increment,
            view_precision: dbApply.view_precision,
            min_order_size: dbApply.min_order_size,
            max_order_size: dbApply.max_order_size,
            min_order_value: dbApply.min_order_value,
            max_order_value: dbApply.max_order_value,
            is_active: dbApply.is_active,
            is_hide: dbApply.is_hide,
            status: 0,
            createdAt: new Date(),
            updatedAt: new Date(),
        }
        //现货
        if (type == 1) {
            insertInfo['base'] = base;
            insertInfo['quote'] = quote;
            await spotPairs.prototype.create(insertInfo);
        }
        else {//合约
            insertInfo['index'] = pair;
            insertInfo['settlement_asset'] = dbApply.settlement_asset;
            insertInfo['funding_times_perday'] = dbApply.funding_times_perday;
            insertInfo['default_initial_margin'] = dbApply.default_initial_margin;
            insertInfo['base_initial_margin'] = dbApply.base_initial_margin;
            insertInfo['base_maintenance_margin'] = dbApply.base_maintenance_margin;
            insertInfo['base_interest'] = dbApply.base_interest;
            insertInfo['quote_interest'] = dbApply.quote_interest;
            insertInfo['impact_notional'] = dbApply.impact_notional;
            insertInfo['base_risk_limit'] = dbApply.base_risk_limit;
            insertInfo['risk_limit_step'] = dbApply.risk_limit_step;
            await contractPairs.prototype.create(insertInfo);
        }

        let areaRouterInfo = {
            pair: pair,
            area_id: dbApply.business_area_id,
            createdAt: new Date(),
            updatedAt: new Date(),
        }
        await exBusinessAreaRouter.prototype.create(areaRouterInfo);
        reason = "交易对创建完成";
        await updateApply(Number(id), PAIR_APPLY_STATUS.CREATE_PAIR, currentUser.account, reason, pair);
    }
    else if (status == PAIR_APPLY_STATUS.CREATE_PAIR) {//4-5
        //检查交易对是否存在
        let dbPair = await checkPairExist(pair, type);
        let param: any;
        //现货
        if (type == 1) {
            param = {
                "base": dbPair.base, "quote": dbPair.quote, "symbol": dbPair.symbol, "name": dbPair.symbol,
                "price_scale": dbPair.price_scale, "quantity_scale": dbPair.quantity_scale,
                "maker_fee": dbPair.maker_fee, "taker_fee": dbPair.taker_fee
            }
        }
        else {
            param = {
                "market": "lpc",
                "base": base, "quote": quote, "symbol": dbPair.symbol, "name": dbPair.symbol,
                "price_scale": dbPair.price_scale, "quantity_scale": dbPair.quantity_scale,
                "maker_fee": dbPair.maker_fee, "taker_fee": dbPair.taker_fee,
                "default_initial_margin": dbPair.default_initial_margin,
                "base_initial_margin": dbPair.base_initial_margin,
                "base_maintenance_margin": dbPair.base_maintenance_margin,
                "settlement_asset": dbPair.settlement_asset,
                "index": dbPair.symbol,
                "base_interest": dbPair.base_interest,
                "quote_interest": dbPair.quote_interest,
                "impact_notional": dbPair.impact_notional,
                "base_risk_limit": dbPair.base_risk_limit,
                "risk_limit_step": dbPair.risk_limit_step,
                "funding_times_perday": dbPair.funding_times_perday
            }
        }
        if (dbPair.min_order_size) {
            param["min_order_size"] = dbPair.min_order_size;
        }
        if (dbPair.max_order_size) {
            param["max_order_size"] = dbPair.max_order_size;
        }
        if (dbPair.min_order_value) {
            param["min_order_value"] = dbPair.min_order_value;
        }
        if (dbPair.max_order_value) {
            param["max_order_value"] = dbPair.max_order_value;
        }
        let optResult = await addPairToCore(param);
        if (!optResult.is_success) {
            RobotUtil.sendRobotMessage(RobotUtil.ROBOT_KEYS.COMMON_KEY, optResult.err_msg);
            throw ErrorCode.ADD_PAIR_TO_CORE_ERR;
        }
        //现货
        if (type == 1) {
            await spotPairs.prototype.update({ status: 1 }, {
                where: {
                    id: dbPair.id
                }
            });
        }
        else {//合约
            await contractPairs.prototype.update({ status: 1 }, {
                where: {
                    id: dbPair.id
                }
            });
        }


        reason = "交易对增加到撮合完成";
        await updateApply(Number(id), PAIR_APPLY_STATUS.ADMIN_ADD_PAIR, currentUser.account, reason, pair);
    }
    else if (status == PAIR_APPLY_STATUS.ADMIN_ADD_PAIR) {//5-6

        reason = "等待撮合系统重启";
        await updateApply(Number(id), PAIR_APPLY_STATUS.WAIT_ADMIN_RESTART, currentUser.account, reason, pair);
    }
    else if (status == PAIR_APPLY_STATUS.WAIT_ADMIN_RESTART) {//6-7

        //检查交易对是否存在
        await checkPairExist(pair, type);
        reason = "发布交易对到撮合完成";
        await updateApply(Number(id), PAIR_APPLY_STATUS.ADMIN_ISSUE_PAIR, currentUser.account, reason, pair);
    }
    else if (status == PAIR_APPLY_STATUS.ADMIN_ISSUE_PAIR) {//7-8
        //检查交易对是否存在
        let dbPair = await checkPairExist(pair, type);
        let tm = datetimeUtils.add(new Date(), datetimeUtils.SECONED);
        let tm_active = dbApply.tm_active;
        if (datetimeUtils.between(tm_active, tm) < 0) {
            throw ErrorCode.ACTIVE_TM_EXPIRE;
        }
        if (!dbPair.is_active){
            await systemTrigger.prototype.create({
                trigger_symbol: pair,
                trigger_type: 2,
                trigger_action: 2011,
                trigger_time: tm_active,
                status: 0,
                createdAt: new Date(),
                updatedAt: new Date(),
            });
        }
        reason = "交易对激活定时器完成";
        await updateApply(Number(id), PAIR_APPLY_STATUS.TIMER_ACTIVE, currentUser.account, reason, pair);
    }
    else {
        throw ErrorCode.CURRENT_STATUS_NOT_APPLY
    }

    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '上新交易对申请审核', ip, `msg:${reason},dbInfo:${dbApply.id}`, '交易上下线管理');
    return 'success';
}

async function updateApply(id: number, status: number, checker: string, reason: string, symbol: string) {
    let updateInfo = {
        status: status,
        checker: checker,
        reason: reason ? reason : "",
        updatedAt: new Date()
    }
    await exPairApply.prototype.update(updateInfo, {
        where: {
            id: id
        }
    });
    sendMsg(status, symbol);
}

async function sendMsg(status: number, symbol: string) {
    // 1被驳回2申请待审批3审批通过4交易对创建完成5交易对增加到撮合完成6等待撮合系统重启7发布交易对到撮合完成8交易对激活定时器完成9取消
    let content = "";
    if (status == PAIR_APPLY_STATUS.RE_WRITE) {
        content = "申请上交易对,被驳回：" + symbol;
    }
    else if (status == PAIR_APPLY_STATUS.WAIT_VIEW) {
        content = "申请上交易对：" + symbol;
    }
    else if (status == PAIR_APPLY_STATUS.PASS) {
        content = "申请上交易对,审批通过：" + symbol + "，下一步交易对创建";
    }
    else if (status == PAIR_APPLY_STATUS.CREATE_PAIR) {
        content = "申请上交易对,交易对创建完成：" + symbol + "，下一步增加到撮合";
    }
    else if (status == PAIR_APPLY_STATUS.ADMIN_ADD_PAIR) {
        content = "申请上交易对,增加到撮合完成：" + symbol + "，下一步等待撮合系统重启(不需要真的重启撮合)";
    }
    else if (status == PAIR_APPLY_STATUS.WAIT_ADMIN_RESTART) {
        content = "申请上交易对,撮合系统重启完成(不需要真的重启撮合)：" + symbol + "，下一步发布交易对到撮合";
    }
    else if (status == PAIR_APPLY_STATUS.ADMIN_ISSUE_PAIR) {
        content = "申请上交易对,发布交易对到撮合完成：" + symbol + "，下一步激活定时器";
    }
    else if (status == PAIR_APPLY_STATUS.TIMER_ACTIVE) {
        content = "申请上交易对,激活定时器完成：" + symbol + "，流程结束";
    }
    else if (status == PAIR_APPLY_STATUS.CANCEL) {
        content = "申请上交易对,被取消：" + symbol + "，流程结束";
    }
    if (content != null) {
        RobotUtil.sendRobotMessage(RobotUtil.ROBOT_KEYS.COMMON_KEY, content);

    }
}

/**
 * 检查交易对是否存在
 * @param pair
 */
async function checkPairExist(pair: string, type: number) {
    let dbPair: any;
    //现货
    if (type == 1) {
        dbPair = await spotPairs.prototype.findOne({
            where: { symbol: pair },
            raw: true
        });
    }
    else {//合约
        dbPair = await contractPairs.prototype.findOne({
            where: { symbol: pair },
            raw: true
        });
    }
    if (!dbPair) {
        throw ErrorCode.PAIR_NOT_EXIST;
    }
    return dbPair;
}

async function checkPairExistBeforeCreate(pair: string, type: number) {
    let dbPair: any;
    //现货
    if (type == 1) {
        dbPair = await spotPairs.prototype.findOne({
            where: { symbol: pair },
            raw: true
        });
    }
    else {//合约
        dbPair = await contractPairs.prototype.findOne({
            where: { symbol: pair },
            raw: true
        });
    }
    if (dbPair) {
        throw ErrorCode.PAIR_EXIST;
    }
    return dbPair;
}

