import { ormDB, noticeModel, noticeRead } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";


let _ = require('lodash');
let { logger, datetimeUtils } = require('@madex/ex-js-public');


export interface NoticeVO {
    id?: number;

    content?: string | any;

    publish_flag?: number;

    user_id?: string | any;

    notice_type?: number;

    push_type?: number;

    push_time?: Date | any;

    del_sign?: number;

    status?: number;

    createdAt?: Date | any;

    updatedAt?: Date | any;


}

export interface NoticePageVO extends NoticeVO {
    page?: number,

    size?: number
}

export async function list(noticePageVO: NoticePageVO) {
    let where = Object.create(null);
    if (noticePageVO.publish_flag) {
        where.publish_flag = noticePageVO.publish_flag;
    }
    if (Number(noticePageVO.user_id) >= 0) {
        where.user_id = noticePageVO.user_id;
    }
    if (noticePageVO.notice_type) {
        where.notice_type = noticePageVO.notice_type;
    }
    if (noticePageVO.push_type) {
        where.push_type = noticePageVO.push_type;
    }
    if (noticePageVO.push_time) {
        let date = datetimeUtils.trim(noticePageVO.push_time, 's');
        where.push_time = { [ormDB.Op.gte]: date };
    }
    if (noticePageVO.del_sign || noticePageVO.del_sign === 0) {
        where.del_sign = noticePageVO.del_sign;
    }
    if (noticePageVO.status) {
        where.status = noticePageVO.status;
    }

    if (noticePageVO.createdAt) {
        let date = datetimeUtils.trim(noticePageVO.createdAt, 's');
        where.createdAt = { [ormDB.Op.gte]: date }
    }

    let resList = await noticeModel.prototype.findAndCount({
        where: where,
        limit: noticePageVO.size,
        offset: (Number(noticePageVO.page) - 1) * Number(noticePageVO.size),
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}

export async function add(noticeVO: NoticeVO, currentUserId: any, ip: any) {
    let insertList: any = [];

    if (!noticeVO.publish_flag) {
        noticeVO.publish_flag = 0;
    }
    if (!noticeVO.push_type) {
        noticeVO.push_type = 0;
    }
    noticeVO.del_sign = 0;
    noticeVO.status = 0;
    noticeVO.createdAt = new Date();
    noticeVO.updatedAt = new Date();
    if (!noticeVO.user_id) {
        noticeVO.user_id = 0;
        insertList.push(noticeVO);
    }
    else {//多个uid 的消息
        let uids = noticeVO.user_id.split(',');
        for (let oneUid of uids) {
            let item = {
                content: noticeVO.content,
                publish_flag: noticeVO.publish_flag,
                user_id: oneUid,
                notice_type: noticeVO.notice_type,
                push_type: noticeVO.push_type,
                push_time: noticeVO.push_time,
                del_sign: noticeVO.del_sign,
                status: noticeVO.status,
                createdAt: noticeVO.createdAt,
                updatedAt: noticeVO.updatedAt,
            }
            insertList.push(item);
        }
    }

    await noticeModel.prototype.bulkCreate(insertList);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增消息推送', ip, JSON.stringify(noticeVO), '消息推送');
    return 'success'
}


export async function update(noticeVO: NoticeVO, currentUserId: any, ip: any) {

    let dbInfo = await noticeModel.prototype.findOne({
        where: {
            id: noticeVO.id
        },
        raw: true
    });

    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    if (dbInfo.status == 1) {
        throw ErrorCode.PUSH_NOT_UPDATE;
    }

    let updateInfo = {};
    if (noticeVO.content) {
        updateInfo['content'] = noticeVO.content;
    }
    if (noticeVO.publish_flag || noticeVO.publish_flag === 0) {
        updateInfo['publish_flag'] = noticeVO.publish_flag;
    }
    if (noticeVO.notice_type) {
        updateInfo['notice_type'] = noticeVO.notice_type;
    }
    if (noticeVO.push_type || noticeVO.push_type === 0) {
        updateInfo['push_type'] = noticeVO.push_type;
    }
    if (noticeVO.push_time) {
        updateInfo['push_time'] = noticeVO.push_time;
    }
    if (noticeVO.del_sign) {
        updateInfo['del_sign'] = noticeVO.del_sign;
    }
    updateInfo['updatedAt'] = new Date();

    await noticeModel.prototype.update(updateInfo, {
        where: {
            id: Number(noticeVO.id)
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '更新消息推送', ip, JSON.stringify(noticeVO), '消息推送');

    return 'success'
}





