import * as commonUserFeeSettingService from "../service/commonUserFeeSetting.service";
import { CommonUserFeeVO, CommonUserFeePageVO } from "../service/commonUserFeeSetting.service";

import { vipLevelFeeSetting } from "@madex/ex-ts-dao";


let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert, datetimeUtils } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { IS_ACTIVE, IS_ACTIVE_ARR, LINK_TYPE_ARR } from "../../../constant/usefulLinkConst";
import { getCurrentUserId } from "../../../utils/aclUserUtils";
import { FEE_TYPE, FEE_TYPE_LIST } from "../../../constant/marketMakerConst";
import { baseCoinCheck, baseUCheck, spotCheck } from "../../../utils/feeRateCheckValidUtils";
import { getMUserInfoByUid } from "../../../utils/mUserUtils";
import { getFatherUserId } from "../service/mUserInfoSon.service";

let isIp = require('is-ip');
/**
 * 手续费列表
 * @param req
 * @param infoVO
 */
export const list = async (req: any, pageVO: CommonUserFeePageVO) => {
    let func_name = "commonUserFeeSetting.control.list";
    try {
        pageVO.page = Optional.opt(pageVO, 'page', 1);
        pageVO.size = Optional.opt(pageVO, 'size', 20);
        let res = await commonUserFeeSettingService.list(pageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 添加用户费率
 * @param req
 * @param authConfigVO
 */
export const add = async (req: any, commonUserFeeVO: CommonUserFeeVO) => {
    let func_name = "commonUserFeeSetting.control.add";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await addParamValid(commonUserFeeVO);

        let res = await commonUserFeeSettingService.add(commonUserFeeVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 修改用户费率
 * @param req
 * @param authConfigVO
 */
export const update = async (req: any, commonUserFeeVO: CommonUserFeeVO) => {
    let func_name = "commonUserFeeSetting.control.update";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await updateParamValid(commonUserFeeVO);

        let res = await commonUserFeeSettingService.update(commonUserFeeVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 删除用户费率
 * @param req
 * @param authConfigVO
 */
export const del = async (req: any, commonUserFeeVO: CommonUserFeeVO) => {
    let func_name = "commonUserFeeSetting.control.del";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!commonUserFeeVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await commonUserFeeSettingService.del(commonUserFeeVO.id, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 提交用户费率
 * @param req
 * @param authConfigVO
 */
export const submit = async (req: any, commonUserFeeVO: CommonUserFeeVO) => {
    let func_name = "commonUserFeeSetting.control.submit";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!commonUserFeeVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await commonUserFeeSettingService.submit(commonUserFeeVO.id, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


async function addParamValid(commonUserFeeVO: CommonUserFeeVO) {
    let userId = commonUserFeeVO.user_id;
    let beginTime = commonUserFeeVO.begin_time;
    let expireTime = commonUserFeeVO.expire_time;
    let type = commonUserFeeVO.type;
    let levelId = commonUserFeeVO.level_id;

    if (!userId || !beginTime || !expireTime) {
        throw ErrorCode.PARAM_MISS;
    }

    await paramCheck(beginTime, expireTime, Number(type), userId);

    if (levelId) {
        let vipLevelSetting = await vipLevelFeeSetting.prototype.findOne({
            where: {
                id: levelId
            },
            raw: true
        });
        if (!vipLevelSetting) {
            throw ErrorCode.LEVEL_NOT_EXIST;
        }
        commonUserFeeVO.spot_maker_fee = vipLevelSetting.spot_maker;
        commonUserFeeVO.spot_taker_fee = vipLevelSetting.spot_taker;
        commonUserFeeVO.contract_maker_fee = vipLevelSetting.contract_maker;
        commonUserFeeVO.contract_taker_fee = vipLevelSetting.contract_taker;

        return;
    }

    if ((type == FEE_TYPE.FEE_TYPE_ALL || type == FEE_TYPE.FEE_TYPE_SPOT)
        && (!commonUserFeeVO.spot_maker_fee || !commonUserFeeVO.spot_taker_fee)) {
        throw ErrorCode.SPOT_FEE_NEED_SET;
    }

    if ((type == FEE_TYPE.FEE_TYPE_ALL || type == FEE_TYPE.FEE_TYPE_CONTRACT_ALL
            || type == FEE_TYPE.FEE_TYPE_CONTRACT || type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT)
        && (!commonUserFeeVO.contract_maker_fee || !commonUserFeeVO.contract_taker_fee)) {
        throw ErrorCode.CONTRACT_FEE_NEED_SET;
    }

    let user = await getMUserInfoByUid(userId);
    if (!user || user.deleted_flag == 1) {
        throw ErrorCode.USER_NOT_EXIST;
    }

    let fuid = await getFatherUserId(userId);
    if (fuid) {
        throw ErrorCode.SON_NOT_ADD_FEE;
    }

}


async function updateParamValid(commonUserFeeVO: CommonUserFeeVO) {
    let id = commonUserFeeVO.id;
    let userId = commonUserFeeVO.user_id;
    let beginTime = commonUserFeeVO.begin_time;
    let expireTime = commonUserFeeVO.expire_time;
    let type = commonUserFeeVO.type;
    let makerFee = commonUserFeeVO.maker_fee;
    let takerFee = commonUserFeeVO.taker_fee;

    if (!id || !userId || !beginTime || !expireTime || !makerFee || !takerFee) {
        throw ErrorCode.PARAM_MISS;
    }
    await paramCheck(beginTime, expireTime, Number(type), userId);

}

/**
 * 添加和更新校验入参
 * @param beginTime
 * @param expireTime
 * @param type
 * @param userId
 */
async function paramCheck(beginTime: any, expireTime: any, type: number, userId: number) {
//结束时间最少需要是一小时后；开始时间和结束时间最少需要间隔一小时
    if (!datetimeUtils.between(expireTime, new Date()) > datetimeUtils.HOUR
        || !datetimeUtils.between(expireTime, beginTime) > datetimeUtils.HOUR) {
        throw ErrorCode.PARAM_MISS;
    }
    if (isNaN(Number(type)) || !FEE_TYPE_LIST.includes(Number(type))) {
        throw ErrorCode.PARAM_MISS;
    }

    if (type == FEE_TYPE.FEE_TYPE_SPOT) {
        if (!await spotCheck(userId)) {
            throw ErrorCode.SPOT_NO_ASSETS;
        }
    }
    else if (type == FEE_TYPE.FEE_TYPE_CONTRACT) {
        if (!await baseUCheck(userId)) {
            throw ErrorCode.U_NO_ASSETS;
        }
    }
    else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {
        if (!await baseCoinCheck(userId)) {
            throw ErrorCode.COIN_NO_ASSETS;
        }
    }
    else if (type == FEE_TYPE.FEE_TYPE_CONTRACT_ALL) {
        if (!await baseUCheck(userId)) {
            throw ErrorCode.U_NO_ASSETS;
        }
        if (!await baseCoinCheck(userId)) {
            throw ErrorCode.COIN_NO_ASSETS;
        }
    }
    else if (type == FEE_TYPE.FEE_TYPE_ALL) {
        if (!await spotCheck(userId)) {
            throw ErrorCode.SPOT_NO_ASSETS;
        }
        if (!await baseUCheck(userId)) {
            throw ErrorCode.U_NO_ASSETS;
        }
        if (!await baseCoinCheck(userId)) {
            throw ErrorCode.COIN_NO_ASSETS;
        }
    }
}
