import { coinAddress, coinWithdraw, ormDB, userAccountChangeLog, userInfo, userProfile } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { getProfileByUserIds } from "./mUserProfile.service";
import { getRealNameMapByUserIds, getRealNameByUserId } from "./mUserRealName.service";
import { MUserRedisKey } from "../../../constant/mUserRedisKey";
import { getMUserInfoByEmail, getMUserInfoByUid, getUserInfoMapByUids, sendEmail, updateMUserInfo } from "../../../utils/mUserUtils";
import { existEmail, getProcessingEmailListByMUserId } from "./mUserAccountChangeLog.service";
import { AFTER_MODIFIED_MAILBOX_TIP, EMAIL_FORBID_TEMPLATE } from "../../../constant/emailTemplateType";
import { addLog, MUserLogType } from "./mUserOptLog.service";
import { addOptLog, LogType } from "./userOptLog.service";
import { checkTotp } from "../../../utils/aclUserUtils";
import { getFatherUserId, getUidsByFatherUid } from "./mUserInfoSon.service";
import { recordMUserOperateLog, TYPE } from "./mUserAccountOperateLog.service";
import { tradeAsset2USDTByUids, walletAsset2USDTByUids } from "./mUserAssets.service";
import BigNumber from "bignumber.js";
import { getCoinWithdrawMapByUids } from "./coinWithdraw.service";
import { getCoinAddressMapByUids } from "./coinAddress.service";

let _ = require('lodash');
let { logger, datetimeUtils } = require('@madex/ex-js-public');
let { redisUtilsCommon, authCommon } = require('@madex/ex-js-common');


export interface QueryVO {

    page?: number,

    size?: number

    user_id?: number | any;

    email?: string | any

    totp_code?: string | any

    lock_type?: number | any//锁定类型 1 锁定当前账户 2 锁定当前账户及其所有子账户

    condition?: string

}


export async function userList(page: number, size: number, condition: string, condition_type: number) {

    let pageData: any;
    let where: any
    //uid/姓名/邮箱/证件号 查询
    if (condition_type == 1) {
        //没有查询条件
        if (condition == "undefined" || condition == undefined || condition == "") {
            where = "";
        }
        else {
            where = {
                email: String(condition),
                real_name: String(condition),
                phone: String(condition),
                identity: String(condition),
            }
            //数字类型的话 补充uid作为查询条件
            if (!isNaN(Number(condition))) {
                where['user_id'] = Number(condition)
            }
        }

        //
        pageData = await userInfo.prototype.findAndCount({
            attributes: ['user_id', 'email', 'real_name', 'createdAt', 'totp_encrypt', 'is_lock', 'identity'],
            where: where ? { [ormDB.Op.or]: where } : {},
            limit: size,
            offset: (page - 1) * size,
            order: [["user_id", "asc"]],
            raw: true
        });
        await buildReturnData(pageData, false, true, true);
        return pageData;
    }
    else {
        if (condition) {
            where = {
                address: String(condition)
            }
        }
        pageData = await coinAddress.prototype.findAndCount({
            attributes: ['user_id', 'address'],
            where: where,
            limit: size,
            offset: (page - 1) * size,
            order: [["id", "asc"]],
            raw: true
        });
        if (!pageData.rows.length) {
            if (condition) {
                where = {
                    to_address: String(condition)
                }
            }
            pageData = await coinWithdraw.prototype.findAndCount({
                attributes: ['user_id', 'to_address'],
                where: where,
                limit: size,
                offset: (page - 1) * size,
                order: [["id", "asc"]],
                raw: true
            });
            await buildReturnData(pageData, true, true, false);
            return pageData;
        }
        else {
            await buildReturnData(pageData, true, false, true);
            return pageData;

        }
    }


}


export async function oneUserDetail(m_user_id: number) {

    let oneUser = await userInfo.prototype.findOne({
        attributes: ['user_id', 'email', 'real_name', 'createdAt', 'totp_encrypt', 'is_lock', 'secure_modifiedAt'],
        where: {
            user_id: m_user_id
        },
        raw: true
    });

    if (!oneUser) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let oneRealName = await getRealNameByUserId(m_user_id);
    let isLoginLock = await redisUtilsCommon.getSync(m_user_id + MUserRedisKey.USER_LOCK_SUFFIX);
    //24小时提现限制
    let dead_line = datetimeUtils.add(oneUser.secure_modifiedAt, datetimeUtils.DAY);
    let is_withdraw = datetimeUtils.between(new Date(), dead_line);
    let res = {
        user_id: oneUser.user_id,
        real_name: oneUser.real_name,
        email: oneUser.email,
        register_date: oneUser.createdAt,
        kyc_status: oneRealName ? oneRealName.status : 0,//0未提交，1审核中，2审核不通过，3审核通过
        kyc_type: oneRealName ? oneRealName.type : 0,//1身份证，2护照，3驾驶证
        kyc_identity: oneRealName ? oneRealName.identity : 0,//证件号
        is_bind_totp: oneUser.totp_encrypt ? 1 : 0,
        lock_status: oneUser.is_lock, //0 否 1 是
        login_lock: isLoginLock ? 1 : 0,//登陆锁定 0 否 1 是
        withdraw_limit_24: is_withdraw > 0 ? 0 : 1,//24小时提现限制
    }
    return res;

}


export async function updateUserEmail(currentUserId: number, m_user_id: any, email: any, totp_code: any, ip: any) {

    //校验谷歌
    await checkTotp(currentUserId, totp_code);

    let mUserDbInfo = await getMUserInfoByEmail(email);
    if (mUserDbInfo) {
        throw ErrorCode.UPDATE_EMAIL_EXIST;
    }

    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }

    //检查邮箱是否存在于 userAccountChangeLog中，是则不可用
    let isExist = await existEmail(email);
    if (isExist) {
        throw ErrorCode.EMAIL_USED_NTO_CHANGE;
    }
    //检查是否已经有一条待修改记录 记录新/旧邮箱
    let isExistList = await getProcessingEmailListByMUserId(m_user_id);
    if (isExistList.length > 0) {
        throw ErrorCode.EMAIL_CHANGE_OFTEN;
    }

    let dbEmail = dnUidUserInfo.email;

    await modifyUserEmail(m_user_id, email, dbEmail);
    //踢出登陆
    authCommon.delSessionIdList(m_user_id);
    //新邮箱发邮件
    sendEmail(email, m_user_id, EMAIL_FORBID_TEMPLATE);
    //旧邮箱发送邮件
    sendEmail(dbEmail, m_user_id, AFTER_MODIFIED_MAILBOX_TIP);
    //记录Madex的用户自己的日志
    addLog(m_user_id, MUserLogType.UP_EMAIL, ip, currentUserId, `运营修改用户邮箱为:${email}`);
    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},修改用户:${m_user_id} 邮箱,原邮箱: ${dbEmail},新邮箱:${email}`, LogType.UPDATE_MUSER_EMAIL);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(m_user_id, currentUserId, TYPE.MAIL, `ip:${ip},修改邮箱`, dbEmail, email);
    return 'success';
}


export async function lockAccount(currentUserId: number, m_user_id: any, ip: any, lock_type: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);
    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    let comment = "";
    let fatherUserId = dnUidUserInfo.user_id;
    //锁定当前用户
    if (lock_type == 1) {
        await updateMUserInfo(m_user_id, { is_lock: 1 });
        //踢出登陆
        authCommon.delSessionIdList(m_user_id);
        comment = `ip:${ip},锁定用户:${fatherUserId}`;
    }
    else {
        if (dnUidUserInfo.user_type == userInfo.USER_TYPE.SON) {
            let temp = await getFatherUserId(m_user_id);
            if (!temp) {
                throw ErrorCode.DAD_ACCOUNT_NOT_EXIST;
            }
            fatherUserId = temp;
            let uids = await getUidsByFatherUid(fatherUserId);
            uids.push(fatherUserId);
            //更新主账户及其子账户
            await userInfo.prototype.update({
                is_lock: 1,
                updatedAt: new Date()
            }, {
                where: {
                    user_id: { [ormDB.Op.in]: uids }
                }
            });
            //踢出登陆
            for (let uid of uids) {
                authCommon.delSessionIdList(uid);
            }
            comment = `ip:${ip},锁定用户:${uids}`;
        }

    }
    //管理后台操作日志
    addOptLog(currentUserId, comment, LogType.LOCK_MUSER);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(fatherUserId, currentUserId, TYPE.ACCOUNT_STATUS, comment, "0", "1");

    return 'success';
}


export async function unlockAccount(currentUserId: number, m_user_id: any, ip: any, lock_type: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);
    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    if (Number(dnUidUserInfo.deleted_flag) > 0) {
        throw ErrorCode.DEL_USER_NO_UNLOCK;
    }
    if (!dnUidUserInfo.is_lock) {
        throw ErrorCode.USER_NOT_LOCK;
    }
    let comment = "";
    let fatherUserId = dnUidUserInfo.user_id;
    //解锁当前用户
    if (lock_type == 1) {
        await updateMUserInfo(m_user_id, {
            is_lock: 0,
            login_error_times: 0
        });
        comment = `ip:${ip},解锁用户:${fatherUserId}`;
    }
    else {
        if (dnUidUserInfo.user_type == userInfo.USER_TYPE.SON) {
            let temp = await getFatherUserId(m_user_id);
            if (!temp) {
                throw ErrorCode.DAD_ACCOUNT_NOT_EXIST;
            }
            fatherUserId = temp;
            let uids = await getUidsByFatherUid(fatherUserId);
            uids.push(fatherUserId);
            //更新主账户及其子账户
            await userInfo.prototype.update({
                is_lock: 0,
                login_error_times: 0,
                updatedAt: new Date()
            }, {
                where: {
                    user_id: { [ormDB.Op.in]: uids }
                }
            });

            comment = `ip:${ip},解锁用户:${uids}`;
        }

    }
    //管理后台操作日志
    addOptLog(currentUserId, comment, LogType.UNLOCK_MUSER);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(fatherUserId, currentUserId, TYPE.ACCOUNT_STATUS, comment, "1", "0");

    return 'success';
}


export async function clearLoginLimit(currentUserId: number, m_user_id: any, ip: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);
    await redisUtilsCommon.delSync(m_user_id + MUserRedisKey.USER_LOCK_SUFFIX);

    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},清除用户登陆限制:${m_user_id}`, LogType.CLEAR_LOGIN_LIMIT);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(m_user_id, currentUserId, TYPE.TWO_HOUR_LIMIT, `ip:${ip},清除用户登陆限制:${m_user_id}`, "", "");

    return 'success';
}


export async function clear24WithdrawLimit(currentUserId: number, m_user_id: any, ip: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);

    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    let date = datetimeUtils.add(new Date(), datetimeUtils.DAY * -1)
    await updateMUserInfo(m_user_id, {
        secure_modifiedAt: date,
    });

    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},清除用户24小时提现限制:${m_user_id}`, LogType.WITHDRAW_24_LIMIT);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(m_user_id, currentUserId, TYPE.TWENTY_FOUR_HOUR_LIMIT, `ip:${ip},清除用户24小时提现限制:${m_user_id}`, "", "");

    return 'success';
}

async function modifyUserEmail(m_user_id: number, email_new: string, email_old: string) {
    let tx;
    try {
        tx = await ormDB.transaction();
        await userAccountChangeLog.prototype.create({
                user_id: m_user_id,
                state: 3,
                account_old: email_old,
                account_new: email_new,
                type: 1,
                createdAt: new Date(),
                updatedAt: new Date(),
                finish_time: new Date(),
            },
            {
                transaction: tx,
            });

        await userInfo.prototype.update({
            email: email_new,
            updatedAt: new Date(),
        }, {
            where: {
                user_id: m_user_id
            },
            transaction: tx
        });
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('modifyUserEmail.error:' + e)
        throw e;
    }

}

/**
 * 构造返回数据
 * @param pageData 分页数据
 * @param fill_user_info 是否需要 填充 user_info 的数据 0 否 1 是
 * @param fill_coin_address 是否需要 填充 coin_address 的数据 0 否 1 是
 * @param fill_coin_withdraw 是否需要 填充 coin_withdraw 的数据 0 否 1 是
 */
async function buildReturnData(pageData: any, fill_user_info: boolean, fill_coin_address: boolean, fill_coin_withdraw: boolean) {
    if (pageData.rows.length > 0) {
        let uids = pageData.rows.map(item => item.user_id);
        //注意顺序
        let task: any = [];
        //实名信息
        task.push(getRealNameMapByUserIds(uids));
        //钱包余额
        task.push(walletAsset2USDTByUids(uids));
        //交易账户余额
        task.push(tradeAsset2USDTByUids(uids));
        //用户信息
        task.push(fill_user_info ? getUserInfoMapByUids(uids) : null);
        //充值地址
        task.push(fill_coin_address ? getCoinAddressMapByUids(uids) : null);
        //提现地址
        task.push(fill_coin_withdraw ? getCoinWithdrawMapByUids(uids) : null);

        let [realNameMap, walletAssetMap, tradeAssetMap, userInfoMap, coinAddrMap, withdrawMap] = await Promise.all(task);

        //替换rows的集合
        let infoList: any = [];

        for (let item of pageData.rows) {
            let user_id = item.user_id;
            //初始化
            let wallet_account_balance = new BigNumber(String(walletAssetMap[item.user_id].balance_usdt)).add(new BigNumber(String(walletAssetMap[item.user_id].holds_usdt)));
            let trade_account_balance = new BigNumber(String(tradeAssetMap[item.user_id].balance_usdt)).add(new BigNumber(String(tradeAssetMap[item.user_id].holds_usdt)));
            let oneRealName = realNameMap[item.user_id];
            let one = {
                user_id: user_id,
                real_name: item.real_name ? item.real_name : "",
                email: item.email ? item.email : "",
                register_date: item.createdAt ? item.createdAt : "",
                spot_trade_total: 0,//TODO:现货交易总量
                contract_trade_total: 0,//TODO:永续合约交易总量
                spot_balance: 0,//TODO:现货余额
                contract_balance: 0,//TODO:永续合约余额
                assets_total: 0,//TODO: 总资产
                spot_order: 0,//TODO: 现货订单数量
                contract_order: 0,//TODO: 永续合约订单数量
                wallet_account_balance: wallet_account_balance,//钱包账户余额
                trade_account_balance: trade_account_balance,//TODO: 交易账户余额
                kyc_status: oneRealName ? oneRealName.status : 0,//0未提交，1审核中，2审核不通过，3审核通过
                kyc_type: oneRealName ? oneRealName.type : 0,//1身份证，2护照，3驾驶证
                kyc_identity: item.kyc_identity ? item.kyc_identity : "",
                is_bind_totp: item.totp_encrypt ? 1 : 0,
                lock_status: item.is_lock ? item.is_lock : "",
                address: item.address ? item.address : "",
                to_address: item.to_address ? item.to_address : "",
            };
            //填充用户信息
            if (userInfoMap) {
                one['real_name'] = userInfoMap[user_id] ? userInfoMap[user_id].real_name : "";
                one['email'] = userInfoMap[user_id] ? userInfoMap[user_id].email : "";
                one['register_date'] = userInfoMap[user_id] ? userInfoMap[user_id].createdAt : "";
                one['kyc_identity'] = userInfoMap[user_id] ? userInfoMap[user_id].identity : "";
                one['is_bind_totp'] = userInfoMap[user_id] && userInfoMap[user_id].totp_encrypt ? 1 : 0;
                one['lock_status'] = userInfoMap[user_id] ? userInfoMap[user_id].is_lock : "";
            }
            //充币地址
            if (coinAddrMap) {
                one['address'] = coinAddrMap[user_id] ? coinAddrMap[user_id].address : "";
            }
            //提币地址
            if (withdrawMap) {
                one['to_address'] = withdrawMap[user_id] ? withdrawMap[user_id].to_address : "";
            }
            infoList.push(one);
        }
        pageData.rows = infoList;
    }
}





