import { madAdminOrmDB, coinAddress, coinType, mainUserAsset, dwdSpotAssetsApi, ormDB, coinTx, coinWithdraw, walletAssets } from "@madex/ex-ts-dao";
import BigNumber from "bignumber.js";

let { logger, apiAssertUtils: ApiAssert, BigNumberUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');

let _ = require('lodash');


export async function userAddrList(page: any, size: any, user_id: any) {
    let where: any = {};
    if (user_id) {
        where["user_id"] = Number(user_id);
    }
    let res = await coinAddress.prototype.findAndCount({
        attributes: ['user_id', 'coin_id', 'coin_symbol', 'address', 'createdAt'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["createdAt", "desc"]],
        raw: true
    });
    if (res.rows.length) {
        let coinIds = res.rows.map(item => item.coin_id);
        let coinMap = await getCoinTypeMapByCoinOrAssetIds(coinIds);
        for (let item of res.rows) {
            item.chain_type = coinMap[item.coin_id].chain_type;
            item.coin_symbol = coinMap[item.coin_id].symbol;
        }
    }
    return res;
}

export async function assetDetails(page: any, size: any, user_id: number) {
    let res = await coinAddress.prototype.findAndCount({
        attributes: ['coin_id', 'coin_symbol', 'balance'],
        where: {
            user_id: user_id
        },
        limit: size,
        offset: (page - 1) * size,
        order: [["createdAt", "desc"]],
        raw: true
    });
    if (res.rows.length) {
        let coinIds = res.rows.map(item => item.coin_id);
        let coinMap = await getCoinTypeMapByCoinOrAssetIds(coinIds);
        for (let item of res.rows) {
            item.chain_type = coinMap[item.coin_id].chain_type;
            item.coin_symbol = coinMap[item.coin_id].symbol;
        }
    }
    return res;
}


export async function siteAssets(page: any, size: any) {

    let pageData = await coinType.prototype.findAndCount({
        attributes: [ormDB.literal('DISTINCT(asset_id) as asset_id')],
        where: {
            main_status: 2//已经提交到撮合
        },
        limit: size,
        offset: (page - 1) * size,
        order: [['asset_id', 'asc']],
        raw: true
    });
    if (!pageData.rows.length) {
        return pageData;
    }
    let asset_ids = pageData.rows.map(item => item.asset_id);

    let dbCoinList = await coinType.prototype.findAll({
        attributes: ['id', 'asset_id', 'symbol', 'chain_type'],
        where: {
            asset_id: asset_ids
        },
        order: [['asset_id', 'asc'], ['id', 'asc']],
        raw: true
    });

    let coinMap: any = {};
    for (let item of dbCoinList) {
        //同名币 asset_id 一样 id 不一样
        if (!coinMap[item.asset_id]) {
            coinMap[item.asset_id] = item
        }
    }


    let [mainAssetsMap, tradeAssetsMap, { hotWalletMap, coldWalletMap, totalWalletMap }] = await Promise.all([getSiteMainAssetsGroupBy(asset_ids, 'asset_id'),
        getSiteTradeAssetsGroupBy(asset_ids, 'asset_id'),
        getSiteWalletAssetsGroupBy(asset_ids, 'asset_id')]);

    for (let item of pageData.rows) {
        let assetId = item.asset_id;
        item.symbol = coinMap[assetId] ? coinMap[assetId].symbol : "";
        item.chain_type = coinMap[assetId] ? coinMap[assetId].chain_type : "";
        item.site_assets = (mainAssetsMap[assetId] ? mainAssetsMap[assetId] : new BigNumber(0)).add(
            tradeAssetsMap[assetId] ? tradeAssetsMap[assetId] : new BigNumber(0)
        );
        item.total_wallet_assets = totalWalletMap[assetId] ? totalWalletMap[assetId] : new BigNumber(0);
        item.hot_wallet_assets = hotWalletMap[assetId] ? hotWalletMap[assetId] : new BigNumber(0);
        item.cold_wallet_assets = coldWalletMap[assetId] ? coldWalletMap[assetId] : new BigNumber(0);
    }
    return pageData;
}


export async function siteAssetsDetails(asset_id: number) {

    let symbolList = await coinType.prototype.findAll({
        attributes: ['symbol'],
        where: {
            asset_id
        },
        raw: true
    });
    //单个币种没必要查
    if (symbolList.length < 2) {
        return [];
    }

    let symbols = symbolList.map(item => item.symbol);

    let dbCoinList = await coinType.prototype.findAll({
        attributes: ['id', 'asset_id', 'symbol', 'chain_type'],
        where: {
            symbol: symbols
        },
        raw: true
    });

    let coinMap: any = {};
    for (let item of dbCoinList) {
        coinMap[item.symbol] = item
    }


    let [mainAssetsMap, tradeAssetsMap, { hotWalletMap, coldWalletMap, totalWalletMap }] = await Promise.all([getSiteMainAssetsGroupBy(symbols, 'symbol'),
        getSiteTradeAssetsGroupBy(symbols, 'asset_symbol'),
        getSiteWalletAssetsGroupBy(symbols, 'coin_symbol')]);

    for (let item of symbolList) {
        let symbol = item.symbol;
        item.chain_type = coinMap[symbol] ? coinMap[symbol].chain_type : "";
        item.site_assets = (mainAssetsMap[symbol] ? mainAssetsMap[symbol] : new BigNumber(0)).add(
            tradeAssetsMap[symbol] ? tradeAssetsMap[symbol] : new BigNumber(0)
        );
        item.total_wallet_assets = totalWalletMap[symbol] ? totalWalletMap[symbol] : new BigNumber(0);
        item.hot_wallet_assets = hotWalletMap[symbol] ? hotWalletMap[symbol] : new BigNumber(0);
        item.cold_wallet_assets = coldWalletMap[symbol] ? coldWalletMap[symbol] : new BigNumber(0);
    }
    return symbolList;
}

export async function depositList(page: any, size: any, uid_or_addr: any) {
    let where: any;
    if (!isNaN(Number(uid_or_addr))) {
        where = {
            [ormDB.Op.or]: {
                user_id: Number(uid_or_addr),
                to: String(uid_or_addr)
            }
        }
    }
    else {
        where = { to: String(uid_or_addr) }
    }

    let pageData = await coinTx.prototype.findAndCount({
        attributes: ['user_id', 'coin_id', 'coin_symbol', 'amount', 'to', 'status', 'updatedAt'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [['updatedAt', 'desc']],
        raw: true
    });
    let coinIds = pageData.rows.map(item => item.coin_id);
    let coinTypeMap = await getCoinTypeMapByCoinOrAssetIds(coinIds, []);
    for (let item of pageData.rows) {
        item.chain_type = coinTypeMap[item.coin_id] ? coinTypeMap[item.coin_id].chain_type : ""
    }
    return pageData
}

export async function withdrawList(page: any, size: any, uid_or_addr: any) {
    let where: any;
    if (!isNaN(Number(uid_or_addr))) {
        where = {
            [ormDB.Op.or]: {
                user_id: Number(uid_or_addr),
                to_address: String(uid_or_addr)
            }
        }
    }
    else {
        where = { to_address: String(uid_or_addr) }
    }

    let pageData = await coinWithdraw.prototype.findAndCount({
        attributes: ['user_id', 'coin_id', 'coin_symbol', 'amount', 'to_address', 'status', 'updatedAt'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [['updatedAt', 'desc']],
        raw: true
    });
    let coinIds = pageData.rows.map(item => item.coin_id);
    let coinTypeMap = await getCoinTypeMapByCoinOrAssetIds(coinIds, []);
    for (let item of pageData.rows) {
        item.chain_type = coinTypeMap[item.coin_id] ? coinTypeMap[item.coin_id].chain_type : ""
    }
    return pageData
}


async function getCoinTypeMapByCoinOrAssetIds(coin_ids: any[], asset_ids?: any[]) {
    let map: any = {};
    let where: any = {};
    let coin_id_flag = false;
    if (coin_ids.length) {
        where["id"] = coin_ids;
        coin_id_flag = true;
    }
    else {
        where["asset_id"] = asset_ids;
    }

    let dbList = await coinType.prototype.findAll({
        where: where,
        raw: true
    });

    for (let item of dbList) {
        if (coin_id_flag) {
            map[item.id] = item;
        }
        else {
            map[item.asset_id] = item;
        }
    }
    return map;
}

async function getSiteMainAssetsGroupBy(conditions: any[], group_by_field: string) {
    let map: any = {};
    let where: any = {};
    where[group_by_field] = conditions;
    let mainAssetsList = await mainUserAsset.prototype.findAll({
        attributes: [group_by_field, ormDB.literal('sum(balance) as balance'),
            ormDB.literal('sum(holds) as holds'),],
        where: where,
        group: [group_by_field, 'symbol'],
        raw: true
    });
    for (let item of mainAssetsList) {
        map[item[group_by_field]] = new BigNumber(item.balance).add(new BigNumber(item.holds));
    }
    return map;
}

async function getSiteTradeAssetsGroupBy(conditions: any[], group_by_field: string) {
    let map: any = {};
    let where: any = {};
    where[group_by_field] = conditions;
    let tradeAssetsList = await dwdSpotAssetsApi.prototype.findAll({
        attributes: [group_by_field, ormDB.literal('sum(balance) as balance'),
            ormDB.literal('sum(unreal_profit) as unreal_profit'),],
        where: where,
        group: [group_by_field],
        raw: true
    });
    for (let item of tradeAssetsList) {
        map[item[group_by_field]] = new BigNumber(item.balance).add(new BigNumber(item.unreal_profit));
    }
    return map;
}

async function getSiteWalletAssetsGroupBy(conditions: any[], group_by_field: string) {
    let where: any = {};
    where[group_by_field] = conditions;

    let walletAssetsList = await walletAssets.prototype.findAll({
        attributes: [group_by_field, 'type', ormDB.literal('sum(balance) as balance')],
        where: where,
        group: [group_by_field, 'type'],
        raw: true
    });
    let hotWalletMap: any = {};
    let coldWalletMap: any = {};
    let totalWalletMap: any = {};
    for (let item of walletAssetsList) {
        if (item.type == 3) {//冷钱包
            coldWalletMap[item[group_by_field]] = new BigNumber(item.balance);
        }
        else {//热钱包
            hotWalletMap[item[group_by_field]] = hotWalletMap[item[group_by_field]] ? hotWalletMap[item[group_by_field]].add(new BigNumber(item.balance)) : new BigNumber(item.balance);
        }
        totalWalletMap[item[group_by_field]] = totalWalletMap[item[group_by_field]] ? totalWalletMap[item[group_by_field]].add(new BigNumber(item.balance)) : new BigNumber(item.balance);
    }

    return {
        hotWalletMap, coldWalletMap, totalWalletMap
    }
}



