import { addOptLog } from "../service/userOptLog.service";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert, datetimeUtils } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUserId } from "../../../utils/aclUserUtils";
import { it } from "node:test";

let isIp = require('is-ip');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');

interface LimitTradeVO {
    type?: number,//1U本位，2币本位

    status?: number//1限制交易，2限制开仓，3取消限制交易，4取消限制开仓

    user_id?: number

    pair?: string
}

/**
 * 金融部-其他管理-合约交易限制配置列表
 * @param req
 * @param infoVO
 */
export const list = async (req: any, limitTradeVO: LimitTradeVO) => {
    let func_name = "contractLimitTrade.control.list";
    try {
        let redisKey = 'contractLimitTrade.list';
        let cacheMap = await RedisClient.getSync(redisKey);
        let list: any = [];
        if (cacheMap) {
            let keys = Object.keys(cacheMap);
            for (let key of keys) {
                let splitArr = key.split(".");
                let item = {
                    type: splitArr[0] == "baseu" ? 1 : 2,
                    status: splitArr[2] == "trade" ? 1 : 2,
                    pair: splitArr[3],
                    user_id: splitArr[4],
                }
                list.push(item)
            }
        }
        let status = limitTradeVO.status;
        let pair = limitTradeVO.pair;
        let userId = limitTradeVO.user_id;
        let type = limitTradeVO.type;

        if (pair) {
            list = list.filter(item => item.pair == pair);
        }
        if (status) {
            list = list.filter(item => item.status == status);
        }
        if (userId || userId === 0) {
            list = list.filter(item => item.user_id == userId);
        }
        if (type) {
            list = list.filter(item => item.type == type);
        }
        return Res3Utils.result(list);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 金融部-其他管理-合约交易限制配置
 * @param req
 * @param authConfigVO
 */
export const set = async (req: any, limitTradeVO: LimitTradeVO) => {
    let func_name = "contractLimitTrade.control.set";
    try {

        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        //1U本位，2币本位
        let type = Number(limitTradeVO.type);
        //1限制交易，2限制开仓，3取消限制交易，4取消限制开仓
        let status = Number(limitTradeVO.status);
        let pair = String(limitTradeVO.pair);
        let userId = Number(limitTradeVO.user_id);
        await writeCacheData(type, status, pair, userId);

        //管理后台操作日志
        addOptLog(currentUserId, 0, '合约限制交易配置-设置', ip, JSON.stringify(limitTradeVO), '合约限制交易配置');

        return Res3Utils.result("ok");
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

async function writeCacheData(type: number, status: number, pair: string, user_id: number) {
    let redisKey = 'contractLimitTrade.list';
    if (!type || !status) {
        throw ErrorCode.PARAM_MISS
    }
    if (!user_id) {
        user_id = 0
    }
    if (!pair) {
        pair = "all";
    }
    let typeStr = type == 1 ? "baseu.limit" : "basecoin.limit";
    let statusStr = status == 1 || status == 3 ? "trade" : "open";
    let mapKey = `${typeStr}.${statusStr}.${pair}.${user_id}`;

    let cacheMap = await RedisClient.getSync(redisKey);
    if (status == 1 || status == 2) {
        cacheMap = cacheMap ? cacheMap : {};
        cacheMap[mapKey] = 1
    }
    else {
        if (cacheMap) {
            delete cacheMap[mapKey]
        }
    }
    await RedisClient.writeSync(redisKey, cacheMap);

}



