import {
    ormDB,
    userLog,
    userInfo,
    userRealName,
    mainUserBills,
    dwsUserDailyTrade,
    dwsUserDailyFee,
    dwsMadOrmDB
} from "@madex/ex-ts-dao";
import BigNumber from "bignumber.js";


let _ = require('lodash');
let { tickerUtils } = require('@madex/ex-js-common');
let { datetimeUtils } = require('@madex/ex-js-public');


export interface QueryVO {
    page?: number,

    size?: number,

    is_export?: number,//是否导出

    from_time?: Date | any,

    to_time?: Date | any,

}


export async function increaseUserList(pageVO: QueryVO) {
    let where = {
        type: 22,//目前只有邮箱注册
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: pageVO.from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: pageVO.to_time },
            }]
    }
    let resList: any = [];
    if (!pageVO.is_export) {
        resList = await userLog.prototype.findAndCount({
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealIncreaseUserReturnData(resList.rows);
    }
    else {//导出不分页
        resList = await userLog.prototype.findAll({
            where: where,
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealIncreaseUserReturnData(resList);
    }
    return resList;
}


export async function tradeUserList(pageVO: QueryVO) {

    let where: any = {
        type: 0,//UTC 0
        category: 1,//现货
        [dwsMadOrmDB.Op.and]: [
            {
                trade_date: { [dwsMadOrmDB.Op.gte]: pageVO.from_time }
            },
            {
                trade_date: { [dwsMadOrmDB.Op.lt]: pageVO.to_time },
            }]
    };
    let resList: any = [];
    if (!pageVO.is_export) {
        resList = await dwsUserDailyTrade.prototype.findAndCount({
            attributes: ['user_id', dwsMadOrmDB.literal('sum(equal_usdt + equal_usdt_free) as trade_amount')],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [[dwsMadOrmDB.literal('trade_amount'), "desc"]],
            group: ['user_id'],
            raw: true
        });
        resList.count = resList.count.length;
        let uids = resList.rows.map(item => item.user_id);
        if (uids) {
            where["user_id"] = uids
            await dealFeeData(where, resList.rows);
        }
    }
    else {//导出不分页
        resList = await dwsUserDailyTrade.prototype.findAll({
            attributes: ['user_id', dwsMadOrmDB.literal('sum(equal_usdt + equal_usdt_free) as trade_amount')],
            where: where,
            order: [["trade_amount", "desc"]],
            group: ['user_id'],
            raw: true
        });
        let uids = resList.map(item => item.user_id);
        if (uids) {
            where["user_id"] = uids
            await dealFeeData(where, resList);
        }
    }
    return resList;
}

export async function activeUserList(pageVO: QueryVO) {

    if (!pageVO.is_export) {
        let res = {
            count: 0,
            rows: []
        }
        let count_sql = `SELECT COUNT(*) AS num
                         FROM user_log AS t1
                                  INNER JOIN
                              (SELECT user_id, MAX(createdAt) AS createdAt
                               FROM user_log
                               WHERE type = 0
                                 AND createdAt >= '${pageVO.from_time}'
                                 AND createdAt < '${pageVO.to_time}'
                               GROUP BY user_id) AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;

        let data_sql = `SELECT t1.*
                        FROM user_log AS t1
                                 INNER JOIN
                             (SELECT user_id, MAX(createdAt) AS createdAt
                              FROM user_log
                              WHERE type = 0
                                AND createdAt >= '${pageVO.from_time}'
                                AND createdAt < '${pageVO.to_time}'
                              GROUP BY user_id
                              ORDER BY createdAt DESC LIMIT ${(Number(pageVO.page) - 1) * Number(pageVO.size)}, ${Number(pageVO.size)})
                                 AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;
        let dbCount = await ormDB.query(count_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
        let dbData = await ormDB.query(data_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
        res.count = dbCount[0].num;
        res.rows = dbData;
        return res
    }
    else {//导出不分页
        let data_sql = `SELECT t1.*
                        FROM user_log AS t1
                                 INNER JOIN
                             (SELECT user_id, MAX(createdAt) AS createdAt
                              FROM user_log
                              WHERE type = 0
                                AND createdAt >= '${pageVO.from_time}'
                                AND createdAt < '${pageVO.to_time}'
                              GROUP BY user_id
                              ORDER BY createdAt DESC)
                                 AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;
        let dbData = await ormDB.query(data_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
        return dbData;
    }

}


export async function userDepositList(pageVO: QueryVO) {
    let where = {
        type: 100,//充值
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let dbDataList: any = [];
    if (!pageVO.is_export) {
        dbDataList = await mainUserBills.prototype.findAndCount({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList.rows);
    }
    else {//导出不分页
        dbDataList = await mainUserBills.prototype.findAll({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList);
    }

    return dbDataList;
}


export async function userWithdrawList(pageVO: QueryVO) {
    let where = {
        type: 102,//提现
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let dbDataList: any = [];
    if (!pageVO.is_export) {
        dbDataList = await mainUserBills.prototype.findAndCount({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList.rows);

    }
    else {//导出不分页
        dbDataList = await mainUserBills.prototype.findAll({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList);

    }

    return dbDataList;
}

export async function userDepositAndWithdrawList(pageVO: QueryVO) {
    let where = {
        type: { [ormDB.Op.in]: [100, 102] },//充值、提现
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let dbDataList: any = [];
    if (!pageVO.is_export) {
        dbDataList = await mainUserBills.prototype.findAndCount({
            attributes: ['user_id', 'symbol',
                ormDB.literal('sum(case when type = 100 then ABS(`change`) else 0 end) as deposit_amount'),
                ormDB.literal('sum(case when type = 102 then ABS(`change`) else 0 end) as withdraw_amount')
            ],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            group: ['user_id', 'symbol'],
            raw: true
        });
        //这里分组count的是每个分组的条数 count 实际 = 几个分组数
        dbDataList.count = dbDataList.count.length;
        await dealDepositAndWithdrawReturnData(dbDataList.rows);

    }
    else {//导出不分页
        dbDataList = await mainUserBills.prototype.findAll({
            attributes: ['user_id', 'symbol',
                ormDB.literal('sum(case when type = 100 then ABS(`change`) else 0 end) as deposit_amount'),
                ormDB.literal('sum(case when type = 102 then ABS(`change`) else 0 end) as withdraw_amount')
            ],
            where: where,
            group: ['user_id', 'symbol'],
            raw: true
        });
        await dealDepositAndWithdrawReturnData(dbDataList);

    }


    return dbDataList;
}


export async function gatherData(pageVO: QueryVO) {
    let [increaseObj, tradeObj, activeObj, depositAndWithdrawObj] =
        await Promise.all([increaseUserCount(pageVO), tradeUserCount(pageVO), activeUserCount(pageVO), depositAndWithdrawCount(pageVO)]);
    //处理日期
    let daysArr = getAllDay(pageVO.from_time, pageVO.to_time);

    let increase_data: any = [];
    let trade_data: any = [];
    let active_data: any = [];
    let deposit_data: any = [];
    let withdraw_data: any = [];
    let deposit_withdraw_data: any = [];

    for (let item of daysArr) {
        increase_data.push({
            time: item,
            data: increaseObj.increaseDataMap[item] || 0
        })

        trade_data.push({
            time: item,
            data: tradeObj.tradeDataMap[item] || 0
        })

        active_data.push({
            time: item,
            data: activeObj.activeDataMap[item] || 0
        })

        deposit_data.push({
            time: item,
            data: depositAndWithdrawObj.depositDataMap[item] || 0
        })

        withdraw_data.push({
            time: item,
            data: depositAndWithdrawObj.withdrawDataMap[item] || 0
        })
        deposit_withdraw_data.push({
            time: item,
            data: depositAndWithdrawObj.depositWithdrawDataMap[item] || 0
        })

    }

    return {
        increase: increaseObj.increaseCount,
        increase_data,
        trade: tradeObj.tradeCount,
        trade_data,
        active: activeObj.activeCount,
        active_data,
        deposit: depositAndWithdrawObj.deposit,
        deposit_data,
        withdraw: depositAndWithdrawObj.withdraw,
        withdraw_data,
        deposit_withdraw: depositAndWithdrawObj.deposit_withdraw,
        deposit_withdraw_data
    }
}

async function increaseUserCount(pageVO: QueryVO) {
    let where = {
        type: 22,//目前只有邮箱注册
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: pageVO.from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: pageVO.to_time },
            }]
    }
    let dbDataList = await userLog.prototype.findAll({
        attributes: [
            ormDB.literal(`DATE_FORMAT(createdAt,'%Y-%m-%d') AS dt`),
            ormDB.literal('COUNT(1) AS count')
        ],
        where: where,
        group: [`dt`],
        raw: true

    });
    let dataMap: any = {};
    let count = 0;
    for (let item of dbDataList) {
        count += item.count;
        dataMap[item.dt] = item.count;
    }

    return { increaseCount: count, increaseDataMap: dataMap };

}

async function tradeUserCount(pageVO: QueryVO) {

    let where: any = {
        type: 0,//UTC 0
        category: 1,//现货
        [dwsMadOrmDB.Op.and]: [
            {
                trade_date: { [dwsMadOrmDB.Op.gte]: pageVO.from_time }
            },
            {
                trade_date: { [dwsMadOrmDB.Op.lt]: pageVO.to_time },
            }]
    };

    let dbDataList = await dwsUserDailyTrade.prototype.findAll({
        attributes: ['user_id',
            dwsMadOrmDB.literal(`DATE_FORMAT(trade_date,'%Y-%m-%d') AS dt`),
            dwsMadOrmDB.literal('sum(equal_usdt + equal_usdt_free) AS amount')],
        where: where,
        group: [`dt`, 'user_id'],
        raw: true
    });

    let uidList: any = [];
    let dataMap: any = [];
    for (let item of dbDataList) {
        if (!uidList.includes) {
            uidList.push(item.user_id)
        }
        dataMap[item.dt] = dataMap[item.dt] ? Number(dataMap[item.dt]) + Number(item.amount) : Number(item.amount)

    }


    return {
        tradeCount: uidList.length, tradeDataMap: dataMap
    };
}


async function activeUserCount(pageVO: QueryVO) {

    let count_sql = `SELECT DATE_FORMAT(t2.createdAt, '%Y-%m-%d') AS dt, COUNT(t2.user_id) AS num
                     FROM user_log AS t1
                              INNER JOIN
                          (SELECT user_id, MAX(createdAt) AS createdAt
                           FROM user_log
                           WHERE type = 0
                             AND createdAt >= '${pageVO.from_time}'
                             AND createdAt < '${pageVO.to_time}'
                           GROUP BY user_id) AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt
                     GROUP BY dt`;

    let dbDataList = await ormDB.query(count_sql, { raw: true, type: ormDB.QueryTypes.SELECT });

    let count = 0;
    let dataMap: any = {}
    for (let item of dbDataList) {
        count += item.num;
        dataMap[item.dt] = item.num;
    }
    return {
        activeCount: count, activeDataMap: dataMap
    };
}

async function depositAndWithdrawCount(pageVO: QueryVO) {
    let res = {
        deposit: new BigNumber(0),
        withdraw: new BigNumber(0),
        deposit_withdraw: new BigNumber(0),
        depositDataMap: {},
        withdrawDataMap: {},
        depositWithdrawDataMap: {},
    }
    let where = {
        type: { [ormDB.Op.in]: [100, 102] },//充值、提现
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let amountList = await mainUserBills.prototype.findAll({
        attributes: ['symbol',
            ormDB.literal(`DATE_FORMAT(createdAt,'%Y-%m-%d') AS dt`),
            ormDB.literal('sum(case when type = 100 then ABS(`change`) else 0 end) as deposit_amount'),
            ormDB.literal('sum(case when type = 102 then ABS(`change`) else 0 end) as withdraw_amount'),
        ],
        where: where,
        group: [`dt`, 'symbol'],
        raw: true
    });
    let depositDataMap: any = {};
    let withdrawDataMap: any = {};
    let depositWithdrawDataMap: any = {};
    if (amountList.length) {
        let deposit_usdt = new BigNumber(0);
        let withdraw_usdt = new BigNumber(0);
        for (let item of amountList) {
            let symbol = item.symbol;
            let deposit_amount = item.deposit_amount;
            let withdraw_amount = item.withdraw_amount;
            let usdt = await tickerUtils.rateCoin2USDT(symbol);
            let deposit_amount_usdt = new BigNumber(deposit_amount).mul(new BigNumber(usdt));
            deposit_usdt = deposit_usdt.add(deposit_amount_usdt);

            depositDataMap[item.dt] = depositDataMap[item.dt] ? depositDataMap[item.dt].add(deposit_amount_usdt) : deposit_amount_usdt;

            let withdraw_amount_usdt = new BigNumber(withdraw_amount).mul(new BigNumber(usdt));
            withdraw_usdt = withdraw_usdt.add(withdraw_amount_usdt);

            withdrawDataMap[item.dt] = withdrawDataMap[item.dt] ? withdrawDataMap[item.dt].add(withdraw_amount_usdt) : withdraw_amount_usdt;

            let d_w_usdt = deposit_amount_usdt.sub(withdraw_amount_usdt);

            depositWithdrawDataMap[item.dt] = depositWithdrawDataMap[item.dt] ? depositWithdrawDataMap[item.dt].add(d_w_usdt) : d_w_usdt;


        }
        let deposit_withdraw_usdt = deposit_usdt.sub(withdraw_usdt);

        res.deposit = deposit_usdt;
        res.withdraw = withdraw_usdt;
        res.deposit_withdraw = deposit_withdraw_usdt;
        res.depositDataMap = depositDataMap;
        res.withdrawDataMap = withdrawDataMap;
        res.depositWithdrawDataMap = depositWithdrawDataMap;
    }
    return res;
}

async function dealIncreaseUserReturnData(resList: any) {
    if (resList.length) {
        let uids = resList.map(item => item.user_id);
        let dbUserListTask = userInfo.prototype.findAll({
            attributes: ['user_id', 'createdAt', 'user_from', 'invited_by_id', 'is_active'],
            where: {
                user_id: { [ormDB.Op.in]: uids }
            },
            raw: true
        });

        let dbRealNameListTask = userRealName.prototype.findAll({
            attributes: ['user_id', 'status'],
            where: {
                user_id: { [ormDB.Op.in]: uids }
            },
            raw: true
        })
        let [dbUserList, dbRealNameList] = await Promise.all([dbUserListTask, dbRealNameListTask]);
        let userMap = {};
        let realNameMap = {};
        for (let item of dbUserList) {
            userMap[item.user_id] = item;
        }

        for (let item of dbRealNameList) {
            realNameMap[item.user_id] = item;
        }
        for (let item of resList) {
            let userId = item.user_id;
            item.createdAt = userMap[userId] ? userMap[userId].createdAt : item.createdAt;
            item.user_from = userMap[userId] ? userMap[userId].user_from : 0;
            item.invited_by_id = userMap[userId] ? userMap[userId].invited_by_id : '';
            item.is_active = userMap[userId] ? userMap[userId].is_active : 0;
            item.status = realNameMap[userId] ? realNameMap[userId].status : 0;
        }
    }
}


async function dealDepositOrWithdrawReturnData(resList: any) {
    if (resList.length) {
        let usdtTickerMap = {};
        for (let item of resList) {
            let symbol = item.symbol;
            let change = item.change;
            let usdt = usdtTickerMap[symbol] ? usdtTickerMap[symbol] : await tickerUtils.rateCoin2USDT(symbol);
            usdtTickerMap[symbol] = usdt;
            item.equal_usdt = new BigNumber(usdt).mul(change);
        }
    }
}

async function dealDepositAndWithdrawReturnData(resList: any) {
    if (resList.length) {
        let usdtTickerMap = {};
        for (let item of resList) {
            let symbol = item.symbol;
            let deposit_amount = item.deposit_amount;
            let withdraw_amount = item.withdraw_amount;
            let usdt = usdtTickerMap[symbol] ? usdtTickerMap[symbol] : await tickerUtils.rateCoin2USDT(symbol);
            usdtTickerMap[symbol] = usdt;
            item.deposit_usdt = new BigNumber(usdt).mul(deposit_amount);
            item.withdraw_usdt = new BigNumber(usdt).mul(withdraw_amount);
            item.deposit_withdraw_usdt = item.deposit_usdt.sub(item.withdraw_usdt);
        }
    }
}

async function dealFeeData(where: any, dbList: any) {
    let dbFeeList = await dwsUserDailyFee.prototype.findAll({
        attributes: ['user_id', dwsMadOrmDB.literal('sum(equalusdt_amount) as fee_amount')],
        where: where,
        group: ['user_id'],
        raw: true
    });
    let fee_map: any = {};
    for (let item of dbFeeList) {
        fee_map[item.user_id] = item.fee_amount;
    }
    for (let item of dbList) {
        item.fee_amount = fee_map[item.user_id] ? fee_map[item.user_id] : 0
    }
}

/**
 * 获取开始到结束每一天
 * @param start
 * @param end
 */
function getAllDay(start: any, end: any) {
    let startTime = datetimeUtils.trim(start, 'd');
    let endTime = datetimeUtils.trim(end, 'd');
    let dateArr: any = [];
    while ((endTime.getTime() - startTime.getTime()) > 0) {
        let year = startTime.getFullYear();
        let month = (startTime.getMonth() + 1).toString().length === 1 ? "0" + (parseInt(startTime.getMonth().toString(), 10) + 1) : (startTime.getMonth() + 1);
        let day = startTime.getDate().toString().length === 1 ? "0" + startTime.getDate() : startTime.getDate();
        dateArr.push(year + "-" + month + "-" + day);
        startTime.setDate(startTime.getDate() + 1);
    }
    return dateArr;
}

//let allDay = getAllDay("2024-11-11T00:00:00.000Z","2024-11-11T23:59:59.999Z");






