import { madAdminOrmDB, aclUserInfo, ormDB, userInfo, userRealName } from "@madex/ex-ts-dao";
import { isLimitUserId, UID_MIN, UID_LIMIT } from "../../../utils/mUserCommonUtils";
import { ErrorCode } from "../../../constant/errorCode";
import * as ossUtils from "../../../utils/OSSUtils";
import { getMUserInfoByUid, sendEmail } from "../../../utils/mUserUtils";
import { EMAIL_FORBID_TEMPLATE, REAL_NAME_DENY, REAL_NAME_PASS } from "../../../constant/emailTemplateType";
import { KYC_STATUS, SETTING_FLAG } from "../../../constant/mUserInfoConst";
import { addOptLog, LogType } from "./userOptLog.service";
import { recordMUserOperateLog, TYPE } from "./mUserAccountOperateLog.service";
import { isNaN } from "lodash";

let { logger, } = require('@madex/ex-js-public');


let _ = require('lodash');

export interface QueryVO {

    page?: number,

    size?: number

    id?: number | any;

    user_id?: number | any;

    totp_code?: string | any

    status?: number | any

    audit_deny?: string,

    comment?: string

    type?: number//证件类型 1身份证，2护照，3驾驶证

    source?: number// 认证来源  1 web  2 app

    auditor?: number//审核人员id

    from_time?: Date | any

    to_time?: Date | any
}

export async function getRealNameByUserId(user_id: number | any) {
    let dbInfo = await userRealName.prototype.findOne({
        where: {
            user_id: user_id
        },
        raw: true
    });
    return dbInfo ? dbInfo : null
}

export async function getRealNameMapByUserIds(userIds: number[]) {

    let dbInfoList = await userRealName.prototype.findAll({
        where: {
            user_id: { [ormDB.Op.in]: userIds }
        },
        raw: true
    });
    let resMap = {};
    for (let item of dbInfoList) {
        resMap[item.user_id] = item
    }
    return resMap;
}


export async function kycList(queryVO: QueryVO) {
    let where = {}
    if (queryVO.user_id) {
        //uid 是否在受限范围内
        if (isLimitUserId(queryVO.user_id)) {
            throw ErrorCode.UID_LIMIT
        }
        where["user_id"] = queryVO.user_id;
    }
    else {
        if (UID_MIN && UID_LIMIT.length) {
            where[ormDB.Op.and] = [{
                user_id: { [ormDB.Op.gte]: UID_MIN }
            },
                {
                    user_id: { [ormDB.Op.notIn]: UID_LIMIT },
                }]
        }
        else {
            where["user_id"] = { [ormDB.Op.gte]: UID_MIN }
        }
    }
    if (!isNaN(Number(queryVO.status))) {
        where["status"] = queryVO.status
    }
    if (queryVO.source) {
        //来源
        if (queryVO.source == 1) {
            where["delta"] = ""
        }
        else {
            where[ormDB.Op.and] = [{
                delta: { [ormDB.Op.not]: null }
            },
                {
                    delta: { [ormDB.Op.ne]: "" },
                }]
        }

    }
    if (queryVO.type) {
        where["type"] = queryVO.type
    }
    if (queryVO.auditor) {
        where["auditor"] = queryVO.auditor
    }
    if (queryVO.from_time && queryVO.to_time) {
        where['updatedAt'] = { [ormDB.Op.between]: [queryVO.from_time, queryVO.to_time] }
    }

    let resList = await userRealName.prototype.findAndCount({
        attributes: ['id', 'user_id', 'status', 'createdAt'],
        where: where,
        limit: queryVO.size,
        offset: (Number(queryVO.page) - 1) * Number(queryVO.size),
        order: [["updatedAt", "desc"]],
        raw: true
    });

    if (resList.rows.length) {
        let uids = resList.rows.map(item => item.user_id);
        let userMap = {};
        let dbUserInfos = await userInfo.prototype.findAll({
            attributes: ['user_id', 'email'],
            where: {
                user_id: { [ormDB.Op.in]: uids }
            },
            raw: true
        });
        for (let dbUserInfo of dbUserInfos) {
            userMap[dbUserInfo.user_id] = dbUserInfo.email;
        }
        for (let item of resList.rows) {
            item.email = userMap[item.user_id] ? userMap[item.user_id] : ""
        }
    }

    return resList;
}


export async function oneDetail(id: number) {

    let dbOne = await userRealName.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!dbOne) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    //uid 是否在受限范围内
    if (isLimitUserId(dbOne.user_id)) {
        throw ErrorCode.UID_LIMIT
    }

    let dbUserInfo = await userInfo.prototype.findOne({
        attributes: ['user_id', 'email'],
        where: {
            user_id: dbOne.user_id
        },
        raw: true
    });
    dbOne.email = dbUserInfo ? dbUserInfo.email : "";

    await updateImageUrl(dbOne);
    return dbOne;
}


export async function audit(queryVO: QueryVO, currentUserId: any, ip: string | undefined) {

    let dbOne = await userRealName.prototype.findOne({
        where: {
            id: queryVO.id
        },
        raw: true
    });
    if (!dbOne) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    //uid 是否在受限范围内
    if (isLimitUserId(dbOne.user_id)) {
        throw ErrorCode.UID_LIMIT
    }

    if (queryVO.audit_deny && queryVO.audit_deny.length > 250) {
        throw ErrorCode.REASON_TOO_LONG
    }

    let dbUserInfo = await getMUserInfoByUid(dbOne.user_id);
    if (!dbUserInfo) {
        throw ErrorCode.USER_NOT_EXIST
    }

    let settingFlag = dbUserInfo.setting_flag;
    let dbEmail = dbUserInfo.email;

    let remark = "用户实名认证审核不通过";
    let template = "";
    if (!queryVO.audit_deny) {
        remark = "用户实名认证审核通过";
        template = REAL_NAME_PASS;
        if (dbOne.status == KYC_STATUS.STATUS_AUDIT_PASS) {
            throw ErrorCode.DATA_STATUS_CHANGED
        }
        let firstName = dbOne.first_name;
        let realName = dbOne.real_name;
        // firstName和realName字段不为空，且是国内数据认证；并且realName中不包含firstName，则将firstName拼接到realName中。
        if (firstName && realName && "cn" == dbOne.country.toLowerCase()
            && "test" != firstName && !realName.startsWith(firstName)) {
            realName = firstName + realName;
        }
        await auditPass(dbOne.id, dbUserInfo.user_id, settingFlag, realName, currentUserId);
    }
    else {
        template = REAL_NAME_DENY
        if (dbOne.status == KYC_STATUS.STATUS_AUDIT_DENY) {
            throw ErrorCode.DATA_STATUS_CHANGED
        }
        await auditDeny(dbOne.id, dbUserInfo.user_id, settingFlag, queryVO.audit_deny, currentUserId, queryVO.comment ? queryVO.comment : "")
    }

    //发邮件
    sendEmail(dbEmail, dbUserInfo.user_id, template);
    //日志记录
    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},实名认证审核:${dbUserInfo.user_id} ， ${remark}`, LogType.KYC_AUDIT);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(dbUserInfo.user_id, currentUserId, TYPE.KYC, `ip:${ip},实名认证审核:${dbUserInfo.user_id} ， ${remark}`, "", "");
    return 'success';
}

export async function auditors() {

    let auditors = await userRealName.prototype.findAll({
        attributes: [ormDB.literal('DISTINCT(auditor) as auditor')],
        where: {},
        raw: true
    });

    let where = {};
    if (auditors.length) {
        where['user_id'] = { [madAdminOrmDB.Op.in]: auditors.map(item => item.auditor) }
    }
    let resList = await aclUserInfo.prototype.findAll({
        attributes: ['user_id', 'account', 'remark'],
        where: where,
        raw: true
    });
    return resList;

}

/**
 * 根据图片的情况设置缩略图
 * @param dbRealNameInfo
 */
async function updateImageUrl(dbRealNameInfo: any) {
    let identityFrondSide = dbRealNameInfo.identity_frond_side;
    if (identityFrondSide) {
        if (!identityFrondSide.includes("http")) {
            dbRealNameInfo.identity_frond_side_small = ossUtils.resizeWithWaterMark(identityFrondSide, "Madex", 300);
            dbRealNameInfo.identity_frond_side = ossUtils.withWaterMark(identityFrondSide, "Madex");
        }
        else {
            dbRealNameInfo.identity_frond_side_small = identityFrondSide
        }
    }
    else {
        dbRealNameInfo.identity_frond_side_small = ""

    }

    let identityOtherSide = dbRealNameInfo.identity_other_side;
    if (identityOtherSide) {
        if (!identityOtherSide.includes("http")) {
            dbRealNameInfo.identity_other_side_small = ossUtils.resizeWithWaterMark(identityOtherSide, "Madex", 300);
            dbRealNameInfo.identity_other_side = ossUtils.withWaterMark(identityOtherSide, "Madex");
        }
        else {
            dbRealNameInfo.identity_other_side_small = identityOtherSide
        }
    }
    else {
        dbRealNameInfo.identity_other_side_small = ""

    }


    let identityInHand = dbRealNameInfo.identity_in_hand;
    if (identityInHand) {
        if (!identityInHand.includes("http")) {
            dbRealNameInfo.identity_in_hand_small = ossUtils.resizeWithWaterMark(identityInHand, "Madex", 300);
            dbRealNameInfo.identity_in_hand = ossUtils.withWaterMark(identityInHand, "Madex");
        }
        else {
            dbRealNameInfo.identity_in_hand_small = identityInHand
        }
    }
    else {
        dbRealNameInfo.identity_in_hand_small = ""

    }


    let latestPhoto = dbRealNameInfo.latest_photo;
    if (latestPhoto) {
        if (!latestPhoto.includes("http")) {
            dbRealNameInfo.latest_photo_small = ossUtils.resizeWithWaterMark(latestPhoto, "Madex", 300);
            dbRealNameInfo.latest_photo = ossUtils.withWaterMark(latestPhoto, "Madex");
        }
        else {
            dbRealNameInfo.latest_photo_small = latestPhoto
        }
    }
    else {
        dbRealNameInfo.latest_photo_small = ""

    }


}

async function auditPass(id: number, m_user_id: number, setting_flag: any, real_name: string, operator: number) {

    let tx;
    try {
        tx = await ormDB.transaction();
        await userInfo.prototype.update({
            real_name: real_name,
            setting_flag: setting_flag | SETTING_FLAG.FLAG_REAL_NAME_AUTH,
            updatedAt: new Date()
        }, {
            where: {
                user_id: m_user_id,
                setting_flag: setting_flag
            },
            transaction: tx,
        });

        await userRealName.prototype.update({
            status: KYC_STATUS.STATUS_AUDIT_PASS,
            auditor: operator,
            updatedAt: new Date()
        }, {
            where: {
                id: id
            },
            transaction: tx,
        })

        await tx.commit()
    }
    catch (e) {
        logger.error("mUserRealNameService.auditPass.err:" + e);
        if (tx) {
            await tx.rollback();
        }
        throw e;
    }
}

async function auditDeny(id: number, m_user_id: number, setting_flag: any, audit_deny: string, operator: number, comment: string) {


    let tx;
    try {
        tx = await ormDB.transaction();
        await userInfo.prototype.update({
            real_name: "",
            setting_flag: setting_flag & (~SETTING_FLAG.FLAG_REAL_NAME_AUTH),
            updatedAt: new Date()
        }, {
            where: {
                user_id: m_user_id,
                setting_flag: setting_flag
            },
            transaction: tx,
        });

        await userRealName.prototype.update({
            status: KYC_STATUS.STATUS_AUDIT_DENY,
            auditor: operator,
            auth_deny: audit_deny,
            comment: comment,
            updatedAt: new Date()
        }, {
            where: {
                id: id
            },
            transaction: tx,
        })

        await tx.commit()
    }
    catch (e) {
        logger.error("mUserRealNameService.auditDeny.err:" + e);
        if (tx) {
            await tx.rollback();
        }
        throw e;
    }
}



