// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { spotPairs, madSpotOrmDB, coinType } from "@madex/ex-ts-dao";
import { NUMBER } from "sequelize";
import { addPairToCore } from "../../../utils/coreSystemUtils";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { checkPairInCoinType } from "../../../utils/coinTypeUtils";

let { logger, datetimeUtils } = require('@madex/ex-js-public');

export interface ListParam {
    symbol?: string | any;

    page: number;

    size: number;

    status?: number

    from_time?: Date | any

    to_time?: Date | any
}

export interface AddParam {
    id?: number;

    symbol?: string;

    product_id?: number;

    base?: string;

    quote?: string;

    name?: string;

    price_scale?: number;

    quantity_scale?: number;

    maker_fee?: string;

    taker_fee?: string;

    price_increment?: number;

    quantity_increment?: number;

    status?: number

    view_precision?: number

    min_order_size?: number

    max_order_size?: number

    min_order_value?: number

    max_order_value?: number

    createdAt?: Date | any,

    updatedAt?: Date | any,
}


export async function list(param: ListParam) {
    let where = {};
    if (param.symbol) {
        where["symbol"] = { [madSpotOrmDB.Op.like]: `%${param.symbol}%` };
    }
    if (param.status || param.status === 0) {
        where["status"] = param.status
    }
    if (param.from_time && param.to_time) {
        where['createdAt'] = { [madSpotOrmDB.Op.between]: [param.from_time, param.to_time] }
    }
    let resList = await spotPairs.prototype.findAndCount({
        where: where,
        limit: param.size,
        offset: (Number(param.page) - 1) * Number(param.size),
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}

export const save = async (param: AddParam, currentUserId: any, ip: any) => {

    //检查交易对中的币种是否存在于coin_type 中
    await checkPairInCoinType(String(param.symbol));

    param.updatedAt = new Date();
    param.createdAt = new Date();
    let dbInfo = await spotPairs.prototype.findOne({
        where: {
            symbol: param.symbol
        },
        raw: true
    });
    if (dbInfo) {
        throw ErrorCode.DATA_EXIST;
    }

    if (!param.price_increment) {
        param.price_increment = 1;
    }
    if (!param.quantity_increment) {
        param.quantity_increment = 1;
    }
    await spotPairs.prototype.create(param);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '交易对配置新增', ip, JSON.stringify(param), '交易对配置');
    return 'ok';
};


export const update = async (param: AddParam, currentUserId: any, ip: any) => {
    let dbInfo = await spotPairs.prototype.findOne({
        where: {
            id: param.id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    let updateInfo = { updatedAt: new Date() };
    if (param.view_precision || param.view_precision === 0) {
        updateInfo['view_precision'] = param.view_precision
    }
    if (param.price_increment) {
        updateInfo['price_increment'] = param.price_increment
    }
    if (param.quantity_increment) {
        updateInfo['quantity_increment'] = param.quantity_increment
    }

    await spotPairs.prototype.update(updateInfo, {
        where: {
            id: Number(param.id)
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '交易对配置更新', ip, JSON.stringify(param), '交易对配置');
    return 'ok';
};

export const pushToCoreSystem = async (id: any, currentUserId: any, ip: any) => {
    let where = { id: id, status: 0 };

    let cfg = await spotPairs.prototype.find({
        where: where, raw: true
    });
    if (!cfg) {
        throw ErrorCode.SUBMIT_STATUS_ERR;
    }
    let symbol = cfg.symbol;
    let sps = symbol.split("_");
    let base = sps[0];
    let quote = sps[1]
    let param = {
        "base": base, "quote": quote, "symbol": symbol, "name": symbol,
        "price_scale": cfg.price_scale, "quantity_scale": cfg.quantity_scale,
        "maker_fee": cfg.maker_fee, "taker_fee": cfg.taker_fee
    }
    if (cfg.min_order_size) {
        param["min_order_size"] = cfg.min_order_size;
    }
    if (cfg.max_order_size) {
        param["max_order_size"] = cfg.max_order_size;
    }
    if (cfg.min_order_value) {
        param["min_order_value"] = cfg.min_order_value;
    }
    if (cfg.max_order_value) {
        param["max_order_value"] = cfg.max_order_value;
    }

    let optResult = await addPairToCore(param);
    if (!optResult.is_success) {
        //管理后台操作日志
        addOptLog(currentUserId, 0, '提交交易对到撮合-失败', ip, `pair:${symbol},error_msg:${optResult.err_msg}`, '交易对配置');
        throw ErrorCode.ADD_PAIR_TO_CORE_ERR;
    }

    await spotPairs.prototype.update({ status: 1 }, {
        where: where
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '提交交易对到撮合-成功', ip, JSON.stringify(param), '交易对配置');
    return 'ok';
};

export async function getAllSubmitSuccess() {
    let resList = await spotPairs.prototype.findAll({
        where: {
            status: 2
        },
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}


