import * as userOptService from "../service/userOpt.service";
import { AclUserInfoVO, AclUserInfoPageVO } from "../service/aclUser.service";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUser, getCurrentUserId, } from "../../../utils/aclUserUtils";

let isIp = require('is-ip');

/**
 * 获取当前登陆的用户信息
 * @param req
 * @param infoVO
 */
export const getInfo = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.getInfo";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await userOptService.getInfo(currentUserId, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 获取用户信息根据uid
 * @param req
 * @param infoVO
 */
export const getInfoByUserId = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.getInfoByUserId";
    let cmd = req.path;
    try {
        if (!aclUserInfoVO.user_id){
            throw ErrorCode.PARAM_MISS
        }
        let res = await userOptService.getInfoByUserId(aclUserInfoVO.user_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 获取用户信息详情（这里主要包含密码、google等敏感信息）根据uid
 * @param req
 * @param infoVO
 */
export const getInfoDetailByUserId = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.getInfoDetailByUserId";
    let cmd = req.path;
    try {
        if (!aclUserInfoVO.user_id){
            throw ErrorCode.PARAM_MISS
        }
        let res = await userOptService.getInfoDetailByUserId(aclUserInfoVO.user_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 登陆
 * @param req
 * @param infoVO
 */
export const login = async (req: any, aclUserInfoVO: AclUserInfoVO, res: any) => {
    let func_name = "userOptCtrl.login";
    let cmd = req.path;
    let result: any;
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';

        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, aclUserInfoVO.account);
        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, aclUserInfoVO.pwd);
        ApiAssert.isFalse(ErrorCode.ACCOUNT_OR_PWD_ERR, aclUserInfoVO.pwd.length < 8);
        ApiAssert.isFalse(ErrorCode.ACCOUNT_OR_PWD_ERR, aclUserInfoVO.pwd.length > 30);

        result = await userOptService.login(aclUserInfoVO.account, aclUserInfoVO.pwd, ip || "");
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
    if (result.result && result.sessionId) {
        if (req.headers.origin){
            let iSecure = req.headers.origin.startsWith("https");

            let options = {
                expires: new Date(new Date().getTime() + 1000 * 60 * 60 * 8), // 15 days
                httpOnly: true,
                secure: iSecure
            };

            res.cookie('session_id', result.sessionId, options);
            delete result.sessionId;
        }
    }
    return Res3Utils.result(result);
};

/**
 * 登出
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const logout = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.logout";
    let cmd = req.path;
    try {
        if (req.cookies.session_id) {
            let currentUser = await getCurrentUser(req.cookies.session_id);
            if (currentUser) {
                await userOptService.deleteAllSessionByUserId(currentUser.userId);
            }
        }
        return Res3Utils.result('ok');
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};

/**
 * 登录-二次验证
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const loginConfirm = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.loginConfirm";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totpCode);

        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await userOptService.loginConfirm(req.cookies.session_id, currentUserId, aclUserInfoVO.totpCode);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};




/**
 * 绑定谷歌-生成新的密钥
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const bindTotpAsk = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.bindTotpAsk";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await userOptService.bindTotpAsk(req.cookies.session_id, currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};

/**
 * 绑定谷歌-验证新密钥
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const bindTotpConfirm = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.bindTotpConfirm";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totpCode);
        let res = await userOptService.bindTotpConfirm(req.cookies.session_id, currentUserId, aclUserInfoVO.totpCode);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};


