// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { madAdminOrmDB, aclDepartment, aclUserDepartmentPosition, aclPosition, aclUserInfo } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog, } from "./userOptLog.service";

let _ = require("lodash");

let { logger } = require('@madex/ex-js-public');

export interface AclDepartmentVO {
    id?: number;

    pid?: number;

    name?: string | any;

    level?: number | any;

    remark?: string | any;

    createdAt?: Date | any;

    updatedAt?: Date | any;

}

export interface AclDepartmentPageVO extends AclDepartmentVO {
    page?: number,

    size?: number

    account?: string | any

    dIds?: string | any
}

export const getAllDepartment = async function () {

    let res = await aclDepartment.prototype.findAll({
        raw: true,
    });
    return res
}

export const pageList = async function (aclDepartmentPageVO: AclDepartmentPageVO) {
    let where = {};
    if (aclDepartmentPageVO.name) {
        where["name"] = { [madAdminOrmDB.Op.like]: `${aclDepartmentPageVO.name}%` }
    }
    if (aclDepartmentPageVO.level) {
        where["level"] = aclDepartmentPageVO.level
    }
    let size = Number(aclDepartmentPageVO.size);
    let page = Number(aclDepartmentPageVO.page);
    let resList = await aclDepartment.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["level", "asc"]],
        raw: true
    });

    return resList;
}

export const allList = async function () {

    let resList = await aclDepartment.prototype.findAll({
        order: [["level", "asc"]],
        raw: true
    });

    return resList;
}

export const getDepartmentTree = async function () {

    let departmentMap = {};
    let resList: any = [];
    let departmentList = await aclDepartment.prototype.findAll({
        attributes: ['id', 'pid', 'name', 'level'],
        order: [["pid"], ["level"]],
        raw: true
    });
    let departmentUserMap = await getAllUserDepartmentPositionMap();
    for (let oneDepartment of departmentList) {
        departmentMap[oneDepartment.id] = oneDepartment;
        let pid = oneDepartment.pid;
        if (Number(pid) === -1) {
            let departmentIds: number[] = [oneDepartment.id];
            oneDepartment.department_ids = departmentIds;
            resList.push(oneDepartment);
            oneDepartment.userList = departmentUserMap[departmentIds.toString()] ? departmentUserMap[departmentIds.toString()] : []
        }
        else {
            let pDepartment = departmentMap[pid];
            if (!pDepartment) {
                logger.warn("上级部门被删除或者不存在");
                continue
            }
            else {
                let departmentIds = pDepartment.department_ids;
                let tmp: number[] = [];
                tmp.push(...departmentIds, oneDepartment.id);
                oneDepartment.department_ids = tmp;
                oneDepartment.userList = departmentUserMap[tmp.toString()] ? departmentUserMap[tmp.toString()] : []
            }
            if (!pDepartment['nextList']) {
                pDepartment['nextList'] = [];
            }
            pDepartment['nextList'].push(oneDepartment);
        }

    }

    return resList;
}


export const add = async function (aclDepartmentVO: AclDepartmentVO, currentUserId: any, ip: string | undefined) {
    if (!aclDepartmentVO.name || !aclDepartmentVO.level) {
        throw ErrorCode.PARAM_MISS
    }
    await aclDepartment.prototype.create({
        pid: aclDepartmentVO.pid ? aclDepartmentVO.pid : -1,
        name: aclDepartmentVO.name,
        level: aclDepartmentVO.level,
        createdAt: new Date(),
        updatedAt: new Date(),
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增部门', ip, JSON.stringify(aclDepartmentVO), '组织结构');
    return 'success'
}

export const update = async function (aclDepartmentVO: AclDepartmentVO, currentUserId: any, ip: string | undefined) {
    if (!aclDepartmentVO.id) {
        throw ErrorCode.PARAM_MISS
    }
    let dbInfo = await aclDepartment.prototype.findOne({
        where: { id: aclDepartmentVO.id },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let dbPid = dbInfo.pid;
    //要修改部门的上级，需要确定当前部门下没有用户
    if (aclDepartmentVO.pid && aclDepartmentVO.pid != dbPid) {
        let udpInfo = await aclUserDepartmentPosition.prototype.findOne({
            where: {
                department_id: { [madAdminOrmDB.Op.like]: `%${aclDepartmentVO.id}%` }
            },
            raw: true
        });
        if (udpInfo) {
            throw ErrorCode.DEPARTMENT_HAS_USER
        }
    }

    let updateInfo = {
        updatedAt: new Date(),
    };
    if (aclDepartmentVO.name) {
        updateInfo["name"] = aclDepartmentVO.name
    }
    if (aclDepartmentVO.pid) {
        updateInfo["pid"] = aclDepartmentVO.pid
    }
    if (aclDepartmentVO.level) {
        updateInfo["level"] = aclDepartmentVO.level
    }


    await aclDepartment.prototype.update(updateInfo, {
        where: {
            id: aclDepartmentVO.id
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '修改部门', ip, JSON.stringify(aclDepartmentVO), '组织结构');
    return 'success'
}


export const del = async function (id: number, currentUserId: any, ip: string | undefined) {
    if (!id) {
        throw ErrorCode.PARAM_MISS
    }
    let dbInfo = await aclDepartment.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    //要删除部门，需要确定当前部门下没有用户
    let udpInfo = await aclUserDepartmentPosition.prototype.findOne({
        where: {
            department_id: { [madAdminOrmDB.Op.like]: `%${id}%` }
        },
        raw: true
    });
    if (udpInfo) {
        throw ErrorCode.DEPARTMENT_HAS_USER
    }


    await aclDepartment.prototype.destroy({
        where: {
            id: id
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '删除部门', ip, JSON.stringify(dbInfo), '组织结构');
    return 'success'
}

export const getUserList = async function (aclDepartmentPageVO: AclDepartmentPageVO) {
    let page = Number(aclDepartmentPageVO.page);
    let size = Number(aclDepartmentPageVO.size);
    let account = aclDepartmentPageVO.account;
    let dIds = aclDepartmentPageVO.dIds;
    let pageData: any;
    //部门 和 account 都作为查询条件时 先查用户
    if (dIds && account) {
        let userList = await getAclUserData(account);
        //没有直接返回
        if (!userList.length) {
            pageData = {
                count: 0,
                rows: []
            }
            return pageData;
        }
        let uids: number[] = [];
        let userMap = {}
        for (let item of userList) {
            userMap[item.user_id] = item;
            uids.push(item.user_id);
        }
        //有则分页查询部门用户
        pageData = await getDpUserPageData(dIds, page, size, uids);
        for (let oneItem of pageData.rows) {
            oneItem.account = userMap[oneItem.user_id] ? userMap[oneItem.user_id].account : "";
            oneItem.phone = userMap[oneItem.user_id] ? userMap[oneItem.user_id].phone : "";
            oneItem.email = userMap[oneItem.user_id] ? userMap[oneItem.user_id].email : "";
            oneItem.user_status = userMap[oneItem.user_id] ? userMap[oneItem.user_id].user_status : "";
            oneItem.remark = userMap[oneItem.user_id] ? userMap[oneItem.user_id].remark : "";
        }
    }
    else if (account) {
        pageData = await getAclUserPageData(account, page, size);
        if (pageData.rows && pageData.rows.length) {
            let uids = pageData.rows.map(item => item.user_id);
            let udpList = await aclUserDepartmentPosition.prototype.findAll({
                where: {
                    user_id: { [madAdminOrmDB.Op.in]: uids }
                },
                raw: true
            });
            let udpMap = {}
            for (let item of udpList) {
                udpMap[item.user_id] = item;
            }
            for (let oneItem of pageData.rows) {
                oneItem.department_id = udpMap[oneItem.user_id] ? udpMap[oneItem.user_id].department_id : "";
                oneItem.position_id = udpMap[oneItem.user_id] ? udpMap[oneItem.user_id].position_id : "";
            }

        }
    }
    else {
        pageData = await getDpUserPageData(dIds, page, size, []);
        if (pageData.rows && pageData.rows.length) {
            let uids = pageData.rows.map(item => item.user_id);
            let userInfoList = await aclUserInfo.prototype.findAll({
                where: {
                    user_id: { [madAdminOrmDB.Op.in]: uids }
                },
                raw: true
            });
            let userMap = {}
            for (let item of userInfoList) {
                userMap[item.user_id] = item;
            }
            for (let oneItem of pageData.rows) {
                oneItem.account = userMap[oneItem.user_id] ? userMap[oneItem.user_id].account : "";
                oneItem.phone = userMap[oneItem.user_id] ? userMap[oneItem.user_id].phone : "";
                oneItem.email = userMap[oneItem.user_id] ? userMap[oneItem.user_id].email : "";
                oneItem.user_status = userMap[oneItem.user_id] ? userMap[oneItem.user_id].user_status : "";
                oneItem.remark = userMap[oneItem.user_id] ? userMap[oneItem.user_id].remark : "";
            }

        }
    }
    return pageData;
}

async function getDpUserPageData(dIds: string, page: number, size: number, uids: number[]) {
    let where = {};
    if (dIds) {
        where['department_id'] = { [madAdminOrmDB.Op.like]: `${dIds}%` };
    }
    if (uids.length) {
        where['user_id'] = { [madAdminOrmDB.Op.in]: uids }
    }
    let resList = await aclUserDepartmentPosition.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
}

async function getAclUserPageData(account: string, page: number, size: number) {
    let where = {}
    if (!isNaN(Number(account))) {
        where = {
            [madAdminOrmDB.Op.or]: {
                user_id: Number(account),
                account: { [madAdminOrmDB.Op.like]: `${account}%` },
                email: { [madAdminOrmDB.Op.like]: `${account}%` },
            }
        }
    }
    else {
        where = {
            [madAdminOrmDB.Op.or]: {
                account: { [madAdminOrmDB.Op.like]: `${account}%` },
                email: { [madAdminOrmDB.Op.like]: `${account}%` },
            }
        }
    }
    let resList = await aclUserInfo.prototype.findAndCount({
        attributes: ['user_id', 'account', 'phone', 'email', 'user_status', 'remark'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["user_id", "asc"]],
        raw: true
    });
    return resList;

}

async function getAllUserDepartmentPositionMap() {
    let udpList = await aclUserDepartmentPosition.prototype.findAll({
        attributes: ['user_id', 'department_id', 'position_id'],
        raw: true
    });
    let positionList = await aclPosition.prototype.findAll({
        raw: true
    });
    let positionMap = {};
    for (let item of positionList) {
        positionMap[item.id] = item.name;
    }

    let userList = await aclUserInfo.prototype.findAll({
        raw: true
    });
    let userMap = {};
    for (let item of userList) {
        userMap[item.user_id] = item.remark;
    }

    for (let item of udpList) {
        let positionId = item.position_id;
        let ids = positionId.split(",");
        let positionNameList: any = [];
        if (ids.length) {
            for (let positionId of ids) {
                positionNameList.push(positionMap[positionId]);
            }
        }
        let position_name = positionNameList.join(",");
        let user_name = userMap[item.user_id];
        item.user_position = user_name + " | " + position_name;
    }
    let departmentUserMap = _.groupBy(udpList, 'department_id');
    return departmentUserMap;
}


async function getAclUserData(account: string) {
    let where: {};
    if (!isNaN(Number(account))) {
        where = {
            [madAdminOrmDB.Op.or]: {
                user_id: Number(account),
                account: { [madAdminOrmDB.Op.like]: `${account}%` },
                email: { [madAdminOrmDB.Op.like]: `${account}%` },
            }
        }
    }
    else {
        where = {
            [madAdminOrmDB.Op.or]: {
                account: { [madAdminOrmDB.Op.like]: `${account}%` },
                email: { [madAdminOrmDB.Op.like]: `${account}%` },
            }
        }
    }
    let resList = await aclUserInfo.prototype.findAll({
        attributes: ['user_id', 'account', 'phone', 'email', 'user_status', 'remark'],
        where: where,
        order: [["user_id", "asc"]],
        raw: true
    });
    return resList;

}





