import { coinType } from "@madex/ex-ts-dao";

const axios = require("axios");
const _ = require("lodash");

import configSetting from "./../../config"

const webadmin_endpoint = configSetting.webadmin_endpoint
let { logger } = require('@madex/ex-js-public');

// { id: 1, symbol: 'BTC' }
export const getCoinFromCore = async (symbol) => {
    const url = `${webadmin_endpoint}/asset/by_symbol?symbol=${symbol}`
    let { data } = await axios.get(url);
    if (data.state != 0) {
        logger.error("getCoinFromCore", data.msg);
        return null
    }
    return data.result[0];
}

// {"market":"spot",
// "symbol":"BTC_USDT",
// "takerFee":"0.001",
// "minOrderSize":"0",
// "quantityIncrement":"0.01",
// "makerFee":"0.001",
// "priceIncrement":1,"
// quantityScale":2,
// "priceScale":4,
// "maxOrderSize":"0",
// "maxOrderValue":"0",
// "minOrderValue":"0"
// "productId:1"
// }
// 合约的话还有多余对应db字段 predictedFundingRate nextFundingTime ...
export const getPairFromCore = async (symbol) => {
    let market = "spot"
    if (symbol.endsWith("_SWAP")) {
        market = "lpc"
    }
    const url = `${webadmin_endpoint}/pair/by_symbol?symbol=${symbol}&market=${market}`

    let { data } = await axios.get(url);
    if (data.state != 0) {
        logger.error("getPairFromCore", data.msg);
        return null
    }
    return data.result[0]
}

// res：
// [{"name":"BTC","asset":1,"balance":"10","holds":"0"}]
export const getUserAccountFromCore = async (userId, withName = true) => {
    const url = `${webadmin_endpoint}/account/list?user_id=${userId}`
    let { data } = await axios.get(url);

    if (data.state != 0) {
        logger.error("getUserAccountFromCore", data.msg);
        return null
    }
    let assets = data.result

    let asset_id = _.map(assets, 'asset');
    if (withName) {
        let names = await coinType.prototype.findAll({
            attributes: ['general_name', 'asset_id'],
            where: {
                asset_id,
                is_main: 1
            },
            raw: true
        })
        let m = _.keyBy(names, 'asset_id');
        for (const item of assets) {
            if (m[item.asset]) {
                item.name = m[item.asset].general_name
            }
        }
    }

    return assets;
}

// market 默认spot 合约为lpc
// user_id
// status, // 必须 unsettled或者settled
// symbol, // 可选，交易对
// start_time, // 成交时间
// end_time, // 成交时间
// before, // update_id
// after, // update_id
// limit, / 数量
// res:
// [{"orderId":"4611689316962271233","clientOrderId":"","createTime":"1725261917795","product":"BTC_USDT","type":1,"side":"buy","quantity":"1","stf":"disabled","price":"9999.0000","visibleQty":"1","timeInForce":"gtc","cancelAfter":0,"postOnly":false,"status":"accepted","executedQty":"0","fillCount":0,"fills":[],"fees":[],"updateTime":"1725261917795"},{"orderId":"4611689316962271234","clientOrderId":"","createTime":"1725261919337","product":"BTC_USDT","type":1,"side":"buy","quantity":"1","stf":"disabled","price":"9998.0000","visibleQty":"1","timeInForce":"gtc","cancelAfter":0,"postOnly":false,"status":"accepted","executedQty":"0","fillCount":0,"fills":[],"fees":[],"updateTime":"1725261919337"}]
export const getUserOrdersFromCore = async (params) => {
    // let params = new URLSearchParams();
    const url = `${webadmin_endpoint}/order/list`
    let { data } = await axios.get(url, { params });
    if (data.state != 0) {
        logger.error("getUserOrdersFromCore", data.msg);
        return null
    }
    return data.result
}

// market 默认spot 合约为lpc
// user_id
// symbol, // 可选，交易对
// start_time, // 成交时间
// end_time, // 成交时间
// before, // update_id
// after, // update_id
// limit, / 数量
// order_id 可选
export const getUserFillsFromCore = async (params) => {
    // let params = new URLSearchParams();
    const url = `${webadmin_endpoint}/fill/list`
    let { data } = await axios.get(url, { params });
    if (data.state != 0) {
        logger.error("getUserFillsFromCore", data.msg);
        return null
    }
    return data.result
}


// getUserOrdersFromCore({user_id:1})

export const addCoin2Core = async (symbol, id) => {
    let res = {
        is_success: true,
        err_msg: ""
    }
    const url = `${webadmin_endpoint}/asset/add`
    let body = {
        "symbol": symbol,
        "id": id,
    }
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("addCoin2Core", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}

export const addPairToCore = async (body) => {
    let res = {
        is_success: true,
        err_msg: ""
    }
    const url = `${webadmin_endpoint}/pair/add_product`
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("addPairToCore", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}

// 设置手续费 至少指定user_id 和 symbol 二者其一
export const changeTradingFee = async (symbol,user_id,maker_fee,taker_fee,market="spot") => {

    let body = {
        maker_fee,taker_fee,
    }
    body["market"] = market;

    if (symbol) {
        body["symbol"] = symbol;
    }
    if (user_id>0) {
        body["user_id"] = user_id;
    }
    let res = {
        is_success: true,
        err_msg: ""
    };
    const url = `${webadmin_endpoint}/user/change_trading_fee`
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("changeTradingFee", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}

// 成为代理 user_id==agent_id 代理商 user_id!=agent_id 代理用户
export const setAgent = async (user_id,agent_id) => {
    let body = {
        user_id,agent_id
    }
    let res = {
        is_success: true,
        err_msg: ""
    };
    const url = `${webadmin_endpoint}/market_maker/add_agent_setting`
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("setAgent", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}

// 移除代理商或者代理用户
export const removeAgent = async (user_id) => {
    let body = {
        user_id
    }
    let res = {
        is_success: true,
        err_msg: ""
    };
    const url = `${webadmin_endpoint}/market_maker/remove_agent_id`
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("removeAgent", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}

// 添加代理对冲配置
// exp:1110,spot,BTC_USDT,0.5
// exp:1110,lpc,BTC_USDT_USDT,0.6
export const addAgentSetting = async (agent_id,market,symbol,ratio) => {
    let body = {
        agent_id, market, symbol, ratio
    };
    let res = {
        is_success: true,
        err_msg: ""
    };
    const url = `${webadmin_endpoint}/market_maker/add_agent_setting`
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("addAgentSetting", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}

// {"agentId":0,"takerFee":"1","makerFee":"1","id":12321321}
// agentId=0 表示不是代理商或者代理用户 user_id==agent_id 代理商 user_id!=agent_id 代理用户
// 所以这个接口的返回里的agentId 可以判断是不是代理商
export const getUserDetail = async (user_id) => {
    const url = `${webadmin_endpoint}/user/detail?user_id=${user_id}`
    let { data } = await axios.get(url);
    if (data.state != 0) {
        logger.error("getUserDetail", data.msg);
        return null
    }
    return data.result;
}

// 获取代理商配置配置
// [{"product":"BTC_USDT","qty":"0","id":1,"ratio":"0.5"}]
// 交易对 对冲数量 交易对id 对冲比例
export const getAgentSettings = async (agent_id) => {
    const url = `${webadmin_endpoint}/market_maker/agent_settings?agent_id=${agent_id}`
    let { data } = await axios.get(url);
    if (data.state != 0) {
        logger.error("getUserDetail", data.msg);
        return null
    }
    return data.result;
}

// 获取币种抵扣
export const getCollaterals = async () => {
    const url = `${webadmin_endpoint}/asset/collaterals`
    let { data } = await axios.get(url);
    if (data.state != 0) {
        logger.error("getUserDetail", data.msg);
        return null
    }
    return data.result;
}

// 设置抵扣
export const setCollateral = async (symbol,discount,order,collateral=true) => {
    let index  = symbol+"_USDT_SWAP"
    let body = {
        index,symbol,discount,order,collateral
    };
    let res = {
        is_success: true,
        err_msg: ""
    };
    const url = `${webadmin_endpoint}/asset/set_collateral`
    let { data } = await axios.post(url, body);
    if (data.state != 0) {
        logger.error("setCollateral", data.msg);
        res.is_success = false;
        res.err_msg = data.msg;
    }
    return res;
}


