// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { madAdminOrmDB, aclAuth, aclUserRole, aclRoleAuth, aclRole } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { getOneAclUserByUid } from "../../../utils/aclUserUtils";

let { logger } = require('@madex/ex-js-public');


export interface AclAuthVO {
    id?: number;

    pid?: number;//当前登陆的用户 的 user_id

    name?: string | any;

    url?: string | any;

    type?: number | any;

    idx_number?: number | any;

    remark?: string | any;

    createdAt?: Date | any;

    updatedAt?: Date | any;

    authIds?: any
}

export interface AclAuthPageVO extends AclAuthVO {
    page?: number,

    size?: number
}


export interface AclRoleVO {
    id?: number;

    user_id?: number;

    name?: string | any;

    type?: number | any;

    remark?: string | any;

    creator?: number | any;

    createdAt?: Date | any;

    updatedAt?: Date | any;

    roleIds?: any
}

export interface AclRolePageVO extends AclRoleVO {
    page?: number,

    size?: number
}

export const authList = async (aclAuthPageVO: AclAuthPageVO) => {

    if (aclAuthPageVO.name) {
        aclAuthPageVO.name = { [madAdminOrmDB.Op.like]: `${aclAuthPageVO.name}%` };
    }
    if (aclAuthPageVO.url) {
        aclAuthPageVO.url = { [madAdminOrmDB.Op.like]: `${aclAuthPageVO.url}%` };
    }
    let page = Number(aclAuthPageVO.page);
    let size = Number(aclAuthPageVO.size);
    let where = Object.assign(aclAuthPageVO);
    delete aclAuthPageVO.page;
    delete aclAuthPageVO.size;
    let resList = await aclAuth.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
};

export const getAuthTree = async () => {
    let authM = {};
    let authList: any = [] // 需要返回给前端tree 数组
    let tmp = await getAllAuthListSorted();
    for (let auth of tmp) {
        authM[auth.id] = auth;
        let pid = auth.pid;
        if (Number(pid) === -1) {
            authList.push(auth)
        }
        else {
            let pAuth = authM[pid];
            if (!pAuth) {
                logger.warn("父菜单被删除或者不存在");
                continue
            }
            if (!pAuth['subAuthList']) {
                pAuth['subAuthList'] = [];
            }
            pAuth['subAuthList'].push(auth);
        }
    }
    return authList;
};

export const getAuthByUser = async (userId: number) => {
    let roleIDArr = await getRoleIDArr(userId);
    if (!roleIDArr.length) {
        return [];
    }

    let authIDArr = await getAuthIDArr(roleIDArr);
    if (!authIDArr.length) {
        return [];
    }


    if (await _hashSuper(roleIDArr)) {
        return getAllAuthListSorted()
    }

    return await getAuth(authIDArr);
};

export const getAuthByRole = async (role_id: number | undefined) => {
    let roleIDArr = [role_id];
    if (!roleIDArr.length) {
        return [];
    }
    let authIDArr = await getAuthIDArr(roleIDArr);
    if (!authIDArr.length) {
        return [];
    }
    return await getAuth(authIDArr);
};


export const saveAuth = async (aclAuthVO: AclAuthVO) => {
    await _checkPid(aclAuthVO.pid);
    if (aclAuthVO.id) {
        await _checkUrl(aclAuthVO.url, aclAuthVO.id);
        aclAuth.prototype.update({
            pid: aclAuthVO.pid,
            name: aclAuthVO.name,
            url: aclAuthVO.url,
            type: aclAuthVO.type,
            idx_number: aclAuthVO.idx_number,
            remark: aclAuthVO.remark
        }, {
            where: { id: aclAuthVO.id }
        })
    }
    else {
        await _checkUrl(aclAuthVO.url);
        aclAuth.prototype.create(aclAuthVO);
    }
    return "ok";
};

export const delAuth = async (id: number | undefined) => {

    let exist = await aclAuth.prototype.find({
        where: { id },
        raw: true
    })
    if (!exist) {
        throw ErrorCode.AUTH_NOT_EXIST;
    }
    let sub = await aclAuth.prototype.find({
        where: {
            pid: id
        },
        raw: true
    })

    if (sub) {
        throw ErrorCode.EXIST_SUB_AUTH;
    }

    let tx;
    try {
        tx = await madAdminOrmDB.transaction()
        await aclAuth.prototype.destroy({
            where: {
                id: Number(id)
            },
            transaction: tx
        })
        await aclRoleAuth.prototype.destroy({
            where: {
                auth_id: Number(id)
            },
            transaction: tx
        })
        await tx.commit();
        tx = null;
    }
    catch (e) {
        if (tx) {
            await tx.rollback()
        }
        throw e
    }
    return "ok"
};
export const roleList = async (aclRolePageVO: AclRolePageVO) => {

    if (aclRolePageVO.name) {
        aclRolePageVO.name = { [madAdminOrmDB.Op.like]: `${aclRolePageVO.name}%` };
    }
    if (aclRolePageVO.remark) {
        aclRolePageVO.remark = { [madAdminOrmDB.Op.like]: `${aclRolePageVO.remark}%` };
    }
    let page = Number(aclRolePageVO.page);
    let size = Number(aclRolePageVO.size);
    let where = Object.assign(aclRolePageVO);
    delete aclRolePageVO.page;
    delete aclRolePageVO.size;
    let resList = await aclRole.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
};

export const getAllRole = async () => {
    return aclRole.prototype.findAll({
        where: {},
        raw: true
    })
};

export const getRoleByUser = async (userId: number | any) => {
    let roleIDArr = await getRoleIDArr(userId);
    if (!roleIDArr.length) {
        return [];
    }
    return await getRole(roleIDArr);
};

export const saveRole = async (aclRoleVO: AclRoleVO) => {

    if (aclRoleVO.id) {
        await aclRole.prototype.update({
            name: aclRoleVO.name,
            remark: aclRoleVO.remark
        }, {
            where: {
                id: aclRoleVO.id,
                creator: aclRoleVO.creator
            }
        })
    }
    else {
        // 创建
        await aclRole.prototype.create({
            name: aclRoleVO.name,
            remark: aclRoleVO.remark,
            creator: aclRoleVO.creator,
            type: aclRoleVO.type ? aclRoleVO.type : 0,
        })
    }
    return "ok";
};


export const delRole = async (id: number | any, currentUserId: number) => {
    //TODO: 此逻辑要不要加
    /*if ([1, 2, 3, 4].includes(Number(id))) {
        throw 'err:固有角色,无法删除'
    }*/
    let exist = await aclRole.prototype.find({
        where: { id },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.ROLE_NOT_EXIST
    }
    let isSuperUser = await _isSuper(currentUserId);
    if (isSuperUser || Number(exist.creator) === Number(currentUserId)) {
        throw ErrorCode.NO_PERMISSION
    }
    let roleUser = await aclUserRole.prototype.find({
        where: {
            role_id: id
        },
        raw: true
    })
    if (roleUser) {
        throw ErrorCode.ROLE_USE;
    }
    let tx

    try {
        tx = await madAdminOrmDB.transaction()
        await aclRole.prototype.destroy({
            where: {
                id
            },
            transaction: tx
        });
        await aclRoleAuth.prototype.destroy({
            where: {
                role_id: id
            },
            transaction: tx
        });
        await tx.commit()
        tx = null;
    }
    catch (e) {
        if (tx) {
            await tx.rollback()
        }
        throw e
    }

    return "ok"

};

export const changeRoleAuth = async (id: number | any, authIds: any) => {
    let sp = authIds.split(",");
    let role = await aclRole.prototype.find({
        where: {
            id: id,
        },
        raw: true
    });
    if (!role) {
        throw ErrorCode.ROLE_NOT_EXIST
    }
    let authIDArr: any[] = [];
    for (let i of sp) {
        if (!isNaN(i)) {
            authIDArr.push(i)
        }
    }

    if (!authIDArr.length) {
        // 没有指定权限的，直接删除角色对应的所有权限。
        await aclRoleAuth.prototype.destroy({
            where: {
                role_id: id
            }
        })
    }
    else {
        // 检查权限是不是都存在
        let authArr = await getAuth(authIDArr);
        if (authArr.length !== authIDArr.length) {
            throw ErrorCode.AUTH_EXIST_ILLEGAL;
        }
        // 找出重叠的部分
        let oldData = await aclRoleAuth.prototype.findAll({
            where: {
                role_id: id,
            },
            raw: true
        });
        let oldIDArr = oldData.map(item => item.auth_id);

        // 找出删除的数据的authID
        let deleteIDArr: any[] = [];
        for (let i of oldIDArr) {
            if (!authIDArr.includes(i)) {
                deleteIDArr.push(i)
            }
        }
        // 找出需要新增的数据
        let now = new Date()
        let addData: any[] = [];
        for (let i of authIDArr) {
            if (!oldIDArr.includes(i)) {
                let item = {
                    role_id: id,
                    auth_id: i,
                    createdAt: now,
                    updatedAt: now,
                }
                addData.push(item)
            }
        }
        let tx;
        try {
            tx = await madAdminOrmDB.transaction()
            if (deleteIDArr.length) {
                await aclRoleAuth.prototype.destroy({
                    where: {
                        role_id: id,
                        auth_id: { [madAdminOrmDB.Op.in]: deleteIDArr }
                    },
                    transaction: tx
                })

            }

            if (addData.length) {
                await aclRoleAuth.prototype.bulkCreate(addData, {
                    transaction: tx
                })
            }

            await tx.commit();
            tx = null;
        }
        catch (e) {
            if (tx) {
                await tx.rollback();
            }
            throw e
        }
    }
    return "ok"

};


export const changeUserRole = async (userId: number | any, roleIds: any, tx?: any) => {
    let sp = roleIds.split(",");
    await getOneAclUserByUid(userId);
    let roleIDArr: any[] = [];
    for (let i of sp) {
        if (!isNaN(i)) {
            /*if (i==1) {
                throw 'err:无法操作admin';
            }*/
            roleIDArr.push(i);
        }
    }

    if (!roleIDArr.length) {
        // 没有指定权限的，直接删除角色对应的所有权限。
        await aclUserRole.prototype.destroy({
            where: {
                user_id: userId
            },
            transaction: tx,
        })
    }
    else {
        // 检查权限是不是都存在
        let roleArr = await getRole(roleIDArr, tx);
        if (roleArr.length !== roleIDArr.length) {
            throw ErrorCode.ROLE_EXIST_ILLEGAL
        }
        // 找出重叠的部分
        let oldData = await aclUserRole.prototype.findAll({
            where: {
                user_id: userId,
            },
            transaction: tx,
            raw: true
        });
        let oldIDArr = oldData.map(item => item.role_id);

        // 找出删除的数据的authID
        let deleteIDArr: any[] = [];
        for (let i of oldIDArr) {
            if (!roleIDArr.includes(i)) {
                deleteIDArr.push(i)
            }
        }
        // 找出需要新增的数据
        let now = new Date()
        let addData: any[] = [];
        for (let i of roleIDArr) {
            if (!oldIDArr.includes(i)) {
                let item = {
                    role_id: i,
                    user_id: userId,
                    createdAt: now,
                    updatedAt: now,
                }
                addData.push(item)
            }
        }
        try {
            if (deleteIDArr.length) {
                await aclUserRole.prototype.destroy({
                    where: {
                        user_id: userId,
                        role_id: { [madAdminOrmDB.Op.in]: deleteIDArr }
                    },
                    transaction: tx
                })

            }

            if (addData.length) {
                await aclUserRole.prototype.bulkCreate(addData, {
                    transaction: tx
                })
            }

        }
        catch (e) {
            throw e
        }
    }
    return "ok"

};

/**
 * 获取用户权限
 * @param userId
 */
export async function getUserAcl(userId: number) {

    let roleIDArr = await getRoleIDArr(userId);
    let roleSet = await getRole(roleIDArr);
    let authArr;
    if (await _hashSuper(roleIDArr)) {
        authArr = await getAllAuthListSorted()
    }
    else {
        let authIDArr = await getAuthIDArr(roleIDArr);
        authArr = await getAuth(authIDArr);
    }
    let authSet = authArr.map(item => item.url);

    authSet = Array.from(new Set(authSet))
    return { roleSet, authSet }
}

async function getAllAuthListSorted() {
    return await aclAuth.prototype.findAll({
        where: {},
        order: [['pid'], ['idx_number']],
        raw: true
    })
}

async function getRoleIDArr(user_id: number) {
    let res = await aclUserRole.prototype.findAll({
        where: {
            user_id
        },
        raw: true
    })
    return res.map(item => item.role_id);
}

async function getAuthIDArr(roleIDArr: any[]) {

    if (!roleIDArr.length) {
        return []
    }
    let where = {
        role_id: { [madAdminOrmDB.Op.in]: roleIDArr }
    }
    let res = await aclRoleAuth.prototype.findAll({
        where,
        raw: true
    });
    return res.map(item => item.auth_id)
}

async function _hashSuper(roleIDArr: string | any[]) {

    if (!roleIDArr.length) {
        return false;
    }
    let res = await aclRole.prototype.find({
        where: {
            id: { [madAdminOrmDB.Op.in]: roleIDArr },
            type: 1
        },
        raw: true
    });
    return !!res;
}

async function getAuth(IDArr: any[]) {

    if (!IDArr.length) {
        return [];
    }
    return await aclAuth.prototype.findAll({
        where: {
            id: { [madAdminOrmDB.Op.in]: IDArr }
        },
        order: [['pid'], ['idx_number']],
        raw: true
    });
}

async function _checkPid(pid: number | undefined) {
    if (pid != -1) {
        let res = await aclAuth.prototype.find({
            where: {
                id: pid
            },
            raw: true
        })
        if (!res) {
            throw ErrorCode.PARENT_MENU_NOT_EXIST;
        }
    }
}

async function _checkUrl(url: string, id?: number) {
    let where = Object.create(null);
    where.url = url;

    if (id) {
        where.id = { [madAdminOrmDB.Op.ne]: id }
    }
    let res = await aclAuth.prototype.find({
        where,
        raw: true
    });
    if (res) {
        throw ErrorCode.AUTH_EXIST;
    }
}

async function getRole(IDArr: number[] | any[], tx?: any) {

    if (!IDArr.length) {
        return [];
    }
    return await aclRole.prototype.findAll({
        where: {
            id: { [madAdminOrmDB.Op.in]: IDArr }
        },
        transaction: tx,
        raw: true
    });
}

async function _isSuper(userId: number) {

    let roleIDArr = await getRoleIDArr(userId);
    if (roleIDArr.length) {
        return await _hashSuper(roleIDArr)
    }
    return false;
}


