// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { i18nInfo, i18nInfoLog, ormDB } from "@madex/ex-ts-dao";
import * as i18nLogService from "../service/i18nlog.service";
import { ErrorCode } from "../../../constant/errorCode";

let { logger } = require('@madex/ex-js-public');

export interface I18nInfoVO {
    id?: number;

    platform: number;

    module: number;

    code?: string | any;

    sub_code?: string | any;

    lang?: string;

    desc?: string | any;

    content?: string | any;

    createdAt?: Date | any;

    updatedAt?: Date | any;
}

export interface I18nInfoPageVO extends I18nInfoVO {
    page?: number,

    size?: number
}

export const list = async (infoPageVO: I18nInfoPageVO) => {

    if (infoPageVO.code) {
        infoPageVO.code = { [ormDB.Op.like]: `${infoPageVO.code}%` };
    }

    if (infoPageVO.sub_code) {
        infoPageVO.sub_code = { [ormDB.Op.like]: `${infoPageVO.sub_code}%` };
    }
    if (infoPageVO.desc) {
        infoPageVO.desc = { [ormDB.Op.like]: `${infoPageVO.desc}%` };
    }
    if (infoPageVO.updatedAt) {
        infoPageVO.updatedAt = { [ormDB.Op.gte]: new Date(infoPageVO.updatedAt).getTime() };
    }
    let page = Number(infoPageVO.page);
    let size = Number(infoPageVO.size);
    let where = Object.assign(infoPageVO);
    delete infoPageVO.page;
    delete infoPageVO.size;
    let resList = await i18nInfo.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
};
export const add = async (infoVO: I18nInfoVO) => {
    if (!infoVO.createdAt) {
        infoVO.createdAt = new Date();
    }
    if (!infoVO.updatedAt) {
        infoVO.updatedAt = new Date();
    }
    let dbInfo = await i18nInfo.prototype.findOne({
        where: {
            platform: infoVO.platform,
            module: infoVO.module,
            code: infoVO.code,
            lang: infoVO.lang,
        },
        raw: true
    });
    if (dbInfo){
        throw ErrorCode.DATA_EXIST;
    }
    await i18nInfo.prototype.create(infoVO);
    return 'ok';
};

export const update = async (infoVO: I18nInfoVO) => {
    let transaction: any;
    try {
        let dbOneI18nInfo = await i18nInfo.prototype.findOne({
            raw: true,
            where: {
                id: infoVO.id
            }
        });
        if (!dbOneI18nInfo) {
            throw ErrorCode.DATA_NOT_EXIST;
        }
        delete infoVO.id;
        let i18nInfoLogOne = await buildOneI18nInfoLog(dbOneI18nInfo, infoVO);

        transaction = await ormDB.transaction();


        await i18nInfo.prototype.update(infoVO, {
            where: {
                id: dbOneI18nInfo.id
            },
            transaction: transaction,
        });
        //日志
        await i18nLogService.add(i18nInfoLogOne, transaction);
        await transaction.commit();
        return 'ok';
    }
    catch (e) {
        logger.error(`i18nService.update.error:${e}`);
        if (transaction) {
            await transaction.rollback();
        }
        throw e;
    }
};

export const del = async (infoVO: I18nInfoVO) => {
    let transaction: any;
    try {
        let dbOneI18nInfo = await i18nInfo.prototype.findOne({
            raw: true,
            where: {
                id: infoVO.id
            }
        });
        if (!dbOneI18nInfo) {
            throw ErrorCode.DATA_NOT_EXIST;
        }
        delete infoVO.id;
        let i18nInfoLogOne = await buildOneI18nInfoLog(dbOneI18nInfo, infoVO);


        transaction = await ormDB.transaction();


        await i18nInfo.prototype.destroy({
            where: {
                id: dbOneI18nInfo.id
            },
            transaction: transaction,
        });
        //日志
        await i18nLogService.add(i18nInfoLogOne, transaction);
        await transaction.commit();
        return 'ok';
    }
    catch (e) {
        logger.error(`i18nService.del.error:${e}`);
        if (transaction) {
            await transaction.rollback();
        }
        throw e;
    }
};


async function buildOneI18nInfoLog(dbOneI18nInfo: any, infoVO: I18nInfoVO) {

    let i18nInfoLogOne = {
        //修改前
        info_id: dbOneI18nInfo.id,
        platform: dbOneI18nInfo.platform,
        module: dbOneI18nInfo.module,
        code: dbOneI18nInfo.code,
        sub_code: dbOneI18nInfo.sub_code,
        lang: dbOneI18nInfo.lang,
        desc: dbOneI18nInfo.desc,
        content: dbOneI18nInfo.content,

        //修改后
        platform_after: infoVO.platform ? infoVO.platform : dbOneI18nInfo.platform,
        module_after: infoVO.module ? infoVO.module : dbOneI18nInfo.module,
        code_after: infoVO.code ? infoVO.code : dbOneI18nInfo.code,
        sub_code_after: infoVO.sub_code ? infoVO.sub_code : dbOneI18nInfo.sub_code,
        lang_after: infoVO.lang ? infoVO.lang : dbOneI18nInfo.lang,
        desc_after: infoVO.desc ? infoVO.desc : dbOneI18nInfo.desc,
        content_after: infoVO.content ? infoVO.content : dbOneI18nInfo.content
    }


    return i18nInfoLogOne;
}
