import { ormDB, usefulLink, commonUserFeeSetting, feeRateContractLog, feeRateBaseCoinContractLog, feeRateSpotLog, vipLevelFeeSetting } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { COMMENT_USER_FEE_SUBMIT, FEE_STATUS, FEE_TYPE } from "../../../constant/marketMakerConst";
import { SYMBOL_ALL } from "../../../constant/symbolConst";

import * as feeRateSpotLogService from "./feeRateSpotLog.service";
import * as feeRateContractLogService from "./feeRateContractLog.service";
import * as feeRateBaseCoinContractLogService from "./feeRateBaseCoinContractLog.service";
import { FEE_MODEL_SPOT_DEFAULT, FEE_RATE_LOG_STATUS } from "../../../constant/feeRateLogConst";


let _ = require('lodash');
let { logger } = require('@madex/ex-js-public');


export interface CommonUserFeeVO {
    id?: number;

    user_id?: number | any;

    vip_level?: number;

    type?: number;

    pair?: string;

    maker_fee?: number,

    taker_fee?: number,

    begin_time?: Date | any,

    expire_time?: Date | any,

    maker_fee_later?: number,

    taker_fee_later?: number,

    status?: number,

    remark?: string;

    fee_log_ids?: string;

    create_time?: Date | any,

    update_time?: Date | any,

    level_id?: number;

    spot_maker_fee?: number;

    spot_taker_fee?: number;

    contract_maker_fee?: number;

    contract_taker_fee?: number;
}


export interface CommonUserFeePageVO extends CommonUserFeeVO {

    page?: number,

    size?: number,

    export?: number,//是不全部数据  1 全部倒出  0 否

}


export async function list(pageVO: CommonUserFeePageVO) {

    if (pageVO.export) {
        let resList = await commonUserFeeSetting.prototype.findAll({
            where: {
                status: { [ormDB.Op.ne]: 4 }
            },
            order: [['id', 'desc']],
            raw: true
        });
        return resList;
    }
    let where = Object.create(null);
    if (pageVO.user_id) {
        where.user_id = pageVO.user_id;
    }
    where.status = { [ormDB.Op.ne]: 4 };
    if (pageVO.type) {
        where.type = pageVO.type;
    }


    let resList = await commonUserFeeSetting.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}


export async function add(commonUserFeeVO: CommonUserFeeVO, currentUserId: any, ip: string | undefined) {
    let insertList: any = [];

    let { spotFeeSetting, contractFeeSetting, baseCoinFeeSetting } = dealData(commonUserFeeVO);

    if (spotFeeSetting) {
        let [spot, all] = await Promise.all([getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_SPOT),
            getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_ALL)]);
        if (spot || all) {
            throw ErrorCode.ONLY_ONE_FEE
        }
        insertList.push(spotFeeSetting);
    }

    if (contractFeeSetting) {
        let [contract, contractAll, all] = await Promise.all([getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_CONTRACT),
            getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_CONTRACT_ALL),
            getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_ALL)]);
        if (contract || contractAll || all) {
            throw ErrorCode.ONLY_ONE_FEE
        }
        insertList.push(contractFeeSetting);
    }

    if (baseCoinFeeSetting) {
        let [contract, contractAll, all] = await Promise.all([getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT),
            getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_CONTRACT_ALL),
            getFeeSetting(commonUserFeeVO.user_id, FEE_TYPE.FEE_TYPE_ALL)]);
        if (contract || contractAll || all) {
            throw ErrorCode.ONLY_ONE_FEE
        }
        insertList.push(baseCoinFeeSetting);
    }

    await commonUserFeeSetting.prototype.bulkCreate(insertList);
    //管理后台操作日志
    addOptLog(currentUserId, Number(commonUserFeeVO.user_id), '新增用户手续费', ip, JSON.stringify(commonUserFeeVO), '费率管理');

}


export async function update(commonUserFeeVO: CommonUserFeeVO, currentUserId: any, ip: string | undefined) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            id: commonUserFeeVO.id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    if (dbInfo.status != FEE_STATUS.STATUS_FEE_SETTING_NEW) {
        throw ErrorCode.ONLY_UN_SUBMIT_UPDATE
    }
    if (dbInfo.user_id != commonUserFeeVO.user_id || dbInfo.type != commonUserFeeVO.type) {
        throw ErrorCode.UID_TYPE_NOT_UPDATE
    }
    await commonUserFeeSetting.prototype.update({
        pair: commonUserFeeVO.pair,
        maker_fee: Number(commonUserFeeVO.maker_fee),
        taker_fee: Number(commonUserFeeVO.taker_fee),
        begin_time: commonUserFeeVO.begin_time,
        expire_time: commonUserFeeVO.expire_time,
        remark: commonUserFeeVO.remark,
        fee_log_ids: "",
        update_time: new Date(),

    }, {
        where: {
            id: Number(commonUserFeeVO.id)
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, Number(commonUserFeeVO.user_id), '修改用户手续费', ip, JSON.stringify(commonUserFeeVO), '费率管理');

}


export async function del(id: number, currentUserId: any, ip: string | undefined) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    let status = dbInfo.status;
    let type = dbInfo.type;
    if (status != FEE_STATUS.STATUS_FEE_SETTING_NEW && status != FEE_STATUS.STATUS_FEE_SETTING_SUBMIT) {
        throw ErrorCode.DEL_UN_SUBMIT_ACTIVE
    }
    if (status == FEE_STATUS.STATUS_FEE_SETTING_NEW) {
        await commonUserFeeSetting.prototype.destroy({
            where: {
                id: id
            }
        });
        //管理后台操作日志
        addOptLog(currentUserId, Number(dbInfo.user_id), '删除用户手续费', ip, JSON.stringify(dbInfo), '费率管理');

        return 'success'
    }


    let feeLogIdsStr = dbInfo.fee_log_ids;
    let feeLogIds = feeLogIdsStr.split(",");

    let isActive = false;
    // 根据费率设置纪录的状态，确定是否有记录被激活。
    if (type == FEE_TYPE.FEE_TYPE_SPOT) {
        let dbList = await feeRateSpotLogService.getByIdList(feeLogIds);
        let filterList = _.filter(dbList, i => i.is_check != FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK);
        if (filterList && filterList.length) {
            isActive = true
        }
    }
    else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {
        let dbList = await feeRateBaseCoinContractLogService.getByIdList(feeLogIds);
        let filterList = _.filter(dbList, i => i.is_check != FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK);
        if (filterList && filterList.length) {
            isActive = true
        }
    }
    else {
        let dbList = await feeRateContractLogService.getByIdList(feeLogIds);
        let filterList = _.filter(dbList, i => i.is_check != FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK);
        if (filterList && filterList.length) {
            isActive = true
        }
    }

    if (isActive) {
        await commonUserFeeSetting.prototype.update({
            status: FEE_STATUS.STATUS_FEE_SETTING_EFFECTED,
            update_time: new Date()
        }, {
            where: {
                id: id,
                status: status
            }
        });
        throw ErrorCode.FEE_USED_NOT_DEL
    }


    let tx;
    try {
        tx = await ormDB.transaction();
        if (type == FEE_TYPE.FEE_TYPE_SPOT) {
            await feeRateSpotLogService.uncheck2Deleted(feeLogIds, tx);

        }
        else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {
            await feeRateBaseCoinContractLogService.uncheck2Deleted(feeLogIds, tx);

        }
        else {
            await feeRateContractLogService.uncheck2Deleted(feeLogIds, tx);
        }

        await commonUserFeeSetting.prototype.update({
            status: FEE_STATUS.STATUS_FEE_SETTING_DELETED,
            update_time: new Date()
        }, {
            where: {
                id: id,
                status: status
            },
            transaction: tx
        });
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('commonUserFeeSetting.service.del.error:' + e);
        throw e;
    }

    //管理后台操作日志
    addOptLog(currentUserId, Number(dbInfo.user_id), '删除用户手续费', ip, JSON.stringify(dbInfo), '费率管理');

    return 'success';

}


export async function submit(id: number, currentUserId: any, ip: string | undefined) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    let status = dbInfo.status;
    let type = dbInfo.type;
    if (status != FEE_STATUS.STATUS_FEE_SETTING_NEW) {
        throw ErrorCode.ONLY_UN_SUBMIT_CAN_SUBMIT
    }
    let pair = String(dbInfo.pair);
    let userId = dbInfo.user_id;
    let feeLogId = "";
    let now = new Date();
    let tx;
    try {
        tx = await ormDB.transaction();
        if (type == FEE_TYPE.FEE_TYPE_SPOT) {
            let insertInfo = buildSubmitItem(pair, dbInfo, now);
            let item = await feeRateSpotLog.prototype.create(insertInfo, {
                transaction: tx
            });
            feeLogId = item.id.toString();
        }
        else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {//币本位
            if (pair.toLowerCase() == "all") {
                //TODO:币本位币对查询 enable = 1 的 pairList = 数据库查询结果
                let pairList = [];
                let insertList: any = [];
                let ids: any = [];
                for (let onePair of pairList) {
                    let oneInsert = buildSubmitItem(onePair, dbInfo, now);
                    insertList.push(oneInsert);
                }
                let bulkList = await feeRateBaseCoinContractLog.prototype.bulkCreate(insertList, {
                    transaction: tx
                });
                for (let one of bulkList) {
                    ids.push(one.id);
                }
                feeLogId = ids.toString();
            }
            else {
                let one = buildSubmitItem(pair, dbInfo, now);
                let insertOne = await feeRateBaseCoinContractLog.prototype.create(one, {
                    transaction: tx
                });
                feeLogId = insertOne.id.toString()
            }
        }
        else {//U本位
            if (pair.toLowerCase() == "all") {
                //TODO:U本位币对查询 enable = 1 的 pairList = 数据库查询结果
                let pairList = [];
                let insertList: any = [];
                let ids: any = [];
                for (let onePair of pairList) {
                    let oneInsert = buildSubmitItem(onePair, dbInfo, now);
                    insertList.push(oneInsert);
                }
                let bulkList = await feeRateContractLog.prototype.bulkCreate(insertList, {
                    transaction: tx
                });
                for (let one of bulkList) {
                    ids.push(one.id);
                }
                feeLogId = ids.toString();
            }
            else {
                let one = buildSubmitItem(pair, dbInfo, now);
                let insertOne = await feeRateContractLog.prototype.create(one, {
                    transaction: tx
                });
                feeLogId = insertOne.id.toString()
            }
        }

        await commonUserFeeSetting.prototype.update({
            status: FEE_STATUS.STATUS_FEE_SETTING_SUBMIT,
            fee_log_ids: feeLogId,
            update_time: new Date()
        }, {
            where: {
                id: id,
                status: FEE_STATUS.STATUS_FEE_SETTING_NEW
            },
            transaction: tx
        });

        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('commonUserFeeSetting.service.submit.error:' + e);
        throw e;
    }
    //管理后台操作日志
    addOptLog(currentUserId, Number(dbInfo.user_id), '提交用户手续费', ip, JSON.stringify(dbInfo), '费率管理');

    return 'success'


}

export async function vipLevelList() {
    let resList = await vipLevelFeeSetting.prototype.findAll({
        order: [['id', 'desc']],
        raw: true
    });
    return resList;
}


function dealData(commonUserFeeVO: CommonUserFeeVO) {

    let spotFeeSetting, contractFeeSetting, baseCoinFeeSetting;
    let type = commonUserFeeVO.type;

    if (type == FEE_TYPE.FEE_TYPE_SPOT) {
        spotFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.spot_maker_fee, commonUserFeeVO.spot_taker_fee, FEE_TYPE.FEE_TYPE_SPOT)
    }
    else if (type == FEE_TYPE.FEE_TYPE_CONTRACT) {
        contractFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.contract_maker_fee, commonUserFeeVO.contract_taker_fee, FEE_TYPE.FEE_TYPE_CONTRACT)
    }
    else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {
        baseCoinFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.contract_maker_fee, commonUserFeeVO.contract_taker_fee, FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT)

    }
    else if (type == FEE_TYPE.FEE_TYPE_CONTRACT_ALL) {
        contractFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.contract_maker_fee, commonUserFeeVO.contract_taker_fee, FEE_TYPE.FEE_TYPE_CONTRACT)
        baseCoinFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.contract_maker_fee, commonUserFeeVO.contract_taker_fee, FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT)
    }
    else {
        spotFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.spot_maker_fee, commonUserFeeVO.spot_taker_fee, FEE_TYPE.FEE_TYPE_SPOT)
        contractFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.contract_maker_fee, commonUserFeeVO.contract_taker_fee, FEE_TYPE.FEE_TYPE_CONTRACT)
        baseCoinFeeSetting = buildOne(commonUserFeeVO, commonUserFeeVO.contract_maker_fee, commonUserFeeVO.contract_taker_fee, FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT)

    }

    return {
        spotFeeSetting,
        contractFeeSetting,
        baseCoinFeeSetting
    }
}

function buildOne(commonUserFeeVO: CommonUserFeeVO, maker_fee: number | any, taker_fee: number | any, type: number | any) {
    let one = {
        user_id: commonUserFeeVO.user_id,
        type: Number(type),
        pair: SYMBOL_ALL,
        maker_fee: Number(maker_fee),
        taker_fee: Number(taker_fee),
        begin_time: commonUserFeeVO.begin_time,
        expire_time: commonUserFeeVO.expire_time,
        remark: commonUserFeeVO.remark,
        status: FEE_STATUS.STATUS_FEE_SETTING_NEW,
        fee_log_ids: "",
        create_time: new Date(),
        update_time: new Date(),
    }
    return one;
}

async function getFeeSetting(user_id: number | any, type: number | any) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            user_id: Number(user_id),
            type: Number(type),
            status: FEE_STATUS.STATUS_FEE_SETTING_NEW
        },
        raw: true
    });
    return dbInfo;


}

function buildSubmitItem(pair: string, dbInfo: any, now: Date) {
    let item = {
        user_id: dbInfo.user_id,
        pair: pair,
        fee_model: FEE_MODEL_SPOT_DEFAULT,
        maker_fee: dbInfo.maker_fee,
        taker_fee: dbInfo.taker_fee,
        beginAt: dbInfo.begin_time,
        expireAt: dbInfo.expire_time,
        is_check: FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK,
        comment: COMMENT_USER_FEE_SUBMIT,
        createdAt: now,
        updatedAt: now
    }
    return item;
}

