// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { madAdminOrmDB, aclUserInfo } from "@madex/ex-ts-dao";
import { AclUserInfoConst } from "../../../constant/aclUserConstant";
import { CryptUtils } from "../../../utils/crypt-utils";
import { ErrorCode } from "../../../constant/errorCode";
import { getOneAclUserByAccount, getOneAclUserByUid } from "../../../utils/aclUserUtils";
import * as userOptLogService from "./userOptLog.service";

let { logger } = require('@madex/ex-js-public');


export interface AclUserInfoVO {
    user_id?: number;

    account?: string | any;

    phone?: string | any;

    email?: string | any;

    remark?: string | any;

    user_type?: number;

    pwd?: string | any;

    pwd_salt?: string;

    pwd_status?: number;

    user_status?: number;

    createdAt?: Date | any;

    updatedAt?: Date | any;

    totpCode?: any,

    originPwd?: any,

    newPwd?: any,


}

export interface AclUserInfoPageVO extends AclUserInfoVO {
    page?: number,

    size?: number
}

export const list = async (aclUserInfoPageVO: AclUserInfoPageVO) => {

    if (aclUserInfoPageVO.account) {
        aclUserInfoPageVO.account = { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.account}%` };
    }
    if (aclUserInfoPageVO.phone) {
        aclUserInfoPageVO.phone = { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.phone}%` };
    }
    if (aclUserInfoPageVO.email) {
        aclUserInfoPageVO.email = { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.email}%` };
    }
    if (aclUserInfoPageVO.remark) {
        aclUserInfoPageVO.remark = { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.remark}%` };
    }

    if (aclUserInfoPageVO.updatedAt) {
        aclUserInfoPageVO.updatedAt = { [madAdminOrmDB.Op.gte]: new Date(madAdminOrmDB.updatedAt).getTime() };
    }
    let page = Number(aclUserInfoPageVO.page);
    let size = Number(aclUserInfoPageVO.size);
    let where = Object.assign(aclUserInfoPageVO);
    delete aclUserInfoPageVO.page;
    delete aclUserInfoPageVO.size;
    let resList = await aclUserInfo.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["user_id", "asc"]],
        raw: true
    });
    return resList;
};
export const add = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    if (!aclUserInfoVO.createdAt) {
        aclUserInfoVO.createdAt = new Date();
    }
    if (!aclUserInfoVO.updatedAt) {
        aclUserInfoVO.updatedAt = new Date();
    }

    let dbInfo = await aclUserInfo.prototype.findOne({
        where: {
            account: aclUserInfoVO.account,
        },
        raw: true
    });

    if (dbInfo) {
        throw ErrorCode.USER_EXIST;
    }
    aclUserInfoVO.pwd = CryptUtils.defPwd();
    aclUserInfoVO.pwd_salt = CryptUtils.salt();

    await aclUserInfo.prototype.create(aclUserInfoVO);
    userOptLogService.addOptLog(null, `add user : ${JSON.stringify(aclUserInfoVO)}`, userOptLogService.LogType.ADD, '', session_id);
    return 'ok';
};

export const update = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    let userId = Number(aclUserInfoVO.user_id);

    await getOneAclUserByUid(userId);

    let updateInfo = Object.create(null);
    updateInfo.remark = aclUserInfoVO.remark;
    updateInfo.updatedAt = new Date();

    if (aclUserInfoVO.phone) {
        updateInfo.phone = aclUserInfoVO.phone
    }
    if (aclUserInfoVO.email) {
        updateInfo.email = aclUserInfoVO.email
    }

    await aclUserInfo.prototype.update(updateInfo, {
        where: {
            user_id: userId
        }
    });
    userOptLogService.addOptLog(null, `update user : ${JSON.stringify(aclUserInfoVO)}`, userOptLogService.LogType.UPDATE, '', session_id);

    return 'ok';
};

export const updateStatus = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    let userId = Number(aclUserInfoVO.user_id);

    await getOneAclUserByUid(userId);

    let updateInfo = Object.create(null);
    updateInfo.status = aclUserInfoVO.user_status;
    updateInfo.updatedAt = new Date();

    await aclUserInfo.prototype.update(updateInfo, {
        where: {
            user_id: userId
        }
    });
    userOptLogService.addOptLog(null, `update status : ${JSON.stringify(aclUserInfoVO)}`, userOptLogService.LogType.UPDATE, '', session_id);
    return 'ok';
};


export const resetPwd = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    let userId = Number(aclUserInfoVO.user_id);

    await getOneAclUserByUid(userId);

    let updateInfo = Object.create(null);
    updateInfo.pwd = CryptUtils.defPwd();
    updateInfo.pwd_status = AclUserInfoConst.PWD_STATUS.DEFAULT;
    updateInfo.user_status = AclUserInfoConst.USER_STATUS.NORMAL;
    updateInfo.updatedAt = new Date();

    await aclUserInfo.prototype.update(updateInfo, {
        where: {
            user_id: userId
        }
    });
    userOptLogService.addOptLog(null, `reset pwd : ${JSON.stringify(aclUserInfoVO)}`, userOptLogService.LogType.UPDATE, '', session_id);

    return 'ok';
};


export async function updateUserStatus(userId: any, userStatus: number) {
    try {
        await aclUserInfo.prototype.update({
            user_status: userStatus
        }, {
            where: {
                user_id: userId,
            }
        });
    }
    catch (e) {
        logger.error('aclUserService.updateUserStatus.error:' + e)
        throw e;
    }
}

export async function findAllForPage(page: number, size: number, userId?: number,
                                     typeArr?: any, statusArr?: any,
                                     fromTime?: any, toTime?: any, remark?: any) {
    try {
        let where = Object.create(null);
        if (userId) {
            where.user_id = userId
        }
        if (typeArr) {
            where.user_type = { [madAdminOrmDB.Op.in]: typeArr };
        }
        if (statusArr) {
            where.user_status = { [madAdminOrmDB.Op.in]: statusArr };
        }
        if (fromTime && toTime) {
            where.createdAt = { [madAdminOrmDB.Op.between]: [fromTime, toTime] };
        }
        if (remark) {
            where.remark = { [madAdminOrmDB.Op.like]: `${remark}%` };
        }

        let resList = await aclUserInfo.prototype.findAndCount({
            where: where,
            limit: size,
            offset: (page - 1) * size,
            order: [["user_id", "asc"]],
            raw: true
        });
        return resList;
    }
    catch (e) {
        logger.error('aclUserService.findAllForPage.error:' + e)
        throw e;
    }

}



