import * as financeAccountService from "../service/fianceAccount.service";
import { FinanceAccountVO, FinanceAccountPageVO, AccountInitAssetVO } from "../service/fianceAccount.service";


let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert, datetimeUtils } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUserId } from "../../../utils/aclUserUtils";
import { coinType, contractPairs, spotPairs } from "@madex/ex-ts-dao";

let isIp = require('is-ip');
/**
 * 账户列表
 * @param req
 * @param infoVO
 */
export const list = async (req: any, pageVO: FinanceAccountPageVO) => {
    let func_name = "financeAccount.control.list";
    try {
        pageVO.page = Optional.opt(pageVO, 'page', 1);
        pageVO.size = Optional.opt(pageVO, 'size', 20);
        let res = await financeAccountService.list(pageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 添加新的账户
 * @param req
 * @param authConfigVO
 */
export const add = async (req: any, financeAccountVO: FinanceAccountVO) => {
    let func_name = "financeAccount.control.add";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await paramValid(financeAccountVO);

        let res = await financeAccountService.add(financeAccountVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 修改账户信息
 * @param req
 * @param authConfigVO
 */
export const update = async (req: any, financeAccountVO: FinanceAccountVO) => {
    let func_name = "financeAccount.control.update";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!financeAccountVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        await paramValid(financeAccountVO);

        let res = await financeAccountService.update(financeAccountVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 删除账户信息
 * @param req
 * @param authConfigVO
 */
export const del = async (req: any, financeAccountVO: FinanceAccountVO) => {
    let func_name = "financeAccount.control.del";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!financeAccountVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await financeAccountService.del(financeAccountVO.id, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 添加初始资金
 * @param req
 * @param authConfigVO
 */
export const addInitAsset = async (req: any, accountInitAssetVO: AccountInitAssetVO) => {
    let func_name = "financeAccount.control.addInitAsset";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        if (!accountInitAssetVO.account_id || !accountInitAssetVO.symbol ||
            !accountInitAssetVO.amount_usdt ||
            !accountInitAssetVO.amount ||
            Number(accountInitAssetVO.amount_usdt) < 0 ||
            Number(accountInitAssetVO.amount) < 0) {
            throw ErrorCode.PARAM_MISS;
        }
        let res = await financeAccountService.addInitAsset(accountInitAssetVO, currentUserId, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 初始资金列表
 * @param req
 * @param authConfigVO
 */
export const initAssetList = async (req: any, pageVO: FinanceAccountPageVO) => {
    let func_name = "financeAccount.control.initAssetList";
    try {
        pageVO.page = Optional.opt(pageVO, 'page', 1);
        pageVO.size = Optional.opt(pageVO, 'size', 20);
        if (!pageVO.account_id) {
            throw ErrorCode.PARAM_MISS;
        }
        let res = await financeAccountService.initAssetList(pageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

async function paramValid(financeAccountVO: FinanceAccountVO) {
    if (!financeAccountVO.category_id || !financeAccountVO.account ||
        !financeAccountVO.platform) {
        throw ErrorCode.PARAM_MISS;
    }
    if (financeAccountVO.platform == 1) {//KTX
        if (!financeAccountVO.uid) {
            throw ErrorCode.PARAM_MISS;
        }
    }
    else if (financeAccountVO.platform == 2) {//币安
        if (!financeAccountVO.apikey || !financeAccountVO.secret) {
            throw ErrorCode.PARAM_MISS;
        }
    }
    else if (financeAccountVO.platform == 3) {
        if (!financeAccountVO.apikey || !financeAccountVO.secret || !financeAccountVO.pwd) {
            throw ErrorCode.PARAM_MISS;
        }
    }
    else {
        throw ErrorCode.PARAM_MISS;
    }
}

