import { financeAccount, ormDB, financeAccountInitAsset, financeAccountCategory } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";


let _ = require('lodash');
let { logger } = require('@madex/ex-js-public');


export interface FinanceAccountVO {
    id?: number;

    category_id?: number;

    uid?: number;

    account?: string | any;

    remark?: string | any;

    apikey?: string | any;

    secret?: string | any;

    pwd?: string | any;

    platform?: number;

    status?: number;

    balance_warning?: number;

    createdAt?: Date | any,

    updatedAt?: Date | any,

}


export interface FinanceAccountPageVO extends FinanceAccountVO, AccountInitAssetVO {

    page?: number,

    size?: number,


}

export interface AccountInitAssetVO {
    id?: number;

    account_id?: number;

    symbol?: string | any;

    amount?: number;

    amount_usdt?: number;

    remark?: string | any;

    createdAt?: Date | any,

    updatedAt?: Date | any,

}


export async function list(pageVO: FinanceAccountPageVO) {
    let where = {
        category_id: pageVO.category_id
    };

    let resList = await financeAccount.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["id", "asc"]],
        raw: true
    });
    if (resList.rows.length) {
        let cids = resList.rows.map(item => item.category_id);
        let cList = await financeAccountCategory.prototype.findAll({
            where: {
                id: cids
            },
            raw: true
        });
        let tmpMap: any = {};
        for (let item of cList) {
            tmpMap[item.id] = item.name;
        }
        for (let item of resList.rows) {
            item.category_name = tmpMap[item.category_id] ? tmpMap[item.category_id] : ""
        }

    }

    return resList;
}


export async function add(financeAccountVO: FinanceAccountVO, currentUserId: any, ip: any) {
    let dbOne = await financeAccount.prototype.findOne({
        where: {
            category_id: financeAccountVO.category_id,
            apikey: financeAccountVO.apikey
        },
        raw: true
    });
    if (dbOne) {
        throw ErrorCode.DATA_EXIST;
    }

    financeAccountVO.createdAt = new Date();
    financeAccountVO.updatedAt = new Date();
    if (!financeAccountVO.status) {
        //默认可用
        financeAccountVO.status = 1;
    }
    if (!financeAccountVO.balance_warning) {
        financeAccountVO.balance_warning = 0;
    }
    if (!financeAccountVO.uid) {
        financeAccountVO.uid = 0;
    }

    await financeAccount.prototype.create(financeAccountVO);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增账户', ip, JSON.stringify(financeAccountVO), '金融部-其他管理');
    return 'success';
}


export async function update(financeAccountVO: FinanceAccountVO, currentUserId: any, ip: any) {
    let exist = await financeAccount.prototype.findOne({
        where: {
            id: financeAccountVO.id
        },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    let dbOne = await financeAccount.prototype.findOne({
        where: {
            category_id: financeAccountVO.category_id,
            apikey: financeAccountVO.apikey,
            id: { [ormDB.Op.ne]: financeAccountVO.id }
        },
        raw: true
    });

    if (dbOne) {
        throw ErrorCode.CATEGORY_HAVE_SAME_ACCOUNT;
    }

    if (!financeAccountVO.balance_warning) {
        financeAccountVO.balance_warning = 0;
    }
    if (!financeAccountVO.uid) {
        financeAccountVO.uid = 0;
    }

    financeAccountVO.updatedAt = new Date();
    await financeAccount.prototype.update(financeAccountVO, {
        where: {
            id: Number(financeAccountVO.id)
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, 0, '修改账户信息', ip, JSON.stringify(financeAccountVO), '金融部-其他管理');

    return 'success';
}

export async function del(id: number, currentUserId: any, ip: string | undefined) {
    let exist = await financeAccount.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    let dbInitAsset = await financeAccountInitAsset.prototype.findOne({
        where: {
            account_id: id
        },
        raw: true
    });
    if (dbInitAsset) {
        throw ErrorCode.ACCOUNT_HAVE_INIT_ASSET
    }


    await financeAccount.prototype.destroy({
        where: {
            id: id
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, 0, '删除账户信息', ip, `id:${id}`, '金融部-其他管理');

    return 'success';
}

export async function addInitAsset(accountInitAssetVO: AccountInitAssetVO, currentUserId: any, ip: any) {


    accountInitAssetVO.createdAt = new Date();
    accountInitAssetVO.updatedAt = new Date();


    await financeAccountInitAsset.prototype.create(accountInitAssetVO);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增初始资金', ip, JSON.stringify(accountInitAssetVO), '金融部-其他管理');
    return 'success';
}

export async function initAssetList(pageVO: FinanceAccountPageVO) {
    let where = {
        account_id: pageVO.account_id
    };

    let resList = await financeAccountInitAsset.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["createdAt", "desc"]],
        raw: true
    });

    if (resList.rows.length) {
        let aids = resList.rows.map(item => item.account_id);
        let aList = await financeAccount.prototype.findAll({
            where: {
                id: aids
            },
            raw: true
        });
        let tmpMap: any = {};
        for (let item of aList) {
            tmpMap[item.id] = item.account;
        }
        for (let item of resList.rows) {
            item.account_name = tmpMap[item.account_id] ? tmpMap[item.account_id] : ""
        }

    }
    return resList;
}



