// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { i18nInfo, i18nInfoHistory, ormDB, i18nInfoPreview } from "@madex/ex-ts-dao";
import * as i18nLogService from "../service/i18nlog.service";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { sendRobotMessage, ROBOT_KEYS } from "../../../utils/robotUtils";

let { logger } = require('@madex/ex-js-public');
let _ = require('lodash');

export interface I18nInfoVO {
    info_json: string;

    site: string;
}

export interface I18nInfoPageVO extends I18nInfoVO {
    page?: number;

    size?: number

    platform: number;

    module: number;

    code?: string | any;

    zh_cn?: string | any;
}

export const list = async (infoPageVO: I18nInfoPageVO) => {
    let condition = {
        site: infoPageVO.site
    };
    if (infoPageVO.platform) {
        condition['platform'] = Number(infoPageVO.platform);
    }
    if (infoPageVO.module) {
        condition['module'] = Number(infoPageVO.module);
    }
    if (infoPageVO.code) {
        condition['code'] = infoPageVO.code;
    }
    if (infoPageVO.zh_cn) {
        condition[ormDB.Op.or] = {
            zh_cn: { [ormDB.Op.like]: `${infoPageVO.zh_cn}%` },
            en_us: { [ormDB.Op.like]: `${infoPageVO.zh_cn}%` },
            ja_jp: { [ormDB.Op.like]: `${infoPageVO.zh_cn}%` },
            ko_kr: { [ormDB.Op.like]: `${infoPageVO.zh_cn}%` },
            vi_vn: { [ormDB.Op.like]: `${infoPageVO.zh_cn}%` },
            ar_ae: { [ormDB.Op.like]: `${infoPageVO.zh_cn}%` },
        };
    }

    let page = Number(infoPageVO.page);
    let size = Number(infoPageVO.size);
    let resList = await i18nInfo.prototype.findAndCount({
        where: condition,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
};
export const add = async (infoVO: I18nInfoVO, currentUserId: any, ip: any) => {

    let infoJson = infoVO.info_json;
    if (!infoJson) {
        throw ErrorCode.PARAM_MISS;
    }
    let infoList = JSON.parse(infoJson);

    let lastVersionList = await getLastVersionList(infoVO.site);

    let timestamp = new Date();
    let insertHistoryList: any = [];
    let idList: any = [];
    let lastVersionMap: any = {};

    for (let item of lastVersionList) {
        let key = item['platform'] + '_' + item['module'] + '_' + item['code'];
        lastVersionMap[key] = item;
        idList.push(item['id']);
        item['info_id'] = item['id'];
        item['createdAt'] = item['updatedAt'];
        delete item['id'];
        insertHistoryList.push(item);
    }

    let version = await getVersion();

    let insertOrUpdateList: any = [];
    for (let insertOne of infoList) {
        if (!insertOne.platform || !insertOne.module || !insertOne.code) {
            throw ErrorCode.PARAM_MISS;
        }
        let key = insertOne['platform'] + '_' + insertOne['module'] + '_' + insertOne['code'];

        insertOne.version = version;
        insertOne.site = infoVO.site;
        insertOne.sub_code = insertOne.sub_code !== null ? insertOne.sub_code :
            (lastVersionMap[key] ? lastVersionMap[key].sub_code : '');

        insertOne.zh_cn = insertOne.zh_cn !== null ? insertOne.zh_cn :
            (lastVersionMap[key] ? lastVersionMap[key].zh_cn : '');

        insertOne.en_us = insertOne.en_us !== null ? insertOne.en_us :
            (lastVersionMap[key] ? lastVersionMap[key].en_us : '');

        insertOne.ja_jp = insertOne.ja_jp !== null ? insertOne.ja_jp :
            (lastVersionMap[key] ? lastVersionMap[key].ja_jp : '');

        insertOne.ko_kr = insertOne.ko_kr !== null ? insertOne.ko_kr :
            (lastVersionMap[key] ? lastVersionMap[key].ko_kr : '');

        insertOne.vi_vn = insertOne.vi_vn !== null ? insertOne.vi_vn :
            (lastVersionMap[key] ? lastVersionMap[key].vi_vn : '');

        insertOne.ar_ae = insertOne.ar_ae !== null ? insertOne.ar_ae :
            (lastVersionMap[key] ? lastVersionMap[key].ar_ae : '');

        insertOne.createdAt = timestamp;
        insertOne.updatedAt = timestamp;
        insertOrUpdateList.push(insertOne);
    }

    let tx;
    try {
        tx = await ormDB.transaction();

        //先迁移历史表
        if (insertHistoryList.length) {
            await i18nInfoHistory.prototype.bulkCreate(insertHistoryList, {
                transaction: tx
            });
        }
        //更新版本
        if (lastVersionList.length) {
            await i18nInfo.prototype.update({
                version: version,
                updatedAt: timestamp
            }, {
                where: {
                    id: idList
                },
                transaction: tx
            })
        }
        //批量插入或更新
        if (insertOrUpdateList.length) {
            await i18nInfo.prototype.bulkCreate(insertOrUpdateList, {
                updateOnDuplicate: ['sub_code', 'zh_cn', 'en_us', 'ja_jp', 'ko_kr', 'vi_vn', 'ar_ae', 'updatedAt'],
                transaction: tx
            });
        }
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('i18nInfo.add.error:' + e);
        throw e;
    }
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增国际化信息', ip, insertOrUpdateList.length, '国际化管理');
    sendRobotMessage(ROBOT_KEYS.COMMON_KEY, `新增国际化信息:${JSON.stringify(infoVO)}`);
    return 'ok';
};

export const preview = async (infoVO: I18nInfoVO, currentUserId: any, ip: any) => {

    let infoJson = infoVO.info_json;
    if (!infoJson) {
        throw ErrorCode.PARAM_MISS;
    }
    let infoList = JSON.parse(infoJson);
    let lastVersionList = await getLastVersionList(infoVO.site);

    let timestamp = new Date();
    let lastVersionMap: any = {};

    for (let item of lastVersionList) {
        let key = item['platform'] + '_' + item['module'] + '_' + item['code'];

        delete item['id'];
        delete item['version'];

        lastVersionMap[key] = item;

    }


    for (let insertOne of infoList) {
        if (!insertOne.platform || !insertOne.module || !insertOne.code) {
            throw ErrorCode.PARAM_MISS;
        }
        let key = insertOne['platform'] + '_' + insertOne['module'] + '_' + insertOne['code'];
        insertOne.site = infoVO.site;
        insertOne.sub_code = insertOne.sub_code !== null ? insertOne.sub_code :
            (lastVersionMap[key] ? lastVersionMap[key].sub_code : '');

        insertOne.zh_cn = insertOne.zh_cn !== null ? insertOne.zh_cn :
            (lastVersionMap[key] ? lastVersionMap[key].zh_cn : '');

        insertOne.en_us = insertOne.en_us !== null ? insertOne.en_us :
            (lastVersionMap[key] ? lastVersionMap[key].en_us : '');

        insertOne.ja_jp = insertOne.ja_jp !== null ? insertOne.ja_jp :
            (lastVersionMap[key] ? lastVersionMap[key].ja_jp : '');

        insertOne.ko_kr = insertOne.ko_kr !== null ? insertOne.ko_kr :
            (lastVersionMap[key] ? lastVersionMap[key].ko_kr : '');

        insertOne.vi_vn = insertOne.vi_vn !== null ? insertOne.vi_vn :
            (lastVersionMap[key] ? lastVersionMap[key].vi_vn : '');

        insertOne.ar_ae = insertOne.ar_ae !== null ? insertOne.ar_ae :
            (lastVersionMap[key] ? lastVersionMap[key].ar_ae : '');

        insertOne.updatedAt = timestamp;
        lastVersionMap[key] = insertOne;
    }

    let resList: any = [];
    for (let key of _.keys(lastVersionMap)) {
        resList.push(lastVersionMap[key]);
    }

    await i18nInfoPreview.prototype.bulkCreate(resList, {
        updateOnDuplicate: ['sub_code', 'zh_cn', 'en_us', 'ja_jp', 'ko_kr', 'vi_vn', 'ar_ae', 'updatedAt'],
    });

    let orderByList = _.orderBy(resList, ['updatedAt'], ['desc']);

    return orderByList;
};

export const versionList = async (site: any) => {
    let options = {
        attributes: ['version', 'createdAt', 'updatedAt'],
        where: {
            site: Number(site)
        },
        order: [['version', 'desc']],
        group: ['version', 'createdAt', 'updatedAt'],
        raw: true
    }
    let task1 = i18nInfo.prototype.findAll(options);
    let task2 = i18nInfoHistory.prototype.findAll(options);

    let [currentInfo, historyInfo] = await Promise.all([task1, task2]);

    return {
        current_version: currentInfo,
        history_version: historyInfo
    }

};

export const checkList = async (infoPageVO: I18nInfoPageVO) => {
    let condition = {
        site: infoPageVO.site,
        [ormDB.Op.or]: {
            zh_cn: '',
            en_us: '',
            ja_jp: '',
            ko_kr: '',
            vi_vn: '',
            ar_ae: '',
        }
    };
    if (infoPageVO.platform) {
        condition['platform'] = Number(infoPageVO.platform);
    }
    if (infoPageVO.module) {
        condition['module'] = Number(infoPageVO.module);
    }
    if (infoPageVO.code) {
        condition['code'] = infoPageVO.code;
    }


    let page = Number(infoPageVO.page);
    let size = Number(infoPageVO.size);
    let resList = await i18nInfo.prototype.findAndCount({
        where: condition,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
};


/**
 * 查询站点版本记录
 */
export async function getLastVersionList(site: any) {
    return await i18nInfo.prototype.findAll({
        where: {
            site: Number(site)
        },
        order: [['id', 'asc']],
        raw: true
    });
}

async function getVersion() {
    let version: any;
    let options = {
        where: {},
        order: [['version', 'desc']],
        raw: true
    }
    let task1 = i18nInfo.prototype.findOne(options);
    let task2 = i18nInfoHistory.prototype.findOne(options);
    let [oneInfo, oneHistory] = await Promise.all([task1, task2]);

    if (oneInfo && oneHistory) {
        version = _.max([oneInfo.version, oneHistory.version]);
    }
    else if (oneInfo && !oneHistory) {
        version = oneInfo.version;
    }
    else {
        return '1.0.0';
    }
    return addVersion(version, 1);

}


/**
 * 版本号+1
 * @param version
 * @param add
 */
function addVersion(version: string, add: number) {

    let versionStrArr = version.split('.');
    let versionNumArr: number[] = []
    for (let i = 0; i < versionStrArr.length; i++) {
        versionNumArr[i] = Number(versionStrArr[i]);
    }

    for (let i = versionNumArr.length - 1; i >= 0; i--) {
        versionNumArr[i] += add;
        while (versionNumArr[i] >= 20) {
            versionNumArr[i] -= 20;
            if (i > 0) {
                versionNumArr[i - 1]++;
            }
        }
        add = 0;
    }

    return versionNumArr.join('.');

}


