import * as mUserManageService from "../service/mUserManage.service";
import { QueryVO } from "../service/mUserManage.service";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUserId, } from "../../../utils/aclUserUtils";

import * as IsEmail from "isemail";
import { isLimitUserId } from "../../../utils/mUserCommonUtils";

let isIp = require('is-ip');

/**
 * Madex 用户列表/kyc用户列表
 * @param req
 * @param infoVO
 */
export const userList = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.userList";
    try {
        queryVO.page = Optional.opt(queryVO, 'page', 1);
        queryVO.size = Optional.opt(queryVO, 'size', 20);
        let condition = String(queryVO.condition);
        //查询类型 1:uid/姓名/邮箱/证件号  2:充币/提币地址
        let condition_type = 1;
        if (condition && condition.length >= 32) {
            condition_type = 2;
        }
        let res = await mUserManageService.userList(Number(queryVO.page), Number(queryVO.size), condition, condition_type);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * Madex 单个用户详情
 * @param req
 * @param queryVO
 */
export const oneUserDetail = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.oneUserDetail";
    try {
        if (!queryVO.user_id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await mUserManageService.oneUserDetail(queryVO.user_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * Madex 修改用户邮箱
 * @param req
 * @param queryVO
 */
export const updateUserEmail = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.updateUserEmail";
    try {
        if (!queryVO.user_id || !queryVO.email || !queryVO.totp_code) {
            throw ErrorCode.PARAM_MISS
        }

        //uid 是否在受限范围内
        if (isLimitUserId(queryVO.user_id)) {
            throw ErrorCode.UID_LIMIT
        }
        if (!IsEmail.validate(queryVO.email, { minDomainAtoms: 2 })) {
            //邮箱格式错误
            throw '2000';
        }
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await mUserManageService.updateUserEmail(currentUserId, queryVO.user_id, queryVO.email, queryVO.totp_code, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
/**
 * Madex 锁定账户
 * @param req
 * @param queryVO
 */
export const lockAccount = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.lockAccount";
    try {
        if (!queryVO.user_id || !queryVO.lock_type || !queryVO.totp_code) {
            throw ErrorCode.PARAM_MISS
        }

        //uid 是否在受限范围内
        if (isLimitUserId(queryVO.user_id)) {
            throw ErrorCode.UID_LIMIT
        }

        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await mUserManageService.lockAccount(currentUserId, queryVO.user_id, ip, queryVO.lock_type, queryVO.totp_code);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 解锁账户
 * @param req
 * @param queryVO
 */
export const unlockAccount = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.unlockAccount";
    try {
        if (!queryVO.user_id || !queryVO.lock_type || !queryVO.totp_code) {
            throw ErrorCode.PARAM_MISS
        }

        //uid 是否在受限范围内
        if (isLimitUserId(queryVO.user_id)) {
            throw ErrorCode.UID_LIMIT
        }

        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await mUserManageService.unlockAccount(currentUserId, queryVO.user_id, ip, queryVO.lock_type, queryVO.totp_code);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
/**
 * Madex 清除登陆限制
 * @param req
 * @param queryVO
 */
export const clearLoginLimit = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.clearLoginLimit";
    try {
        if (!queryVO.user_id || !queryVO.totp_code) {
            throw ErrorCode.PARAM_MISS
        }

        //uid 是否在受限范围内
        if (isLimitUserId(queryVO.user_id)) {
            throw ErrorCode.UID_LIMIT
        }

        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await mUserManageService.clearLoginLimit(currentUserId, queryVO.user_id, ip, queryVO.totp_code);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * Madex 清除24小时提现限制
 * @param req
 * @param queryVO
 */
export const clear24WithdrawLimit = async (req: any, queryVO: QueryVO) => {
    let func_name = "mUserManageCtrl.clear24WithdrawLimit";
    try {
        if (!queryVO.user_id || !queryVO.totp_code) {
            throw ErrorCode.PARAM_MISS
        }

        //uid 是否在受限范围内
        if (isLimitUserId(queryVO.user_id)) {
            throw ErrorCode.UID_LIMIT
        }

        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await mUserManageService.clear24WithdrawLimit(currentUserId, queryVO.user_id, ip, queryVO.totp_code);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


