import * as abkUserService from "../../service/v2/abkUserInfo.service";
import { AbkUserInfoVO, AbkUserInfoPageVO, resetAbkTotp } from "../../service/v2/abkUserInfo.service";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../../constant/errorCode";
import { getCurrentUser, getCurrentUserId, } from "../../../../utils/aclUserUtils";
import { isSuperUser } from "../../../../utils/abkUserUtils";

let isIp = require('is-ip');

/**
 * 登陆
 * @param req
 * @param infoVO
 */
export const login = async (req: any, abkUserInfoVO: AbkUserInfoVO, res: any) => {
    let func_name = "abkUserInfo.control.login";
    let cmd = req.path;
    let result: any;
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';

        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, abkUserInfoVO.account);
        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, abkUserInfoVO.pwd);
        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, abkUserInfoVO.totp_code);
        ApiAssert.isFalse(ErrorCode.ACCOUNT_OR_PWD_ERR, abkUserInfoVO.pwd.length < 8);
        ApiAssert.isFalse(ErrorCode.ACCOUNT_OR_PWD_ERR, abkUserInfoVO.pwd.length > 12);
        checkPwd(abkUserInfoVO.pwd);
        result = await abkUserService.login(abkUserInfoVO.account, abkUserInfoVO.pwd, abkUserInfoVO.totp_code, ip || "");
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
    if (result.result && result.sessionId) {
        if (req.headers.origin) {
            let iSecure = req.headers.origin.startsWith("https");

            let options = {
                expires: new Date(new Date().getTime() + 1000 * 60 * 60 * 8), // 15 days
                httpOnly: true,
                secure: iSecure
            };

            res.cookie('session_id', result.sessionId, options);
            delete result.sessionId;
        }
    }
    return Res3Utils.result(result);
};

/**
 * 登出
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const logout = async (req: any, abkUserInfoVO: AbkUserInfoVO) => {
    let func_name = "abkUserInfo.control.logout";
    let cmd = req.path;
    try {
        if (req.cookies.session_id) {
            let currentUser = await getCurrentUser(req.cookies.session_id);
            if (currentUser) {
                await abkUserService.deleteAllAbkSessionByUserId(currentUser.userId);
            }
        }
        return Res3Utils.result('ok');
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};
/**
 * 获取当前登陆的用户信息
 * @param req
 * @param infoVO
 */
export const getInfo = async (req: any, abkUserInfoVO: AbkUserInfoVO) => {
    let func_name = "abkUserInfo.control.getInfo";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await abkUserService.getInfo(currentUserId, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 获取用户信息详情（这里主要包含密码、google等敏感信息）根据uid
 * @param req
 * @param infoVO
 */
export const getInfoDetailByUserId = async (req: any, abkUserInfoVO: AbkUserInfoVO) => {
    let func_name = "abkUserInfo.control.getInfoDetailByUserId";
    let cmd = req.path;
    try {
        if (!abkUserInfoVO.user_id) {
            throw ErrorCode.PARAM_MISS
        }
        if (!abkUserInfoVO.totp_code) {
            throw ErrorCode.PARAM_MISS
        }
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await isSuperUser(currentUserId);
        let res = await abkUserService.getInfoDetailByUserId(abkUserInfoVO.user_id, abkUserInfoVO.totp_code, currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 重置谷歌-生成一个新的密钥返回，保存时重新绑定谷歌
 * @param req
 * @param authConfigVO
 */
export const resetTotp = async (req: any, abkUserInfoVO: AbkUserInfoVO) => {
    let func_name = "abkUserInfo.control.resetTotp";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await isSuperUser(currentUserId)
        let res = await abkUserService.resetAbkTotp(abkUserInfoVO.user_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 获取用户列表
 * @param req
 * @param infoVO
 */
export const userList = async (req: any, abkUserInfoPageVO: AbkUserInfoPageVO) => {
    let func_name = "abkUserInfo.control.userList";
    let cmd = req.path;
    try {
        abkUserInfoPageVO.page = Optional.opt(abkUserInfoPageVO, 'page', 1);
        abkUserInfoPageVO.size = Optional.opt(abkUserInfoPageVO, 'size', 20);
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await isSuperUser(currentUserId)
        let res = await abkUserService.userList(abkUserInfoPageVO, currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 修改用户状态
 * @param req
 * @param infoVO
 */
export const updateStatus = async (req: any, abkUserInfoPageVO: AbkUserInfoPageVO) => {
    let func_name = "abkUserInfo.control.updateStatus";
    let cmd = req.path;
    try {
        if (!abkUserInfoPageVO.user_id) {
            throw ErrorCode.PARAM_MISS
        }
        if (!abkUserInfoPageVO.user_status) {
            throw ErrorCode.PARAM_MISS
        }
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await isSuperUser(currentUserId)
        let res = await abkUserService.updateStatus(abkUserInfoPageVO.user_id, abkUserInfoPageVO.user_status, currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


function checkPwd(pwd: string) {
    let reg = /^(?=.[0-9])(?=.[A-Z])(?=.[a-z])(?=.[!@#%^&*?]).{8,12}$/;
    if (!reg.test(pwd)) {
        throw ErrorCode.PWD_FORMAT_ERR;
    }
}


/**
 * 新增用户信息
 * @param req
 * @param infoVO
 */
export const addAbkUser = async (req: any, abkUserInfoVO: AbkUserInfoVO) => {
    let func_name = "abkUserInfo.control.getInfo";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        await isSuperUser(currentUserId);
        addCheck(abkUserInfoVO);
        let res = await abkUserService.addAbkUser(abkUserInfoVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

function addCheck(abkUserInfoVO: AbkUserInfoVO) {
    if (!abkUserInfoVO.account || !abkUserInfoVO.pwd || !abkUserInfoVO.totp_encrypt) {
        throw ErrorCode.PARAM_MISS;
    }
    checkPwd(abkUserInfoVO.pwd);
}

