import { madAdminOrmDB, abkUserInfo } from "@madex/ex-ts-dao";
import { getOneAbkUserByAccount, getOneAbkUserByUid, checkAbkTotp } from "../../../../utils/abkUserUtils";
import { ErrorCode } from "../../../../constant/errorCode";
import { AbkUserInfoConst } from "../../../../constant/abkUserConstant";
import { CryptUtils } from "../../../../utils/crypt-utils";
import { RedisVal } from "../../../../constant/redis-val";
import Config from "../../../../../config";

const Otplib = require('otplib');

let { logger, apiAssertUtils: ApiAssert, BigNumberUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');

let _ = require('lodash');

export interface AbkUserInfoVO {
    user_id?: number;

    account?: string | any;

    remark?: string | any;//备注

    pwd?: string | any;

    pwd_salt?: string;

    user_status?: number;

    user_type?: number;

    totp_encrypt?: string;

    allow_ips?: string;

    createdAt?: Date | any;

    updatedAt?: Date | any;

    totp_code?: any

}

export interface AbkUserInfoPageVO extends AbkUserInfoVO {
    page?: number,

    size?: number
}


export async function login(account: any, pwd: any, totp_code: any, ip: string) {

    let userInfo = await getOneAbkUserByAccount(account);
    ApiAssert.isNotEmpty(ErrorCode.ACCOUNT_OR_PWD_ERR, userInfo);

    ApiAssert.isFalse(ErrorCode.ACCOUNT_STOP, userInfo.user_status === AbkUserInfoConst.USER_STATUS.STOP);
    ApiAssert.isFalse(ErrorCode.USER_IS_DEL, userInfo.user_status === AbkUserInfoConst.USER_STATUS.DEL);
    //校验谷歌
    await checkAbkTotp(userInfo.user_id, totp_code);

    await _checkPwd(userInfo, pwd);

    let sessionId = CryptUtils.sessionId(userInfo.user_id);
    RedisClient.rpush(RedisVal.abkSessionListKey(userInfo.user_id), sessionId);

    let cookies = {
        account: userInfo.account,
        userId: userInfo.user_id,
        is_abk_user: 1,
        allow_ips: userInfo.allow_ips,//ip白名单
        totp_encrypt: ''
    };
    let hasTotp = 0

    //如果绑定了谷歌则必须校验, 增加属性 needConfirm = 1
    let totp_encrypt = userInfo.totp_encrypt;
    if (totp_encrypt) {
        cookies.totp_encrypt = totp_encrypt;
        hasTotp = 1;
    }

    await RedisClient.writeSync(sessionId, cookies, Config.LOGIN_EXPIRED);
    await _unlockPwd(userInfo.user_id);

    //第一次登陆状态改为在用
    if (!userInfo.user_status) {
        updateAbkUserStatus(userInfo.user_id, 1);
    }
    return {
        result: "success",
        sessionId: sessionId,
        //是否已绑定谷歌。0:未绑定；1:已绑定
        hasTotp: hasTotp,

    };
}


/**
 * 校验密码
 * @param userInfo
 * @param pwd
 */
async function _checkPwd(userInfo: any, pwd: any) {
    let pwdSuc = userInfo.pwd == pwd;
    if (!pwdSuc) await _pwdError(userInfo.user_id);
}

async function _pwdError(userId: any) {
    let key = RedisVal.abkLoginErrTimesKey(userId)
    let errorCount = await RedisClient.getMemSync(key);
    errorCount = BigNumberUtils.add(errorCount, 1);
    await RedisClient.writeSync(key, errorCount);

    if (errorCount && Number(errorCount) >= Config.LOGIN_ERROR_LIMIT) {
        await updateAbkUserStatus(userId, AbkUserInfoConst.USER_STATUS.STOP);
        throw ErrorCode.ACCOUNT_LOCK;
    }
    throw ErrorCode.ACCOUNT_OR_PWD_ERR;
}

async function _unlockPwd(userId: any) {
    let key = RedisVal.abkLoginErrTimesKey(userId);
    await RedisClient.delSync(key);
}

/**
 * 更新用户状态
 * @param user_id
 * @param user_status
 */
export async function updateAbkUserStatus(user_id: number, user_status: number) {
    try {
        await abkUserInfo.prototype.update({
            user_status: Number(user_status)
        }, {
            where: {
                user_id: Number(user_id),
            }
        });
    }
    catch (e) {
        logger.error('abkUserService.updateAbkUserStatus.error:' + e)
        throw e;
    }
}

export async function deleteAllAbkSessionByUserId(userId: number) {

    //获取该账户使用过的所有sessionId
    let sessionListKey = RedisVal.abkSessionListKey(userId)
    RedisClient.lrange(sessionListKey, 0, -1, async (err, reply) => {
        //删除所有sessionId
        if (!err && reply && reply.length >= 1) {
            await RedisClient.delSync(...reply)
        }
        //删除sessionList
        await RedisClient.delSync(sessionListKey)
    });
}


export const getInfo = async (currentUserId: number | any, sessionId: string) => {
    let dbUserInfo = await getOneAbkUserByUid(currentUserId);

    ApiAssert.isNotEmpty(ErrorCode.USER_NOT_EXIST, dbUserInfo);
    ApiAssert.isFalse(ErrorCode.ACCOUNT_STOP, dbUserInfo.user_status === AbkUserInfoConst.USER_STATUS.STOP);
    ApiAssert.isFalse(ErrorCode.USER_IS_DEL, dbUserInfo.user_status === AbkUserInfoConst.USER_STATUS.DEL);

    let data = {

        userId: dbUserInfo.user_id,
        account: dbUserInfo.account,
        remark: dbUserInfo.remark,
        allow_ips: dbUserInfo.allow_ips,
        user_status: dbUserInfo.user_status,
        user_type: dbUserInfo.user_type,
        sessionId: sessionId,
        hasTotp: dbUserInfo && dbUserInfo.totp_encrypt ? 1 : 0,
    }
    return data

};
export const getInfoDetailByUserId = async (user_id: any, totp_code: any, currentUserId: any) => {
    let dbUserInfo = await getOneAbkUserByUid(user_id);

    await checkAbkTotp(currentUserId, totp_code);
    ApiAssert.isNotEmpty(ErrorCode.USER_NOT_EXIST, dbUserInfo);


    let data = {
        userId: dbUserInfo.user_id,
        account: dbUserInfo.account,
        remark: dbUserInfo.remark,
        allow_ips: dbUserInfo.allow_ips,
        user_status: dbUserInfo.user_status,
        user_type: dbUserInfo.user_type,
        hasTotp: dbUserInfo && dbUserInfo.totp_encrypt ? 1 : 0,
        pwd: dbUserInfo.pwd,
        pwd_salt: dbUserInfo.pwd_salt,
        totp_encrypt: dbUserInfo ? dbUserInfo.totp_encrypt : "",
    }
    return data

};


export async function resetAbkTotp(userId: number | undefined) {
    if (userId) {
        let userInfo = await getOneAbkUserByUid(Number(userId));
        ApiAssert.isNotEmpty(ErrorCode.USER_NOT_EXIST, userInfo);
    }
    //生成新的密钥
    let totpEncrypt = Otplib.authenticator.generateSecret();
    let email = userId ? userId : 0 + '-' + totpEncrypt.slice(0, 3)
    let uri = 'otpauth://totp/' + email + '?secret=' + totpEncrypt + '&issuer=team888';
    return { uri: uri, totpEncrypt: totpEncrypt };
}

export async function userList(abkUserInfoPageVO: AbkUserInfoPageVO, currentUserId: any) {
    let page = Number(abkUserInfoPageVO.page);
    let size = Number(abkUserInfoPageVO.size);

    let res = await abkUserInfo.prototype.findAndCount({
        where: {},
        limit: size,
        offset: (page - 1) * size,
        order: [["user_id", "asc"]],
        raw: true
    });
    for (let item of res.rows) {
        delete item.pwd;
        delete item.pwd_salt;
        delete item.totp_encrypt;
        item.del_time = item.user_status == AbkUserInfoConst.USER_STATUS.DEL ? item.updatedAt : ""
    }
    return res;
}

export async function updateStatus(user_id: number, user_status: number, currentUserId: any) {
    await getOneAbkUserByUid(user_id)
    await updateAbkUserStatus(user_id, user_status);
    return 'success';
}

export async function addAbkUser(abkUserInfoVO: AbkUserInfoVO) {
    let insertData = {
        account: abkUserInfoVO.account,
        pwd: abkUserInfoVO.pwd,
        user_status: 0,
        user_type: 0,
        remark: abkUserInfoVO.remark ? abkUserInfoVO.remark : "",
        totp_encrypt: abkUserInfoVO.totp_encrypt,
        allow_ips: abkUserInfoVO.allow_ips ? abkUserInfoVO.allow_ips : "",
    }
    await abkUserInfo.prototype.create(insertData);
    return 'success';
}



