import {
    madAdminOrmDB, coinAddress, coinType, mainUserAsset, dwdSpotAssetsApi,
    ormDB, coinTx, coinWithdraw, walletAssets,
    financeAccountCategory, financeAccount,
    financeAccountDailyRecord
} from "@madex/ex-ts-dao";
import BigNumber from "bignumber.js";
import { CommonParam } from "./abkCommonService";

let { logger, apiAssertUtils: ApiAssert, BigNumberUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, financeAccountDataUtils, tickerUtils } = require('@madex/ex-js-common');

let _ = require('lodash');


export async function userAddrList(page: any, size: any, user_id: any) {
    let where: any = {};
    if (user_id) {
        where["user_id"] = Number(user_id);
    }
    let res = await coinAddress.prototype.findAndCount({
        attributes: ['user_id', 'coin_id', 'coin_symbol', 'address', 'createdAt'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["createdAt", "desc"]],
        raw: true
    });
    if (res.rows.length) {
        let coinIds = res.rows.map(item => item.coin_id);
        let coinMap = await getCoinTypeMapByCoinOrAssetIds(coinIds);
        for (let item of res.rows) {
            item.chain_type = coinMap[item.coin_id].chain_type;
            item.coin_symbol = coinMap[item.coin_id].symbol;
        }
    }
    return res;
}

export async function assetDetails(page: any, size: any, user_id: number) {
    let res = await coinAddress.prototype.findAndCount({
        attributes: ['coin_id', 'coin_symbol', 'balance'],
        where: {
            user_id: user_id
        },
        limit: size,
        offset: (page - 1) * size,
        order: [["createdAt", "desc"]],
        raw: true
    });
    if (res.rows.length) {
        let coinIds = res.rows.map(item => item.coin_id);
        let coinMap = await getCoinTypeMapByCoinOrAssetIds(coinIds);
        for (let item of res.rows) {
            item.chain_type = coinMap[item.coin_id].chain_type;
            item.coin_symbol = coinMap[item.coin_id].symbol;
        }
    }
    return res;
}


export async function siteAssets(page: any, size: any) {

    let pageData = await coinType.prototype.findAndCount({
        attributes: [ormDB.literal('DISTINCT(asset_id) as asset_id')],
        where: {
            main_status: 2//已经提交到撮合
        },
        limit: size,
        offset: (page - 1) * size,
        order: [['asset_id', 'asc']],
        raw: true
    });
    if (!pageData.rows.length) {
        return pageData;
    }
    let asset_ids = pageData.rows.map(item => item.asset_id);

    let dbCoinList = await coinType.prototype.findAll({
        attributes: ['id', 'asset_id', 'symbol', 'chain_type'],
        where: {
            asset_id: asset_ids
        },
        order: [['asset_id', 'asc'], ['id', 'asc']],
        raw: true
    });

    let coinMap: any = {};
    for (let item of dbCoinList) {
        //同名币 asset_id 一样 id 不一样
        if (!coinMap[item.asset_id]) {
            coinMap[item.asset_id] = item
        }
    }


    let [mainAssetsMap, tradeAssetsMap, { hotWalletMap, coldWalletMap, totalWalletMap }] = await Promise.all([getSiteMainAssetsGroupBy(asset_ids, 'asset_id'),
        getSiteTradeAssetsGroupBy(asset_ids, 'asset_id'),
        getSiteWalletAssetsGroupBy(asset_ids, 'asset_id')]);

    for (let item of pageData.rows) {
        let assetId = item.asset_id;
        item.symbol = coinMap[assetId] ? coinMap[assetId].symbol : "";
        item.chain_type = coinMap[assetId] ? coinMap[assetId].chain_type : "";
        item.site_assets = (mainAssetsMap[assetId] ? mainAssetsMap[assetId] : new BigNumber(0)).add(
            tradeAssetsMap[assetId] ? tradeAssetsMap[assetId] : new BigNumber(0)
        );
        item.total_wallet_assets = totalWalletMap[assetId] ? totalWalletMap[assetId] : new BigNumber(0);
        item.hot_wallet_assets = hotWalletMap[assetId] ? hotWalletMap[assetId] : new BigNumber(0);
        item.cold_wallet_assets = coldWalletMap[assetId] ? coldWalletMap[assetId] : new BigNumber(0);
    }
    return pageData;
}


export async function siteAssetsDetails(asset_id: number) {

    let symbolList = await coinType.prototype.findAll({
        attributes: ['symbol'],
        where: {
            asset_id
        },
        raw: true
    });
    //单个币种没必要查
    if (symbolList.length < 2) {
        return [];
    }

    let symbols = symbolList.map(item => item.symbol);

    let dbCoinList = await coinType.prototype.findAll({
        attributes: ['id', 'asset_id', 'symbol', 'chain_type'],
        where: {
            symbol: symbols
        },
        raw: true
    });

    let coinMap: any = {};
    for (let item of dbCoinList) {
        coinMap[item.symbol] = item
    }


    let [mainAssetsMap, tradeAssetsMap, { hotWalletMap, coldWalletMap, totalWalletMap }] = await Promise.all([getSiteMainAssetsGroupBy(symbols, 'symbol'),
        getSiteTradeAssetsGroupBy(symbols, 'asset_symbol'),
        getSiteWalletAssetsGroupBy(symbols, 'coin_symbol')]);

    for (let item of symbolList) {
        let symbol = item.symbol;
        item.chain_type = coinMap[symbol] ? coinMap[symbol].chain_type : "";
        item.site_assets = (mainAssetsMap[symbol] ? mainAssetsMap[symbol] : new BigNumber(0)).add(
            tradeAssetsMap[symbol] ? tradeAssetsMap[symbol] : new BigNumber(0)
        );
        item.total_wallet_assets = totalWalletMap[symbol] ? totalWalletMap[symbol] : new BigNumber(0);
        item.hot_wallet_assets = hotWalletMap[symbol] ? hotWalletMap[symbol] : new BigNumber(0);
        item.cold_wallet_assets = coldWalletMap[symbol] ? coldWalletMap[symbol] : new BigNumber(0);
    }
    return symbolList;
}

export async function depositList(page: any, size: any, uid_or_addr: any) {
    let where: any;
    if (!isNaN(Number(uid_or_addr))) {
        where = {
            [ormDB.Op.or]: {
                user_id: Number(uid_or_addr),
                to: String(uid_or_addr)
            }
        }
    }
    else {
        where = { to: String(uid_or_addr) }
    }

    let pageData = await coinTx.prototype.findAndCount({
        attributes: ['user_id', 'coin_id', 'coin_symbol', 'amount', 'to', 'status', 'updatedAt'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [['updatedAt', 'desc']],
        raw: true
    });
    let coinIds = pageData.rows.map(item => item.coin_id);
    let coinTypeMap = await getCoinTypeMapByCoinOrAssetIds(coinIds, []);
    for (let item of pageData.rows) {
        item.chain_type = coinTypeMap[item.coin_id] ? coinTypeMap[item.coin_id].chain_type : ""
    }
    return pageData
}

export async function withdrawList(page: any, size: any, uid_or_addr: any) {
    let where: any;
    if (!isNaN(Number(uid_or_addr))) {
        where = {
            [ormDB.Op.or]: {
                user_id: Number(uid_or_addr),
                to_address: String(uid_or_addr)
            }
        }
    }
    else {
        where = { to_address: String(uid_or_addr) }
    }

    let pageData = await coinWithdraw.prototype.findAndCount({
        attributes: ['user_id', 'coin_id', 'coin_symbol', 'amount', 'to_address', 'status', 'updatedAt'],
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [['updatedAt', 'desc']],
        raw: true
    });
    let coinIds = pageData.rows.map(item => item.coin_id);
    let coinTypeMap = await getCoinTypeMapByCoinOrAssetIds(coinIds, []);
    for (let item of pageData.rows) {
        item.chain_type = coinTypeMap[item.coin_id] ? coinTypeMap[item.coin_id].chain_type : ""
    }
    return pageData
}

export async function fundingTotalRecords() {

    //分类数量
    let category_count: any = 0;
    //总初始资金
    let total_init_asset: any = new BigNumber(0);
    //总余额
    let total_balance: any = new BigNumber(0);
    //总差值
    let total_sub_value: any = new BigNumber(0);
    //分类集合
    let category_list: any = [];

    //账户
    let accountList = await financeAccountDataUtils.getAccountInfoList();
    if (!accountList.length) {
        return {
            category_count,
            total_init_asset,
            total_balance,
            total_sub_value,
            category_list
        };
    }
    let dataList = await financeAccountDataUtils.getFundingDataByAccounts(accountList);
    if (dataList.length) {
        let cMap: any = {};
        //查询所有分类
        let cList = await financeAccountCategory.prototype.findAll({
            attributes: ['id', 'name'],
            where: {},
            raw: true
        });
        for (let item of cList) {
            cMap[item.id] = item.name;
        }
        //按类别分组
        let groupMap = _.groupBy(dataList, 'category_id');
        let keys = _.keys(groupMap);
        category_count = keys.length;

        for (let category_id of keys) {
            //一个分类下所有账户的数据
            let oneCategoryList = groupMap[category_id];
            let oneInitAsset = _.sumBy(oneCategoryList, (item) => new BigNumber(item.init_amount_usdt).toNumber());
            let oneBalance = _.sumBy(oneCategoryList, (item) => new BigNumber(item.balance_usdt).toNumber());
            let oneSubValue = new BigNumber(oneBalance).sub(new BigNumber(oneInitAsset));
            let item: any = {
                category_id: category_id,
                category_name: cMap[category_id] ? cMap[category_id] : "",
                init_asset: oneInitAsset,
                balance: oneBalance,
                sub_value: oneSubValue,
            }
            total_init_asset = total_init_asset.add(oneInitAsset);
            total_balance = total_balance.add(oneBalance);
            total_sub_value = total_sub_value.add(oneSubValue);
            category_list.push(item);
        }
    }
    return {
        category_count,
        total_init_asset,
        total_balance,
        total_sub_value,
        category_list
    };
}

export async function fundingDailyRecords(commonParam: CommonParam) {
    let res: any;
    let accountMap: any = {};
    let where: any = {};
    if (commonParam.category_id) {
        where["category_id"] = Number(commonParam.category_id);
    }
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: commonParam.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: commonParam.to_time },
        }];

    if (!commonParam.category_id) {//总体账户
        res = await financeAccountDailyRecord.prototype.findAndCount({
            attributes: ['record_date', ormDB.literal('sum(init_amount_usdt) as init_amount_usdt'),
                ormDB.literal('sum(balance_usdt) as balance_usdt')],
            where: where,
            limit: commonParam.size,
            offset: (Number(commonParam.page) - 1) * Number(commonParam.size),
            group: ['record_date'],
            order: [["record_date", "desc"]],
            raw: true
        });

    }
    else {//某个分类下的账户
        res = await financeAccountDailyRecord.prototype.findAndCount({
            attributes: ['record_date', 'account_id', ormDB.literal('sum(init_amount_usdt) as init_amount_usdt'),
                ormDB.literal('sum(balance_usdt) as balance_usdt')],
            where: where,
            limit: commonParam.size,
            offset: (Number(commonParam.page) - 1) * Number(commonParam.size),
            group: ['record_date', 'account_id'],
            order: [["record_date", "desc"]],
            raw: true
        });
        let accountIds = res.rows.map(item => item.account_id);
        if (accountIds.length) {
            let dbAccounts = await financeAccount.prototype.findAll({
                attributes: ['id', 'account'],
                where: {
                    id: accountIds
                },
                raw: true
            });
            for (let item of dbAccounts) {
                accountMap[item.id] = item.account;
            }
        }

    }
    res.count = res.count.length;
    for (let item of res.rows) {
        item.account_name = item.account_id && accountMap[item.account_id] ? accountMap[item.account_id] : "";
        item.sub_value = new BigNumber(item.balance_usdt).sub(new BigNumber(item.init_amount_usdt));
    }

    return res;

}


export async function autoCheckList() {

    //账号数量
    let account_count: any = 0;
    //总初始资金
    let total_init_asset: any = new BigNumber(0);
    //总余额
    let total_balance: any = new BigNumber(0);
    //总变化量
    let total_change_value: any = new BigNumber(0);
    //分类集合
    let account_data_list: any = [];

    //账户
    let accountList = await financeAccountDataUtils.getAccountInfoList();
    if (!accountList.length) {
        return {
            account_count,
            total_init_asset,
            total_balance,
            total_change_value,
            account_data_list
        };
    }
    let dataList = await financeAccountDataUtils.getMultiCoinFundingDataByAccounts(accountList);
    if (dataList.length) {
        let aMap: any = {};
        for (let item of accountList) {
            aMap[item.id] = item;
        }
        //按账户分组
        let groupMap = _.groupBy(dataList, 'account_id');
        let keys = _.keys(groupMap);
        account_count = keys.length;

        let tickerMap: any = {};

        for (let account_id of keys) {
            //一个账户的数据
            /*[{
                init_amount_map:{
                "USDT":{
                    "account_id":1,
                    "symbol":"USDT",
                    "amount":50,
                    "aomunt_usdt":50
                         }
                                },
                category_id:1,
                account_id:1,
                multi_coin_map:{
                    "BTC":1,
                    "ETH":2,
                }
            }]*/
            let oneAccountList = groupMap[account_id];

            for (let oneAccountOneCoinData of oneAccountList) {
                let initAmountMap = oneAccountOneCoinData.init_amount_map;
                let multiCoinMap = oneAccountOneCoinData.multi_coin_map;
                let initSymbols = _.keys(initAmountMap);
                let balanceSymbols = _.keys(multiCoinMap);
                let symbolsArray = _.union(initSymbols, balanceSymbols);
                //有账户 没有 初始资金 和 余额的情况
                if (!symbolsArray.length) {
                    let listOne = {
                        account_id: account_id,
                        account_name: aMap[account_id] ? aMap[account_id].account : "",
                        remark: aMap[account_id] ? aMap[account_id].remark : "",
                        symbol: "--",
                        init: new BigNumber(0),
                        init_usdt: 0,
                        balance: new BigNumber(0),
                        balance_usdt: new BigNumber(0),
                        change: new BigNumber(0),
                        change_usdt: new BigNumber(0)
                    }
                    account_data_list.push(listOne);
                }
                else {
                    for (let oneSymbol of symbolsArray) {
                        let symbol_usdt_rate = tickerMap[oneSymbol] ? tickerMap[oneSymbol] : await tickerUtils.rateCoin2USDT(oneSymbol);
                        let init = initAmountMap[oneSymbol] ? initAmountMap[oneSymbol].amount : 0;
                        let init_usdt = initAmountMap[oneSymbol] ? initAmountMap[oneSymbol].amount_usdt : 0;
                        let balance = multiCoinMap[oneSymbol] ? multiCoinMap[oneSymbol] : 0;
                        let balance_usdt = new BigNumber(balance).mul(new BigNumber(symbol_usdt_rate));
                        let listOne = {
                            account_id: account_id,
                            account_name: aMap[account_id] ? aMap[account_id].account : "",
                            remark: aMap[account_id] ? aMap[account_id].remark : "",
                            symbol: oneSymbol,
                            init: init,
                            init_usdt: init_usdt,
                            balance: balance,
                            balance_usdt: balance_usdt,
                            change: new BigNumber(balance).sub(new BigNumber(init)),
                            change_usdt: new BigNumber(balance_usdt).sub(new BigNumber(init_usdt)),
                        }
                        account_data_list.push(listOne);
                        total_init_asset = total_init_asset.add(new BigNumber(init_usdt));
                        total_balance = total_balance.add(new BigNumber(balance_usdt));
                        total_change_value = total_change_value.add(new BigNumber(listOne.change_usdt));
                    }
                }
            }
        }
    }
    return {
        account_count,
        total_init_asset,
        total_balance,
        total_change_value,
        account_data_list
    };
}


async function getCoinTypeMapByCoinOrAssetIds(coin_ids: any[], asset_ids?: any[]) {
    let map: any = {};
    let where: any = {};
    let coin_id_flag = false;
    if (coin_ids.length) {
        where["id"] = coin_ids;
        coin_id_flag = true;
    }
    else {
        where["asset_id"] = asset_ids;
    }

    let dbList = await coinType.prototype.findAll({
        where: where,
        raw: true
    });

    for (let item of dbList) {
        if (coin_id_flag) {
            map[item.id] = item;
        }
        else {
            map[item.asset_id] = item;
        }
    }
    return map;
}

async function getSiteMainAssetsGroupBy(conditions: any[], group_by_field: string) {
    let map: any = {};
    let where: any = {};
    where[group_by_field] = conditions;
    let mainAssetsList = await mainUserAsset.prototype.findAll({
        attributes: [group_by_field, ormDB.literal('sum(balance) as balance'),
            ormDB.literal('sum(holds) as holds'),],
        where: where,
        group: [group_by_field, 'symbol'],
        raw: true
    });
    for (let item of mainAssetsList) {
        map[item[group_by_field]] = new BigNumber(item.balance).add(new BigNumber(item.holds));
    }
    return map;
}

async function getSiteTradeAssetsGroupBy(conditions: any[], group_by_field: string) {
    let map: any = {};
    let where: any = {};
    where[group_by_field] = conditions;
    let tradeAssetsList = await dwdSpotAssetsApi.prototype.findAll({
        attributes: [group_by_field, ormDB.literal('sum(balance) as balance'),
            ormDB.literal('sum(unreal_profit) as unreal_profit'),],
        where: where,
        group: [group_by_field],
        raw: true
    });
    for (let item of tradeAssetsList) {
        map[item[group_by_field]] = new BigNumber(item.balance).add(new BigNumber(item.unreal_profit));
    }
    return map;
}

async function getSiteWalletAssetsGroupBy(conditions: any[], group_by_field: string) {
    let where: any = {};
    where[group_by_field] = conditions;

    let walletAssetsList = await walletAssets.prototype.findAll({
        attributes: [group_by_field, 'type', ormDB.literal('sum(balance) as balance')],
        where: where,
        group: [group_by_field, 'type'],
        raw: true
    });
    let hotWalletMap: any = {};
    let coldWalletMap: any = {};
    let totalWalletMap: any = {};
    for (let item of walletAssetsList) {
        if (item.type == 3) {//冷钱包
            coldWalletMap[item[group_by_field]] = new BigNumber(item.balance);
        }
        else {//热钱包
            hotWalletMap[item[group_by_field]] = hotWalletMap[item[group_by_field]] ? hotWalletMap[item[group_by_field]].add(new BigNumber(item.balance)) : new BigNumber(item.balance);
        }
        totalWalletMap[item[group_by_field]] = totalWalletMap[item[group_by_field]] ? totalWalletMap[item[group_by_field]].add(new BigNumber(item.balance)) : new BigNumber(item.balance);
    }

    return {
        hotWalletMap, coldWalletMap, totalWalletMap
    }
}



