import * as ReqUtils from "./req-utils";
import { ErrorCode } from "../constant/errorCode";
import { aclUserInfo, aclUserDepartmentPosition, madAdminOrmDB } from "@madex/ex-ts-dao";

let { logger } = require("@madex/ex-js-public");
import { AclUserInfoConst } from "../constant/aclUserConstant";

let { apiAssertUtils: ApiAssert, BigNumberUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');
/**
 * 获取当前登陆用户信息
 * @param sessionId
 */
export const getCurrentUser = async function (sessionId: string) {
    let currentUser = await ReqUtils.getCookie(sessionId);
    return currentUser;
}
/**
 * 获取当前登陆用户userId
 * @param sessionId
 */
export const getCurrentUserId = async function (sessionId: string) {
    let currentUser = await ReqUtils.getCookie(sessionId);
    return currentUser.userId;
}

/**
 * 判断是否是管理员 通过 UID
 * @param user_id
 */
export const isAdminUserByUid = async function (user_id: number) {
    let dbInfo = await getOneAclUserByUid(user_id);
    if (dbInfo.user_type != AclUserInfoConst.USER_TYPE.ADMIN) {
        throw ErrorCode.NO_PERMISSION
    }
}

/**
 * 判断是否是超管 通过 UID
 * @param user_id
 */
export const isSuperAdminUserByUid = async function (user_id: number) {
    let dbInfo = await getOneAclUserByUid(user_id);
    if (dbInfo.user_type != AclUserInfoConst.USER_TYPE.SUPER_ADMIN) {
        throw ErrorCode.NO_PERMISSION
    }
}

/**
 * 判断是否是管理员 通过 session_id
 * @param sessionId
 */
export const isAdminUserBySessionId = async function (sessionId: string) {
    let currentUserId = await getCurrentUserId(sessionId);
    let dbInfo = await getOneAclUserByUid(currentUserId);
    if (dbInfo.user_type != AclUserInfoConst.USER_TYPE.ADMIN) {
        throw ErrorCode.NO_PERMISSION
    }
}

/**
 * 判断是否是超管 通过 session_id
 * @param sessionId
 */
export const isSuperAdminUserBySessionId = async function (sessionId: string) {
    let currentUserId = await getCurrentUserId(sessionId);
    let dbInfo = await getOneAclUserByUid(currentUserId);
    if (dbInfo.user_type != AclUserInfoConst.USER_TYPE.SUPER_ADMIN) {
        throw ErrorCode.NO_PERMISSION
    }
}
/**
 * 通过 user_id 查询用户
 * @param user_id
 */
export const getOneAclUserByUid = async function (user_id: number) {
    if (!user_id) {
        logger.error('aclUserUtils.getOneAclUserByUid.error:' + 'user_id is null');
        throw ErrorCode.PARAM_MISS
    }
    let dbInfo = await aclUserInfo.prototype.findOne({
        where: {
            user_id: user_id,
        },
        raw: true
    });
    if (!dbInfo) {
        logger.error('aclUserUtils.getOneAclUserByUid.error:' + 'dbInfo is null');
        throw ErrorCode.USER_NOT_EXIST
    }
    return dbInfo;
}
/**
 * 通过 account 查询用户
 * @param account
 */
export const getOneAclUserByAccount = async function (account: string) {
    if (!account) {
        logger.error('aclUserUtils.getOneAclUserByAccount.error:' + 'account is null');
        throw ErrorCode.PARAM_MISS
    }
    let dbInfo = await aclUserInfo.prototype.findOne({
        where: {
            account: account,
        },
        raw: true
    });
    if (!dbInfo) {
        logger.error('aclUserUtils.getOneAclUserByAccount.error:' + 'dbInfo is null');
        throw ErrorCode.USER_NOT_EXIST
    }
    return dbInfo;
}

export const checkTotp = async function (user_id: number, totp_code: string) {

    //获取谷歌密钥并验证
    let dbUserInfo = await getOneAclUserByUid(user_id);
    ApiAssert.isTrue(ErrorCode.UNBOUND_TOTP, dbUserInfo && dbUserInfo.totp_encrypt !== '');
    await AuthCommon.totpCheckSync(totp_code, dbUserInfo.totp_encrypt)

    //判断是否已经使用过
    let latestVerifiedKey = "bastard.totp.used.user." + user_id
    let latestUsed = RedisClient.getSync(latestVerifiedKey)
    ApiAssert.isFalse(ErrorCode.TOTP_CODE_USED, totp_code === latestUsed)
    await RedisClient.writeSync(latestVerifiedKey, totp_code, 60 * 60)
    return 'success';
}


export const getDepartmentPositionByUid = async function (user_id: number) {
    if (!user_id) {
        logger.error('aclUserUtils.getDepartmentPositionByUid.error:' + 'user_id is null');
        throw ErrorCode.PARAM_MISS
    }

    let dbInfo = await aclUserDepartmentPosition.prototype.findOne({
        where: {
            user_id: user_id
        },
        raw: true
    });
    return {
        department_id: dbInfo ? dbInfo.department_id : "",
        position_id: dbInfo ? dbInfo.position_id : ""
    }
}

export const getDepartmentPositionByUids = async function (user_ids: number[]) {
    if (!user_ids || !user_ids.length) {
        logger.error('aclUserUtils.getDepartmentPositionByUids.error:' + 'user_ids is null');
        throw ErrorCode.PARAM_MISS
    }
    let resMap = {};
    let dbInfoList = await aclUserDepartmentPosition.prototype.findAll({
        where: {
            user_id: { [madAdminOrmDB.Op.in]: user_ids }
        },
        raw: true
    });

    if (dbInfoList.length) {
        for (let item of dbInfoList) {
            resMap[item.user_id] = {
                department_id: item ? item.department_id : "",
                position_id: item ? item.position_id : ""
            }
        }
    }
    return resMap
}