import * as aclUserService from "../service/aclUser.service";
import { AclUserInfoVO, AclUserInfoPageVO } from "../service/aclUser.service";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { checkTotp, getCurrentUserId, } from "../../../utils/aclUserUtils";

/**
 * 分页查询用户列表
 * @param req
 * @param infoVO
 */
export const list = async (req: any, aclUserInfoPageVO: AclUserInfoPageVO) => {
    let func_name = "aclUserCtrl.list";
    let cmd = req.path;
    try {
        aclUserInfoPageVO.page = Optional.opt(aclUserInfoPageVO, 'page', 1);
        aclUserInfoPageVO.size = Optional.opt(aclUserInfoPageVO, 'size', 20);
        let res = await aclUserService.list(aclUserInfoPageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
/**
 * 添加用户信息
 * @param req
 * @param infoVO
 */
export const add = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "aclUserCtrl.add";
    let cmd = req.path;
    try {
        await preCheck(aclUserInfoVO);
        let res = await aclUserService.add(aclUserInfoVO, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
/**
 * 修改用户信息
 * @param req
 * @param infoVO
 */
export const update = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "aclUserCtrl.update";
    let cmd = req.path;
    try {
        await updatePreCheck(aclUserInfoVO);
        let res = await aclUserService.update(aclUserInfoVO, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 修改用户状态（目前没用，需要启用的话可能需要调整逻辑，增加日志）
 * @param req
 * @param infoVO
 */
export const updateStatus = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "aclUserCtrl.updateStatus";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.user_id);
        let res = await aclUserService.updateStatus(aclUserInfoVO, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 重置密码 （目前没用，需要启用的话可能需要调整逻辑，增加日志）
 * @param req
 * @param infoVO
 */
export const resetPwd = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "aclUserCtrl.resetPwd";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.user_id);
        let res = await aclUserService.resetPwd(aclUserInfoVO, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 校验谷歌验证码
 * @param req
 * @param infoVO
 */
export const checkTotpCode = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "aclUserCtrl.checkTotpCode";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totpCode);
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await checkTotp(currentUserId, aclUserInfoVO.totpCode);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 校验
 * @param infoVO
 */
async function preCheck(aclUserInfoVO: AclUserInfoVO) {
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.remark);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.account);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.department_id);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.position_id);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.role_ids);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totp_encrypt);
    if (aclUserInfoVO.pwd && (aclUserInfoVO.pwd.length < 8 || aclUserInfoVO.pwd.length > 30)) {
        throw ErrorCode.PWD_ILLEGAL;
    }
}

async function updatePreCheck(aclUserInfoVO: AclUserInfoVO) {
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.user_id);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.remark);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.department_id);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.position_id);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.role_ids);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totp_encrypt);
    if (aclUserInfoVO.pwd && (aclUserInfoVO.pwd.length < 8 || aclUserInfoVO.pwd.length > 30)) {
        throw ErrorCode.PWD_ILLEGAL;
    }
}


/**
 * 分页查询用户操作日志列表
 * @param req
 * @param infoVO
 */
export const optLogList = async (req: any, aclUserInfoPageVO: AclUserInfoPageVO) => {
    let func_name = "aclUserCtrl.optLogList";
    let cmd = req.path;
    try {
        aclUserInfoPageVO.page = Optional.opt(aclUserInfoPageVO, 'page', 1);
        aclUserInfoPageVO.size = Optional.opt(aclUserInfoPageVO, 'size', 20);
        let res = await aclUserService.optLogList(aclUserInfoPageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
