// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { madAdminOrmDB, aclUserInfo, ormDB, aclUserDepartmentPosition, aclUserRole, aclUserOptLog } from "@madex/ex-ts-dao";
import { AclUserInfoConst } from "../../../constant/aclUserConstant";
import { CryptUtils } from "../../../utils/crypt-utils";
import { ErrorCode } from "../../../constant/errorCode";
import { getDepartmentPositionByUids, getOneAclUserByAccount, getOneAclUserByUid } from "../../../utils/aclUserUtils";
import { changeUserRoleWithTx, getUserRoleListByUids } from "./aclRoleAuth.service";
import { _deleteAllSessionByUserId } from "./userAuthConfig.service";
import { addOptLog } from "./userOptLog.service";

let { logger } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');

let _ = require('lodash');

export interface AclUserInfoVO {
    user_id?: number;

    account?: string | any;

    phone?: string | any;

    email?: string | any;

    remark?: string | any;//姓名

    pwd?: string | any;

    pwd_salt?: string;

    user_status?: number;

    del_flag?: number;

    totp_encrypt?: string;

    allow_ips?: string;

    department_id?: string;

    position_id?: string;

    role_ids?: string;

    createdAt?: Date | any;

    updatedAt?: Date | any;

    totpCode?: any,

    originPwd?: any,

    newPwd?: any,


}

export interface AclUserInfoPageVO extends AclUserInfoVO {
    page?: number,

    size?: number

    condition?: string

    from_time?: Date | any;

    to_time?: Date | any;
}

export const list = async (aclUserInfoPageVO: AclUserInfoPageVO) => {
    let where = {}
    if (!aclUserInfoPageVO.account) {
        where = {};
    }
    else if (!isNaN(Number(aclUserInfoPageVO.account))) {
        where = {
            [madAdminOrmDB.Op.or]: {
                user_id: Number(aclUserInfoPageVO.account),
                account: { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.account}%` },
                email: { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.account}%` },
            }
        }
    }
    else {
        where = {
            [madAdminOrmDB.Op.or]: {
                account: { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.account}%` },
                email: { [madAdminOrmDB.Op.like]: `${aclUserInfoPageVO.account}%` },
            }
        }
    }


    let page = Number(aclUserInfoPageVO.page);
    let size = Number(aclUserInfoPageVO.size);

    let res = await aclUserInfo.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["user_id", "asc"]],
        raw: true
    });
    await dealReturnData(res.rows);
    return res;
};
export const add = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    let dbInfo = await aclUserInfo.prototype.findOne({
        where: {
            account: aclUserInfoVO.account,
        },
        raw: true
    });

    if (dbInfo) {
        throw ErrorCode.USER_EXIST;
    }
    let insertInfo = {
        remark: aclUserInfoVO.remark,
        account: aclUserInfoVO.account,
        user_status: aclUserInfoVO.user_status ? aclUserInfoVO.user_status : 0,
        pwd_salt: CryptUtils.salt(),
        totp_encrypt: aclUserInfoVO.totp_encrypt,
        createdAt: new Date(),
        updatedAt: new Date(),
    };
    if (aclUserInfoVO.email) {
        insertInfo['email'] = aclUserInfoVO.email
    }
    if (aclUserInfoVO.phone) {
        insertInfo['phone'] = aclUserInfoVO.phone
    }
    if (aclUserInfoVO.allow_ips) {
        insertInfo['allow_ips'] = aclUserInfoVO.allow_ips
    }
    if (aclUserInfoVO.pwd) {
        insertInfo['pwd'] = aclUserInfoVO.pwd
    }
    else {
        insertInfo['pwd'] = CryptUtils.defPwd();
    }

    let tx;
    let insertUser;
    try {
        tx = await madAdminOrmDB.transaction();
        //创建用户
        insertUser = await aclUserInfo.prototype.create(insertInfo, {
            transaction: tx
        });
        //保存部门和职位
        await aclUserDepartmentPosition.prototype.create({
            user_id: insertUser.user_id,
            department_id: aclUserInfoVO.department_id,
            position_id: aclUserInfoVO.position_id,
            createdAt: new Date(),
            updatedAt: new Date(),
        }, {
            transaction: tx
        });
        let userRoleList: any = [];
        let roleIds = String(aclUserInfoVO.role_ids).split(',');
        for (let roleId of roleIds) {
            let item = {
                role_id: roleId,
                user_id: insertUser.user_id,
                createdAt: new Date(),
                updatedAt: new Date(),
            }
            userRoleList.push(item);
        }
        //保存用户角色
        await aclUserRole.prototype.bulkCreate(userRoleList, {
            transaction: tx
        });

        await tx.commit();
    }
    catch (e) {
        logger.error('add user error:' + e);
        if (tx) {
            await tx.rollback();
        }
        throw e;
    }
    addOptLog(null, insertUser.user_id, '新增管理后台用户', "", JSON.stringify(aclUserInfoVO), '后台用户管理', session_id);

    return 'ok';
};

export const update = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {

    let dbInfo = await aclUserInfo.prototype.findOne({
        where: {
            user_id: aclUserInfoVO.user_id,
        },
        raw: true
    });

    if (!dbInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    let updateInfo = {
        updatedAt: new Date(),
    };
    if (aclUserInfoVO.remark != dbInfo.remark) {
        updateInfo['remark'] = aclUserInfoVO.remark
    }
    if (aclUserInfoVO.email != dbInfo.email) {
        updateInfo['email'] = aclUserInfoVO.email
    }
    if (aclUserInfoVO.phone != dbInfo.phone) {
        updateInfo['phone'] = aclUserInfoVO.phone
    }
    if (aclUserInfoVO.allow_ips != dbInfo.allow_ips) {
        updateInfo['allow_ips'] = aclUserInfoVO.allow_ips
    }
    if (aclUserInfoVO.pwd != dbInfo.pwd) {
        let encrypted = await AuthCommon.getPasswordEncrypt(aclUserInfoVO.pwd, dbInfo.pwd_salt);
        updateInfo['pwd'] = encrypted;
        updateInfo['pwd_status'] = 1;
    }
    if (aclUserInfoVO.user_status != dbInfo.user_status) {
        updateInfo['user_status'] = aclUserInfoVO.user_status
    }
    if (aclUserInfoVO.totp_encrypt != dbInfo.totp_encrypt) {
        updateInfo['totp_encrypt'] = aclUserInfoVO.totp_encrypt
    }


    let tx;
    try {
        tx = await madAdminOrmDB.transaction();
        //修改用户
        await aclUserInfo.prototype.update(updateInfo, {
            where: {
                user_id: Number(aclUserInfoVO.user_id),
            },
            transaction: tx
        });
        //保存部门和职位
        await aclUserDepartmentPosition.prototype.update({
            department_id: aclUserInfoVO.department_id,
            position_id: aclUserInfoVO.position_id,
            updatedAt: new Date(),
        }, {
            where: {
                user_id: Number(aclUserInfoVO.user_id),
            },
            transaction: tx
        });
        //修改用户角色
        await changeUserRoleWithTx(aclUserInfoVO.user_id, aclUserInfoVO.role_ids, tx);

        await tx.commit();
    }
    catch (e) {
        logger.error('update user error:' + e);
        if (tx) {
            await tx.rollback();
        }
        throw e;
    }

    //修改了谷歌 需要踢出登陆
    if (aclUserInfoVO.totp_encrypt != dbInfo.totp_encrypt) {
        _deleteAllSessionByUserId(Number(aclUserInfoVO.user_id));
    }
    addOptLog(null, Number(aclUserInfoVO.user_id), '修改管理后台用户', "", JSON.stringify(aclUserInfoVO), '后台用户管理', session_id);

    return 'ok';
};

export const updateStatus = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    let userId = Number(aclUserInfoVO.user_id);

    await getOneAclUserByUid(userId);

    let updateInfo = Object.create(null);
    updateInfo.status = aclUserInfoVO.user_status;
    updateInfo.updatedAt = new Date();

    await aclUserInfo.prototype.update(updateInfo, {
        where: {
            user_id: userId
        }
    });
    return 'ok';
};


export const resetPwd = async (aclUserInfoVO: AclUserInfoVO, session_id: any) => {
    let userId = Number(aclUserInfoVO.user_id);

    await getOneAclUserByUid(userId);

    let updateInfo = Object.create(null);
    updateInfo.pwd = CryptUtils.defPwd();
    updateInfo.pwd_status = AclUserInfoConst.PWD_STATUS.DEFAULT;
    updateInfo.user_status = AclUserInfoConst.USER_STATUS.NORMAL;
    updateInfo.updatedAt = new Date();

    await aclUserInfo.prototype.update(updateInfo, {
        where: {
            user_id: userId
        }
    });

    return 'ok';
};


export async function updateUserStatus(userId: any, userStatus: number) {
    try {
        await aclUserInfo.prototype.update({
            user_status: userStatus
        }, {
            where: {
                user_id: userId,
            }
        });
    }
    catch (e) {
        logger.error('aclUserService.updateUserStatus.error:' + e)
        throw e;
    }
}

async function dealReturnData(rows: any) {
    if (!rows.length) {
        return
    }

    let uids = rows.map(item => item.user_id);

    let [userRoleList, dpMap] = await Promise.all([getUserRoleListByUids(uids), getDepartmentPositionByUids(uids)]);

    let userRoleMap = _.groupBy(userRoleList, 'user_id');

    for (let oneItem of rows) {
        delete oneItem.pwd;
        delete oneItem.pwd_salt;
        let userId = oneItem.user_id;
        let roleList = userRoleMap[userId];
        let dbOne = dpMap[userId];
        oneItem.role_ids = roleList && roleList.length ? roleList.map(item => item.role_id).join(",") : ""
        oneItem.department_id = dbOne ? dbOne.department_id : ""
        oneItem.position_id = dbOne ? dbOne.position_id : ""
    }
}


export const optLogList = async (aclUserInfoPageVO: AclUserInfoPageVO) => {
    if (!aclUserInfoPageVO.condition) {
        throw ErrorCode.PARAM_MISS
    }
    let where = {}
    if (!isNaN(Number(aclUserInfoPageVO.condition))) {
        where = {
            user_id: Number(aclUserInfoPageVO.condition),
        }
    }
    else {
        where = {
            module: { [madAdminOrmDB.Op.like]: `%${aclUserInfoPageVO.condition}%` },
        }
    }
    if (aclUserInfoPageVO.from_time && aclUserInfoPageVO.to_time){
        where['createdAt'] = { [ormDB.Op.between]: [aclUserInfoPageVO.from_time, aclUserInfoPageVO.to_time] }
    }

    let page = Number(aclUserInfoPageVO.page);
    let size = Number(aclUserInfoPageVO.size);

    let res = await aclUserOptLog.prototype.findAndCount({
        where: where,
        limit: size,
        offset: (page - 1) * size,
        order: [["createdAt", "desc"]],
        raw: true
    });
    return res;
};



