import * as coinTypeApplyService from "../service/coinTypeApply.service";
import { CoinTypeApplyPageVO, CoinTypeApplyVO } from "../service/coinTypeApply.service";


let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert, datetimeUtils } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUser, getCurrentUserId } from "../../../utils/aclUserUtils";

let isIp = require('is-ip');
/**
 * 技术部-上币管理-上币申请-申请上新币列表
 * @param req
 * @param infoVO
 */
export const list = async (req: any, pageVO: CoinTypeApplyPageVO) => {
    let func_name = "coinTypeApply.control.list";
    try {
        pageVO.page = Optional.opt(pageVO, 'page', 1);
        pageVO.size = Optional.opt(pageVO, 'size', 20);
        let res = await coinTypeApplyService.list(pageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 技术部-上币管理-上币申请-已有币种列表
 * @param req
 * @param infoVO
 */
export const listed = async (req: any, pageVO: CoinTypeApplyPageVO) => {
    let func_name = "coinTypeApply.control.listed";
    try {
        pageVO.page = Optional.opt(pageVO, 'page', 1);
        pageVO.size = Optional.opt(pageVO, 'size', 20);
        let res = await coinTypeApplyService.listed(pageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 技术部-上币管理-上币申请-交易浏览器列表
 * @param req
 * @param infoVO
 */
export const exploreUrlList = async (req: any, pageVO: CoinTypeApplyPageVO) => {
    let func_name = "coinTypeApply.control.exploreUrlList";
    try {
        let res = await coinTypeApplyService.exploreUrlList();
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 技术部-上币管理-上币申请-申请上新币
 * @param req
 * @param authConfigVO
 */
export const apply = async (req: any, coinTypeApplyVO: CoinTypeApplyVO) => {
    let func_name = "coinTypeApply.control.apply";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUser = await getCurrentUser(req.cookies.session_id);
        await paramValid(coinTypeApplyVO);

        let res = await coinTypeApplyService.apply(coinTypeApplyVO, currentUser, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 技术部-上币管理-上币申请-编辑上新币
 * @param req
 * @param authConfigVO
 */
export const edit = async (req: any, coinTypeApplyVO: CoinTypeApplyVO) => {
    let func_name = "coinTypeApply.control.edit";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUser = await getCurrentUser(req.cookies.session_id);
        if (!coinTypeApplyVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        await paramValid(coinTypeApplyVO);
        let res = await coinTypeApplyService.edit(coinTypeApplyVO, currentUser, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 技术部-上币管理-上币申请-撤销申请上币
 * @param req
 * @param authConfigVO
 */
export const cancelSelf = async (req: any, coinTypeApplyVO: CoinTypeApplyVO) => {
    let func_name = "coinTypeApply.control.cancelSelf";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUser = await getCurrentUser(req.cookies.session_id);
        if (!coinTypeApplyVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await coinTypeApplyService.cancelSelf(coinTypeApplyVO.id, coinTypeApplyVO.reason, currentUser, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 技术部-上币管理-上币申请-驳回申请上币
 * @param req
 * @param authConfigVO
 */
export const rewrite = async (req: any, coinTypeApplyVO: CoinTypeApplyVO) => {
    let func_name = "coinTypeApply.control.rewrite";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUser = await getCurrentUser(req.cookies.session_id);
        if (!coinTypeApplyVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await coinTypeApplyService.rewrite(coinTypeApplyVO.id, coinTypeApplyVO.reason, currentUser, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 技术部-上币管理-上币申请-管理员取消申请上新币
 * @param req
 * @param authConfigVO
 */
export const cancel = async (req: any, coinTypeApplyVO: CoinTypeApplyVO) => {
    let func_name = "coinTypeApply.control.cancel";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUser = await getCurrentUser(req.cookies.session_id);
        if (!coinTypeApplyVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await coinTypeApplyService.cancel(coinTypeApplyVO.id, coinTypeApplyVO.reason, currentUser, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 技术部-上币管理-上币申请-审核申请上新币
 * @param req
 * @param authConfigVO
 */
export const review = async (req: any, coinTypeApplyVO: CoinTypeApplyVO) => {
    let func_name = "coinTypeApply.control.review";
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';
        let currentUser = await getCurrentUser(req.cookies.session_id);
        if (!coinTypeApplyVO.id) {
            throw ErrorCode.PARAM_MISS
        }
        let res = await coinTypeApplyService.review(coinTypeApplyVO.id, currentUser, ip);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


async function paramValid(coinTypeApplyVO: CoinTypeApplyVO) {
    if (!coinTypeApplyVO.symbol || !coinTypeApplyVO.name
        || !coinTypeApplyVO.original_decimals || !coinTypeApplyVO.valid_decimals
        || !coinTypeApplyVO.deposit_confirm_count || !coinTypeApplyVO.safe_confirm_count
        || !coinTypeApplyVO.withdraw_fee || !coinTypeApplyVO.withdraw_min || !coinTypeApplyVO.max_transfer_amount) {
        throw ErrorCode.PARAM_MISS;
    }
    if (!coinTypeApplyVO.is_erc20) {
        coinTypeApplyVO.is_erc20 = 0
    }
    let tm = datetimeUtils.add(new Date(), datetimeUtils.SECONED * 10);
    if (!coinTypeApplyVO.tm_enable_deposit
        || datetimeUtils.between(coinTypeApplyVO.tm_enable_deposit, tm) < 0) {
        throw ErrorCode.PARAM_MISS;
    }

    if (!coinTypeApplyVO.tm_enable_withdraw
        || datetimeUtils.between(coinTypeApplyVO.tm_enable_withdraw, tm) < 0) {
        throw ErrorCode.PARAM_MISS;
    }
}

