// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { i18nInfo, i18nInfoHistory, ormDB } from "@madex/ex-ts-dao";
import * as i18nService from "./i18n.service";
import { getLastVersionList, I18nInfoPageVO, I18nInfoVO } from "./i18n.service";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { ROBOT_KEYS, sendRobotMessage } from "../../../utils/robotUtils";

let { logger } = require('@madex/ex-js-public');

export interface I18nInfoLogVO extends I18nInfoPageVO {
    info_id?: number;

    version: string;
}


export async function list(i18nInfoLogVO: I18nInfoLogVO) {
    let condition = {
        version:i18nInfoLogVO.version
    };
    if (i18nInfoLogVO.platform) {
        condition['platform'] = Number(i18nInfoLogVO.platform);
    }
    if (i18nInfoLogVO.module) {
        condition['module'] = Number(i18nInfoLogVO.module);
    }
    if (i18nInfoLogVO.code) {
        condition['code'] = i18nInfoLogVO.code;
    }
    if (i18nInfoLogVO.zh_cn) {
        condition['zh_cn'] = { [ormDB.Op.like]: `${i18nInfoLogVO.zh_cn}%` };
    }

    let page = Number(i18nInfoLogVO.page);
    let size = Number(i18nInfoLogVO.size);
    let resList = await i18nInfoHistory.prototype.findAndCount({
        where: condition,
        limit: size,
        offset: (page - 1) * size,
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
}


export async function revert(i18nInfoLogVO: I18nInfoLogVO, currentUserId: any, ip: any) {
    let dbHistoryList = await i18nInfoHistory.prototype.findAll({
        where: {
            version: i18nInfoLogVO.version
        },
        order: [['info_id', 'asc']],
        raw: true,
    });

    let insertInfoList: any = [];
    let insertHistoryList: any = [];

    for (let item of dbHistoryList) {
        delete item.info_id;
        delete item.id;
        insertInfoList.push(item);
    }

    let currentVersionList = await getLastVersionList();

    let current_version = "";
    for (let item of currentVersionList) {
        current_version = item.version;
        let info_id = item.id;
        item.info_id = info_id;
        item.createdAt = item.updatedAt;
        delete item.id;
        insertHistoryList.push(item)
    }

    let tx;

    try {
        tx = await ormDB.transaction();
        if (insertInfoList.length) {
            //删除历史版本
            await i18nInfoHistory.prototype.destroy({
                where: {
                    version: i18nInfoLogVO.version
                },
                transaction: tx
            });
            //删除当前版本
            await i18nInfo.prototype.destroy({
                where: {},
                transaction: tx
            });
            //当前版本存入历史
            await i18nInfoHistory.prototype.bulkCreate(insertHistoryList, {
                transaction: tx
            });
            //恢复到传入的历史版本
            await i18nInfo.prototype.bulkCreate(insertInfoList);
        }
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('i18nInfo.add.error:' + e);
        throw e;
    }
    //管理后台操作日志
    addOptLog(currentUserId, 0, '国际化信息版本回退', ip, `当前版本:${current_version},回退到:${i18nInfoLogVO.version}`, '国际化管理');
    sendRobotMessage(ROBOT_KEYS.COMMON_KEY, `国际化信息版本回退:当前版本:${current_version},回退到:${i18nInfoLogVO.version}`);
    return 'success';
}