import * as aclRoleAuthService from "../service/aclRoleAuth.service";
import { AclAuthVO, AclAuthPageVO, AclRolePageVO, AclRoleVO } from "../service/aclRoleAuth.service";
import { getCurrentUserId, } from "../../../utils/aclUserUtils";
import { ErrorCode } from "../../../constant/errorCode";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert } = require('@madex/ex-js-public');

/**
 * 分页查询权限列表
 * @param req
 * @param infoVO
 */
export const authList = async (req: any, aclAuthPageVO: AclAuthPageVO) => {
    let func_name = "aclRoleAuthCtrl.authList";
    try {
        aclAuthPageVO.page = Optional.opt(aclAuthPageVO, 'page', 1);
        aclAuthPageVO.size = Optional.opt(aclAuthPageVO, 'size', 20);
        let res = await aclRoleAuthService.authList(aclAuthPageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 权限树
 * @param req
 * @param infoVO
 */
export const getAuthTree = async (req: any, aclAuthPageVO: AclAuthPageVO) => {
    let func_name = "aclRoleAuthCtrl.getAuthTree";
    let cmd = req.path;
    try {
        let res = await aclRoleAuthService.getAuthTree([]);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 查询权限
 * @param req
 * @param infoVO
 */
export const getAuthByUser = async (req: any, aclAuthPageVO: AclAuthPageVO) => {
    let func_name = "aclRoleAuthCtrl.getAuthByUser";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await aclRoleAuthService.getAuthByUser(currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 查询权限
 * @param req
 * @param infoVO
 */
export const getAuthByRole = async (req: any, aclAuthVO: AclAuthVO) => {
    let func_name = "aclRoleAuthCtrl.getAuthByRole";
    let cmd = req.path;
    try {
        let res = await aclRoleAuthService.getAuthByRole(aclAuthVO.id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
/**
 * 保存权限信息
 * @param req
 * @param infoVO
 */
export const saveAuth = async (req: any, aclAuthVO: AclAuthVO) => {
    let func_name = "aclUserCtrl.saveAuth";
    let cmd = req.path;
    try {
        await preCheckAuth(aclAuthVO);
        let res = await aclRoleAuthService.saveAuth(aclAuthVO, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 删除权限信息
 * @param req
 * @param infoVO
 */
export const delAuth = async (req: any, aclAuthVO: AclAuthVO) => {
    let func_name = "aclUserCtrl.delAuth";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclAuthVO.id);
        let res = await aclRoleAuthService.delAuth(aclAuthVO.id, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 分页查询角色列表
 * @param req
 * @param infoVO
 */
export const roleList = async (req: any, aclRolePageVO: AclRolePageVO) => {
    let func_name = "aclRoleAuthCtrl.roleList";
    let cmd = req.path;
    try {
        aclRolePageVO.page = Optional.opt(aclRolePageVO, 'page', 1);
        aclRolePageVO.size = Optional.opt(aclRolePageVO, 'size', 20);
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        aclRolePageVO.creator = currentUserId;
        let res = await aclRoleAuthService.roleList(aclRolePageVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};
/**
 * 查询角色列表
 * @param req
 * @param infoVO
 */
export const getAllRole = async (req: any, aclRolePageVO: AclRolePageVO) => {
    let func_name = "aclRoleAuthCtrl.getAllRole";
    let cmd = req.path;
    try {
        let res = await aclRoleAuthService.getAllRole();
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 查询角色
 * @param req
 * @param infoVO
 */
export const getRoleByUser = async (req: any, aclRolePageVO: AclRolePageVO) => {
    let func_name = "aclRoleAuthCtrl.getRoleByUser";
    let cmd = req.path;
    try {
        let userId = aclRolePageVO.user_id;
        let res = await aclRoleAuthService.getRoleByUser(userId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 保存角色
 * @param req
 * @param infoVO
 */
export const saveRole = async (req: any, aclRoleVO: AclRoleVO) => {
    let func_name = "aclRoleAuthCtrl.saveRole";
    let cmd = req.path;
    try {
        await preCheckRole(aclRoleVO);
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        aclRoleVO.creator = currentUserId;
        let res = await aclRoleAuthService.saveRole(aclRoleVO);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 删除角色
 * @param req
 * @param infoVO
 */
export const delRole = async (req: any, aclRoleVO: AclRoleVO) => {
    let func_name = "aclRoleAuthCtrl.delRole";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclRoleVO.id);
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await aclRoleAuthService.delRole(aclRoleVO.id, currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 修改角色权限
 * @param req
 * @param infoVO
 */
export const changeRoleAuth = async (req: any, aclAuthVO: AclAuthVO) => {
    let func_name = "aclRoleAuthCtrl.changeRoleAuth";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclAuthVO.id);
        let res = await aclRoleAuthService.changeRoleAuth(aclAuthVO.id, aclAuthVO.authIds, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};

/**
 * 修改用户角色
 * @param req
 * @param infoVO
 */
export const changeUserRole = async (req: any, aclRoleVO: AclRoleVO) => {
    let func_name = "aclRoleAuthCtrl.changeUserRole";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclRoleVO.user_id);
        let res = await aclRoleAuthService.changeUserRole(aclRoleVO.user_id, aclRoleVO.roleIds, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 校验
 * @param infoVO
 */
async function preCheckAuth(aclAuthVO: AclAuthVO) {
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclAuthVO.pid);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclAuthVO.name);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclAuthVO.url);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclAuthVO.type);
}


/**
 * 校验
 * @param infoVO
 */
async function preCheckRole(aclRoleVO: AclRoleVO) {
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclRoleVO.name);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclRoleVO.remark);
    ApiAssert.notNull(ErrorCode.PARAM_MISS, aclRoleVO.type);
}

