import { coinTypeApply, coinType, ormDB, systemTrigger } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { APPLY_STATUS } from "../../../constant/coinTypeApplyConst";
import { addCoin2Core } from "../../../utils/coreSystemUtils";


let _ = require('lodash');
let { logger, datetimeUtils } = require('@madex/ex-js-public');
import * as RobotUtil from "../../../utils/robotUtils";


export interface CoinTypeApplyVO {
    id?: number;

    symbol?: string;

    mixin_id?: string;

    name?: string;

    general_name?: string;

    chain_type?: string;

    issupport_memo?: number;

    forbid_info?: string;

    explor_url?: string;

    icon_url?: string;

    describe_url?: string;

    original_decimals?: number;

    valid_decimals?: number;

    is_erc20?: number;

    contract?: string;

    contract_father?: string;

    deposit_confirm_count?: number;

    safe_confirm_count?: number;

    withdraw_fee?: string;

    withdraw_min?: string;

    deposit_min?: string;

    tm_enable_deposit?: Date | any,

    tm_enable_withdraw?: Date | any,

    applyer?: number;

    checker?: number;

    status?: number;

    reason?: string;

    max_transfer_amount?: string;

    createdAt?: Date | any,

    updatedAt?: Date | any,

}


export interface CoinTypeApplyPageVO extends CoinTypeApplyVO {

    page?: number,

    size?: number,

}


export async function list(pageVO: CoinTypeApplyPageVO) {

    let resList = await coinTypeApply.prototype.findAndCount({
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [['status', 'asc'], ["id", "desc"]],
        raw: true
    });
    return resList;
}

export async function listed(pageVO: CoinTypeApplyPageVO) {
    let where = {};
    if (pageVO.symbol) {
        where['symbol'] = pageVO.symbol;
    }

    let resList = await coinType.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}

export async function exploreUrlList() {

    let dbList = await coinType.prototype.findAll({
        attributes: ['explor_url'],
        where: {
            explor_url: { [ormDB.Op.ne]: "" }
        },
        raw: true
    });
    let resList = dbList.map(item => item.explor_url);
    return resList;
}


export async function apply(coinTypeApplyVO: CoinTypeApplyVO, currentUser: any, ip: string | undefined) {


    let dbSymbol = await coinType.prototype.findOne({
        where: { symbol: coinTypeApplyVO.symbol },
        raw: true
    });
    if (dbSymbol) {
        throw ErrorCode.COIN_DATA_EXIST;
    }

    coinTypeApplyVO.applyer = currentUser.account;
    coinTypeApplyVO.status = APPLY_STATUS.WAIT_VIEW;
    coinTypeApplyVO.createdAt = new Date();
    coinTypeApplyVO.updatedAt = new Date();
    if (Number(coinTypeApplyVO.id) > 0) {//update
        let dbApply = await coinTypeApply.prototype.findOne({
            where: { id: coinTypeApplyVO.id },
            raw: true
        });
        if (!dbApply) {
            throw ErrorCode.DATA_NOT_EXIST;
        }
        if (dbApply.status != APPLY_STATUS.RE_WRITE) {
            throw ErrorCode.NOT_EDIT
        }
        await coinTypeApply.prototype.update(coinTypeApplyVO, {
            where: {
                id: Number(coinTypeApplyVO.id)
            }
        });
    }
    else {
        await coinTypeApply.prototype.create(coinTypeApplyVO);
    }
    sendMsg(coinTypeApplyVO.status, String(coinTypeApplyVO.symbol));
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '新增上币申请', ip, JSON.stringify(coinTypeApplyVO), '上币管理');
    return 'success';
}


export async function edit(coinTypeApplyVO: CoinTypeApplyVO, currentUser: any, ip: string | undefined) {

    let dbSymbol = await coinType.prototype.findOne({
        where: { symbol: coinTypeApplyVO.symbol },
        raw: true
    });
    if (dbSymbol) {
        throw ErrorCode.COIN_DATA_EXIST;
    }

    let dbApply = await coinTypeApply.prototype.findOne({
        where: { id: coinTypeApplyVO.id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    if (Number(coinTypeApplyVO.status) < APPLY_STATUS.CREATE_SYMBOL) {
        if (dbSymbol) {
            throw ErrorCode.COIN_DATA_EXIST;
        }
    }
    else if (dbSymbol) {
        throw ErrorCode.COIN_DATA_NOT_EXIST
    }
    coinTypeApplyVO.applyer = dbApply.applyer;
    coinTypeApplyVO.updatedAt = new Date();

    await coinTypeApply.prototype.update(coinTypeApplyVO, {
        where: {
            id: Number(coinTypeApplyVO.id)
        }
    });
    sendMsg(Number(coinTypeApplyVO.status), String(coinTypeApplyVO.symbol));
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '上币申请编辑', ip, JSON.stringify(coinTypeApplyVO), '上币管理');
    return 'success';
}

export async function cancelSelf(id: any, reason: any, currentUser: any, ip: string | undefined) {

    let dbApply = await coinTypeApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    if (dbApply.status >= APPLY_STATUS.PASS) {
        throw ErrorCode.NO_CANCEL;
    }
    await updateApply(Number(id), APPLY_STATUS.CANCEL, currentUser.account, String(reason), dbApply.symbol);
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '撤销上币申请', ip, JSON.stringify(dbApply), '上币管理');
    return 'success';
}

export async function rewrite(id: any, reason: any, currentUser: any, ip: string | undefined) {

    let dbApply = await coinTypeApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    await updateApply(Number(id), APPLY_STATUS.RE_WRITE, currentUser.account, String(reason), dbApply.symbol);
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '驳回上币申请', ip, JSON.stringify(dbApply), '上币管理');
    return 'success';
}

export async function cancel(id: any, reason: any, currentUser: any, ip: string | undefined) {

    let dbApply = await coinTypeApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    await updateApply(Number(id), APPLY_STATUS.CANCEL, currentUser.account, String(reason), dbApply.symbol);
    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '管理员取消上币申请', ip, JSON.stringify(dbApply), '上币管理');
    return 'success';
}

export async function review(id: any, currentUser: any, ip: string | undefined) {

    let dbApply = await coinTypeApply.prototype.findOne({
        where: { id: id },
        raw: true
    });
    if (!dbApply) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let reason = "";
    //1被驳回 2申请待审批 3审批通过 4币种创建完成 5充值定时器完成 6提现定时器完成 7增加币种到撮合完成 8取消
    let status = Number(dbApply.status)
    if (status == APPLY_STATUS.WAIT_VIEW) {//2-3
        reason = "审核通过";
        await updateApply(Number(id), APPLY_STATUS.PASS, currentUser.account, reason, dbApply.symbol);
    }
    else if (status == APPLY_STATUS.PASS) {//3-4
        let dbSymbol = await coinType.prototype.findOne({
            where: { symbol: dbApply.symbol },
            raw: true
        });
        if (dbSymbol) {
            throw ErrorCode.COIN_DATA_EXIST;
        }
        let exist = await coinType.prototype.findOne({
            where: {
                is_main:1,
                general_name:dbApply.general_name,
            },
            raw: true
        });
        let is_main = 0;
        let asset_id = 0;
        let main_status = 0;
        if (exist) {
            is_main = 0;  // 同名币
            asset_id = exist.asset_id;
            main_status = exist.main_status;
        }else {
            is_main = 1;
            asset_id = 0;
            main_status = 0;
        }

        let insertInfo = {
            symbol: dbApply.symbol,
            is_active: 1,
            asset_id:asset_id,
            is_main: is_main,
            main_status:main_status,
            name: dbApply.name,
            mixin_id:dbApply.mixin_id,
            issupport_memo:dbApply.issupport_memo,
            deposit_min:dbApply.deposit_min,
            general_name: dbApply.general_name,
            chain_type:dbApply.chain_type,
            original_decimals: dbApply.original_decimals,
            valid_decimals: dbApply.valid_decimals,
            is_erc20: dbApply.is_erc20,
            contract: dbApply.contract,
            contract_father: dbApply.contract_father,
            deposit_confirm_count: dbApply.deposit_confirm_count,
            safe_confirm_count: dbApply.safe_confirm_count,
            withdraw_fee: dbApply.withdraw_fee,
            withdraw_min: dbApply.withdraw_min,
            icon_url: dbApply.icon_url,
            explor_url: dbApply.explor_url,
            describe_url: dbApply.describe_url,
            forbid_info: dbApply.forbid_info,
            enable_deposit: 0,
            enable_withdraw: 0,
            createdAt: new Date(),
            updatedAt: new Date(),
        };
        await coinType.prototype.create(insertInfo);
        reason = "币种创建完成";
        await updateApply(Number(id), APPLY_STATUS.CREATE_SYMBOL, currentUser.account, reason, dbApply.symbol);
    }
    else if (status == APPLY_STATUS.CREATE_SYMBOL) {//4-5
        let dbSymbol = await coinType.prototype.findOne({
            where: { symbol: dbApply.symbol },
            raw: true
        });
        if (!dbSymbol) {
            throw ErrorCode.COIN_DATA_NOT_EXIST;
        }
        let tm = datetimeUtils.add(new Date(), datetimeUtils.SECONED);
        let tm_enable_deposit = dbApply.tm_enable_deposit;
        if (datetimeUtils.between(tm_enable_deposit, tm) < 0) {
            throw ErrorCode.DEPOSIT_WITHDRAW_TM_EXPIRE;
        }
        if (dbSymbol.enable_deposit === 0) {
            await systemTrigger.prototype.create({
                trigger_symbol: dbApply.symbol,
                trigger_type: 1,
                trigger_action: 1021,
                trigger_time: tm_enable_deposit,
                status: 0,
                createdAt: new Date(),
                updatedAt: new Date(),
            });
        }
        reason = "充值定时器完成";
        await updateApply(Number(id), APPLY_STATUS.TIMER_DEPOSIT, currentUser.account, reason, dbApply.symbol);
    }
    else if (status == APPLY_STATUS.TIMER_DEPOSIT) {//5-6
        let dbSymbol = await coinType.prototype.findOne({
            where: { symbol: dbApply.symbol },
            raw: true
        });
        if (!dbSymbol) {
            throw ErrorCode.COIN_DATA_NOT_EXIST;
        }
        let tm = datetimeUtils.add(new Date(), datetimeUtils.SECONED);
        let tm_enable_withdraw = dbApply.tm_enable_withdraw;
        if (datetimeUtils.between(tm_enable_withdraw, tm) < 0) {
            throw ErrorCode.DEPOSIT_WITHDRAW_TM_EXPIRE;
        }
        if (dbSymbol.enable_deposit === 0) {
            await systemTrigger.prototype.create({
                trigger_symbol: dbApply.symbol,
                trigger_type: 1,
                trigger_action: 1031,
                trigger_time: tm_enable_withdraw,
                status: 0,
                createdAt: new Date(),
                updatedAt: new Date(),
            });
        }
        reason = "提现定时器完成";
        await updateApply(Number(id), APPLY_STATUS.TIMER_WITHDRAW, currentUser.account, reason, dbApply.symbol);
    }
    else if (status == APPLY_STATUS.TIMER_WITHDRAW) {//6-7
        let dbSymbol = await coinType.prototype.findOne({
            where: { symbol: dbApply.symbol },
            raw: true
        });
        if (!dbSymbol) {
            throw ErrorCode.COIN_DATA_NOT_EXIST;
        }
        let res = await addCoin2Core(dbSymbol.symbol, dbSymbol.id);
        if (!res.is_success) {
            RobotUtil.sendRobotMessage(RobotUtil.ROBOT_KEYS.COMMON_KEY, res.err_msg);
            throw ErrorCode.ADD_PAIR_TO_CORE_ERR;
        }
        await coinType.prototype.update({ main_status: 1 }, {
            where: { id: dbSymbol.id }
        });
        reason = "增加币种到撮合完成";
        await updateApply(Number(id), APPLY_STATUS.TO_ADMIN, currentUser.account, reason, dbApply.symbol);
    }
    else {
        throw ErrorCode.CURRENT_STATUS_NOT_APPLY
    }

    //管理后台操作日志
    addOptLog(currentUser.userId, 0, '上币申请审核', ip, `msg:${reason},dbInfo:${JSON.stringify(dbApply)}`, '上币管理');
    return 'success';
}

async function updateApply(id: number, status: number, checker: string, reason: string, symbol: string) {
    let updateInfo = {
        status: status,
        checker: checker,
        reason: reason ? reason : "",
        updatedAt: new Date()
    }
    await coinTypeApply.prototype.update(updateInfo, {
        where: {
            id: id
        }
    });
    sendMsg(status, symbol);
}

async function sendMsg(status: number, symbol: string) {
    let content = "";
    if (status == APPLY_STATUS.RE_WRITE) {
        content = "申请上币,被驳回：" + symbol;
    }
    else if (status == APPLY_STATUS.WAIT_VIEW) {
        content = "申请上币：" + symbol;
    }
    else if (status == APPLY_STATUS.PASS) {
        content = "申请上币,审批通过：" + symbol + "，下一步币种创建";
    }
    else if (status == APPLY_STATUS.CREATE_SYMBOL) {
        content = "申请上币,币种创建完成：" + symbol + "，下一步充值定时器";
    }
    else if (status == APPLY_STATUS.TIMER_DEPOSIT) {
        content = "申请上币,充值定时器完成：" + symbol + "，下一步提现定时器";
    }
    else if (status == APPLY_STATUS.TIMER_WITHDRAW) {
        content = "申请上币,提现定时器完成：" + symbol + "，下一步增加币种到撮合";
    }
    else if (status == APPLY_STATUS.TO_ADMIN) {
        content = "申请上币,增加币种到撮合完成：" + symbol + "，流程结束";
    }
    else if (status == APPLY_STATUS.CANCEL) {
        content = "申请上币,被取消：" + symbol + "，流程结束";
    }

    if (content) {
        RobotUtil.sendRobotMessage(RobotUtil.ROBOT_KEYS.COMMON_KEY, content);
    }
}

