// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { madAdminOrmDB, aclUserInfo, aclUserAuthConfig } from "@madex/ex-ts-dao";
import { AclUserInfoConst } from "../../../constant/aclUserConstant";
import { AuthConfigConst } from "../../../constant/aclUserAuthConfigConstant";
import { CryptUtils } from "../../../utils/crypt-utils";
import { ErrorCode } from "../../../constant/errorCode";
import * as aclRoleAuthService from "../service/aclRoleAuth.service";
import * as aclUserService from "../service/aclUser.service";
import * as userAuthConfigService from "../service/userAuthConfig.service";
import { getOneAclUserByAccount, getOneAclUserByUid } from "../../../utils/aclUserUtils";
import { RedisVal } from "../../../constant/redis-val";
import Config from "../../../../config";
import * as userOptLogService from "./userOptLog.service";


const Otplib = require('otplib');
//绑定totp的验证中使用的缓存key的前缀
const BIND_TOTP_REDIS_KEY_PRE = "bastard.totp."


let { apiAssertUtils: ApiAssert, BigNumberUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');


export const getInfo = async (currentUserId: number | any, sessionId: string) => {
    let dbUserInfo = await getOneAclUserByUid(currentUserId);

    ApiAssert.isNotEmpty(ErrorCode.USER_NOT_EXIST, dbUserInfo);
    ApiAssert.isFalse(ErrorCode.ACCOUNT_LOCK, dbUserInfo.user_status === AclUserInfoConst.USER_STATUS.LOCK);
    ApiAssert.isFalse(ErrorCode.ACCOUNT_STOP, dbUserInfo.user_status === AclUserInfoConst.USER_STATUS.DEL);

    let { roleSet, authSet } = await aclRoleAuthService.getUserAcl(dbUserInfo.user_id);


    let data = {
        remark: dbUserInfo.remark,
        userId: dbUserInfo.user_id,
        account: dbUserInfo.account,
        userType: dbUserInfo.user_type,
        sessionId: sessionId,
        roleSet: roleSet,
        authSet: authSet,
    }
    return data

};


export async function login(account: any, pwd: any, s: string) {

    let userInfo = await getOneAclUserByAccount(account);
    ApiAssert.isNotEmpty(ErrorCode.ACCOUNT_OR_PWD_ERR, userInfo);

    ApiAssert.isFalse(ErrorCode.ACCOUNT_LOCK, userInfo.user_status === AclUserInfoConst.USER_STATUS.LOCK);
    ApiAssert.isFalse(ErrorCode.ACCOUNT_STOP, userInfo.user_status === AclUserInfoConst.USER_STATUS.DEL);
    ApiAssert.isFalse(ErrorCode.ACCOUNT_STOP, userInfo.user_status === 3);

    await _checkPwd(userInfo, pwd);

    let sessionId = CryptUtils.sessionId(userInfo.b_user_id);
    RedisClient.rpush(RedisVal.sessionListKey(userInfo.user_id), sessionId);

    let { roleSet, authSet } = await aclRoleAuthService.getUserAcl(userInfo.user_id);
    let cookies = {
        account: userInfo.account,
        userId: userInfo.user_id,
        userType: userInfo.user_type,
        // acl 相关
        roleSet: roleSet,
        authSet: authSet,
        // 添加逻辑
        //是否需要进行二次验证。0:不需要或者已通过验证;1:需要;
        needConfirm: 0,
    };
    let needBindTotp = 0
    let deadline = ''
    let hasTotp = 0
    //如果是管理员，则必须绑定
    if (Number(userInfo.user_type) === AclUserInfoConst.USER_TYPE.ADMIN) {
        cookies.needConfirm = 1
        needBindTotp = 1
    }

    //如果绑定了谷歌则必须校验, 增加属性 needConfirm = 1
    let authConfig = await userAuthConfigService.findByUserId(userInfo.user_id)
    if (authConfig) {
        if (Number(authConfig.is_locked) === AuthConfigConst.IS_LOCKED.TRUE) {
            throw ErrorCode.TOTP_UNBOUND_LOCKED;
        }
        if (authConfig.totp_encrypt && authConfig.totp_encrypt !== '' && Number(authConfig.force) === AuthConfigConst.FORCE.TRUE) {
            cookies.needConfirm = 1
            hasTotp = 1
        }
        if (Number(authConfig.force) === AuthConfigConst.FORCE.TRUE) {
            needBindTotp = 1
            deadline = authConfig.deadline
        }
    }

    await RedisClient.writeSync(sessionId, cookies, Config.LOGIN_EXPIRED);
    await _unlockPwd(userInfo.user_id);


    userOptLogService.addOptLog(userInfo.user_id, `user login`, userOptLogService.LogType.LOGIN, '');

    return {
        result: "success",
        sessionId: sessionId,
        userType: Number(userInfo.user_type),
        //是否已绑定谷歌。0:未绑定；1:已绑定
        hasTotp: hasTotp,
        //是否需要进行安全项绑定。0:不需要;1:需要;
        needBindTotp: needBindTotp,
        deadline: deadline,
    };
}


export async function deleteAllSessionByUserId(userId: number) {

    //获取该账户使用过的所有sessionId
    let sessionListKey = RedisVal.sessionListKey(userId)
    RedisClient.lrange(sessionListKey, 0, -1, async (err, reply) => {
        //删除所有sessionId
        if (!err && reply && reply.length >= 1) {
            await RedisClient.delSync(...reply)
        }
        //删除sessionList
        await RedisClient.delSync(sessionListKey)
    });
}


export async function loginConfirm(sessionId: any, userId: any, totpCode: any) {

    //判断是否在登录锁定中
    let cookies = await RedisClient.getSync(sessionId)
    ApiAssert.isTrue(ErrorCode.PARAM_MISS, Number(cookies.needConfirm) === 1);

    //获取谷歌密钥并验证
    let authInfo = await userAuthConfigService.findByUserId(userId)
    ApiAssert.isTrue(ErrorCode.UNBOUND_TOTP, authInfo && authInfo.totp_encrypt !== '');
    await AuthCommon.totpCheckSync(totpCode, authInfo.totp_encrypt)

    //判断是否已经使用过
    let latestVerifiedKey = "bastard.totp.used.user." + userId
    let latestUsed = RedisClient.getSync(latestVerifiedKey)
    ApiAssert.isFalse(ErrorCode.TOTP_CODE_USED, totpCode === latestUsed)
    await RedisClient.writeSync(latestVerifiedKey, totpCode, 60 * 60)

    //解除缓存中的锁定标识
    cookies.needConfirm = 0
    await RedisClient.writeSync(sessionId, cookies, Config.LOGIN_EXPIRED)
    return "success";
}

export async function updatePwd(userId: any, originPwd: any, newPwd: any) {

    let userInfo = await getOneAclUserByUid(userId);
    ApiAssert.isNotEmpty(ErrorCode.USER_NOT_EXIST, userInfo);
    ApiAssert.isFalse(ErrorCode.ACCOUNT_LOCK, userInfo.user_status === AclUserInfoConst.USER_STATUS.LOCK);
    ApiAssert.isFalse(ErrorCode.USER_NOT_EXIST, userInfo.user_status === AclUserInfoConst.USER_STATUS.DEL);

    await _checkPwd(userInfo, originPwd);

    let encrypted = await AuthCommon.getPasswordEncrypt(newPwd, userInfo.pwd_salt);
    let data = {
        pwd: encrypted,
        pwd_status: 1,
    }
    await aclUserInfo.prototype.update(data, {
        where: {
            user_id: userId
        }
    });
    userOptLogService.addOptLog(userInfo.user_id, `update self pwd`, userOptLogService.LogType.UPDATE, '');
    return 'success';
}

export async function bindTotpAsk(sessionId: any, userId: any) {

    let userInfo = await getOneAclUserByUid(userId);
    ApiAssert.isNotEmpty(ErrorCode.USER_NOT_EXIST, userInfo);

    //判断是否已完成绑定
    let authConfig = await userAuthConfigService.findByUserId(userId)
    ApiAssert.isTrue(ErrorCode.GOOGLE_HAS_BIND, !authConfig || authConfig.totp_encrypt === '');
    ApiAssert.isTrue(ErrorCode.TOTP_UNBOUND_LOCKED, !authConfig || authConfig.is_locked !== AuthConfigConst.IS_LOCKED.TRUE)

    //生成新的密钥
    let totpEncrypt = Otplib.authenticator.generateSecret();
    let email = userId + '-' + totpEncrypt.slice(0, 3)
    let cookieValue = { userId: userId, totpEncrypt: totpEncrypt }

    let uri = 'otpauth://totp/' + email + '?secret=' + totpEncrypt + '&issuer=team888';
    //保存到redis中
    await RedisClient.writeSync(BIND_TOTP_REDIS_KEY_PRE + sessionId, cookieValue, 15 * 60)
    return { uri: uri, totpEncrypt: totpEncrypt };
}


export async function bindTotpConfirm(sessionId: any, userId: any, totpCode: any) {

    let bindKey = BIND_TOTP_REDIS_KEY_PRE + sessionId
    let bindValue = await RedisClient.getSync(bindKey)
    if (!bindValue) {
        throw ErrorCode.TOTP_KEY_OVERSTAYED;
    }
    //验证动态码
    await AuthCommon.totpCheckSync(totpCode, bindValue.totpEncrypt)

    //保存密钥
    await _updateTotpConfig(userId, bindValue.totpEncrypt)

    //删除缓存
    await RedisClient.delSync(bindKey)

    //解除缓存中的锁定标识
    let cookies = await RedisClient.getSync(sessionId);
    if (cookies) {
        cookies.needConfirm = 0
        await RedisClient.writeSync(sessionId, cookies, Config.LOGIN_EXPIRED)
    }
    userOptLogService.addOptLog(null, `bind totp`, userOptLogService.LogType.TOTP, '', sessionId);
    return "success"
}

async function _checkPwd(userInfo: any, pwd: any) {

    let pwdSuc;
    if (userInfo.pwd_status === 0) {
        pwdSuc = userInfo.pwd == pwd;
    }
    else {
        let encrypted = await AuthCommon.getPasswordEncrypt(pwd, userInfo.pwd_salt);
        pwdSuc = encrypted == userInfo.pwd;
    }
    if (!pwdSuc) await _pwdError(userInfo.user_id);
}

async function _pwdError(userId: any) {
    let key = RedisVal.loginErrTimesKey(userId)
    let errorCount = await RedisClient.getMemSync(key);
    errorCount = BigNumberUtils.add(errorCount, 1);
    await RedisClient.writeSync(key, errorCount);

    if (errorCount && Number(errorCount) >= Config.LOGIN_ERROR_LIMIT) {
        await aclUserService.updateUserStatus(userId, AclUserInfoConst.USER_STATUS.LOCK);
        throw ErrorCode.ACCOUNT_LOCK;
    }
    throw ErrorCode.ACCOUNT_OR_PWD_ERR;
}

async function _unlockPwd(userId: any) {
    let key = RedisVal.loginErrTimesKey(userId);
    await RedisClient.delSync(key);
}

async function _updateTotpConfig(userId: number, totpEncrypt: any) {

    //获取当前配置
    let configExist = await userAuthConfigService.findByUserId(userId)
    if (configExist) {
        let data2Update = {
            totp_encrypt: totpEncrypt,
            updatedAt: new Date()
        }
        let condition = {
            where: {
                id: configExist.id
            },
            raw: true
        }
        await aclUserAuthConfig.prototype.update(data2Update, condition)
    }
    else {
        let now = new Date()
        let data2Add = {
            user_id: userId,
            totp_encrypt: totpEncrypt,
            is_locked: AuthConfigConst.IS_LOCKED.FALSE,
            force: AuthConfigConst.FORCE.TRUE,
            deadline: now,
            createdAt: now,
            updatedAt: now,
        }
        await aclUserAuthConfig.prototype.create(data2Add)
    }
}

/*
async function test() {
    let pwd = CryptUtils.defPwd();
    let pwd_salt = CryptUtils.salt();
    console.log(pwd)
    console.log(pwd_salt)
}

test()*/

