import {
    ormDB,
    usefulLink,
    commonUserFeeSetting,
    feeRateContractLog,
    feeRateBaseCoinContractLog,
    feeRateSpotLog,
    vipLevelFeeSetting,
    spotPairs, madSpotOrmDB, contractPairs, madContractOrmDB
} from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { COMMENT_KOL_FEE_SUBMIT, COMMENT_MAKER_FEE_SUBMIT, COMMENT_USER_FEE_SUBMIT, FEE_STATUS, FEE_TYPE } from "../../../constant/marketMakerConst";
import { SYMBOL_ALL } from "../../../constant/symbolConst";

import * as feeRateSpotLogService from "./feeRateSpotLog.service";
import * as feeRateContractLogService from "./feeRateContractLog.service";
import * as feeRateBaseCoinContractLogService from "./feeRateBaseCoinContractLog.service";
import { FEE_MODEL_SPOT_DEFAULT, FEE_RATE_LOG_STATUS } from "../../../constant/feeRateLogConst";
import { where } from "sequelize";
import { it } from "node:test";


let _ = require('lodash');
let { logger, datetimeUtils } = require('@madex/ex-js-public');
let { redisUtilsCommon: RedisClient, } = require('@madex/ex-js-common');


export interface CommonUserFeeVO {
    id?: number;

    user_id?: number | any;

    vip_level?: number;

    type?: number;

    pair?: string;

    maker_fee?: number,

    taker_fee?: number,

    begin_time?: Date | any,

    expire_time?: Date | any,

    maker_fee_later?: number,

    taker_fee_later?: number,

    status?: number,

    remark?: string;

    fee_log_ids?: string;

    create_time?: Date | any,

    update_time?: Date | any,

    level_id?: number;

    user_channel?: number;

    statis_period?: number;

    amount_require_usdt?: number;

    applicant?: string;


}


export interface CommonUserFeePageVO extends CommonUserFeeVO {

    page?: number,

    size?: number,

    export?: number,//是不全部数据  1 全部倒出  0 否

}


export async function list(pageVO: CommonUserFeePageVO) {
    let where = {};
    //即将过期 前端入参 = 5 单独处理 数据库没有此值
    if (pageVO.status) {
        if (pageVO.status == 5) {
            let now = new Date();
            let before = datetimeUtils.sub(now, datetimeUtils.DAY * 3);
            where['status'] = 2;
            where['expire_time'] = { [ormDB.Op.between]: [before, now] }
        }
        else {
            where['status'] = pageVO.status;
        }
    }

    if (pageVO.export) {
        let resList = await commonUserFeeSetting.prototype.findAll({
            where: where,
            order: [['id', 'desc']],
            raw: true
        });
        return resList;
    }
    if (pageVO.user_id) {
        where['user_id'] = pageVO.user_id;
    }
    if (pageVO.type) {
        where['type'] = pageVO.type;
    }


    let resList = await commonUserFeeSetting.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["id", "desc"]],
        raw: true
    });
    if (resList.rows.length) {
        for (let item of resList.rows) {
            let userId = item.user_id;
            let pair = item.pair;
            let userChannel = item.user_channel;
            let cacheData: any;
            if (userChannel == 2) {
                cacheData = await getProjectMakerTradeStat(userId, pair);
            }
            item.avg7 = cacheData ? cacheData.avg7 : 0;
            item.avg15 = cacheData ? cacheData.avg15 : 0;
            item.avgUsdt = cacheData ? cacheData.avgUsdt : 0;
        }
    }
    return resList;
}


export async function add(commonUserFeeVO: CommonUserFeeVO, currentUserId: any, ip: string | undefined) {

    let feeSettingItem = await checkAndBuildFeeSettingItem(commonUserFeeVO);
    let tx: any;
    try {
        tx = await ormDB.transaction();
        //新建
        let dbInfo = await commonUserFeeSetting.prototype.create(feeSettingItem, {
            transaction: tx
        });
        let pair = dbInfo.pair;
        let type = dbInfo.type;
        let now = new Date();
        let insertDbList: any;


        let rateLogs = await buildSubmitItems(pair, dbInfo, now, dbInfo.user_channel);

        //提交到log表
        if (type == FEE_TYPE.FEE_TYPE_SPOT) {
            insertDbList = await feeRateSpotLog.prototype.bulkCreate(rateLogs, {
                transaction: tx,
                returning: true
            });
        }
        else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {//币本位
            insertDbList = await feeRateBaseCoinContractLog.prototype.bulkCreate(rateLogs, {
                transaction: tx,
                returning: true
            });
        }
        else {//U本位
            insertDbList = await feeRateContractLog.prototype.bulkCreate(rateLogs, {
                transaction: tx,
                returning: true
            });

        }
        let idsList = insertDbList.map(item => item.id);
        let ids = idsList.length ? idsList.join(",") : "";

        await commonUserFeeSetting.prototype.update({
            fee_log_ids: ids,
            update_time: new Date()
        }, {
            where: {
                id: dbInfo.id,
            },
            transaction: tx
        });

        await tx.commit();

    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('commonUserFeeSetting.service.add.error:' + e);
        throw e;
    }
    //管理后台操作日志
    addOptLog(currentUserId, Number(commonUserFeeVO.user_id), '新增用户手续费', ip, JSON.stringify(commonUserFeeVO), 'VIP管理');
    return 'success';
}


export async function update(commonUserFeeVO: CommonUserFeeVO, currentUserId: any, ip: string | undefined) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            id: commonUserFeeVO.id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    let pair = dbInfo.pair;
    let type = dbInfo.type;

    if (dbInfo.user_id != commonUserFeeVO.user_id || type != commonUserFeeVO.type || commonUserFeeVO.pair != pair) {
        throw ErrorCode.UID_TYPE_NOT_UPDATE;
    }
    if (dbInfo.status == 3 || dbInfo.status == 4) {
        throw ErrorCode.FEE_EX_OR_DEL;
    }
    let tx;
    try {
        tx = await ormDB.transaction();
        let feeLogIds: any;
        //这四项修改需要 重新写入 rate_log
        if (Number(commonUserFeeVO.maker_fee) != Number(dbInfo.maker_fee) || Number(commonUserFeeVO.taker_fee) != Number(dbInfo.taker_fee)
            || datetimeUtils.trim(commonUserFeeVO.begin_time, 's').getTime() != datetimeUtils.trim(dbInfo.begin_time, 's').getTime()
            || datetimeUtils.trim(commonUserFeeVO.expire_time, 's').getTime() != datetimeUtils.trim(dbInfo.expire_time, 's').getTime()) {
            let insertDbList: any;
            let rateLogs = await buildSubmitItems(pair, commonUserFeeVO, new Date(), Number(commonUserFeeVO.user_channel));
            //提交到log表
            if (type == FEE_TYPE.FEE_TYPE_SPOT) {
                insertDbList = await feeRateSpotLog.prototype.bulkCreate(rateLogs, {
                    transaction: tx,
                    returning: true
                });
            }
            else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {//币本位
                insertDbList = await feeRateBaseCoinContractLog.prototype.bulkCreate(rateLogs, {
                    transaction: tx,
                    returning: true
                });
            }
            else {//U本位
                insertDbList = await feeRateContractLog.prototype.bulkCreate(rateLogs, {
                    transaction: tx,
                    returning: true
                });

            }
            let idsList = insertDbList.map(item => item.id);
            feeLogIds = idsList.length ? idsList.join(",") : dbInfo.fee_log_ids;
        }


        await commonUserFeeSetting.prototype.update({
            maker_fee: Number(commonUserFeeVO.maker_fee),
            taker_fee: Number(commonUserFeeVO.taker_fee),
            begin_time: commonUserFeeVO.begin_time,
            expire_time: commonUserFeeVO.expire_time,
            remark: commonUserFeeVO.remark,
            user_channel: commonUserFeeVO.user_channel,
            statis_period: commonUserFeeVO.statis_period,
            amount_require_usdt: commonUserFeeVO.amount_require_usdt,
            applicant: commonUserFeeVO.applicant,
            update_time: new Date(),
            fee_log_ids: feeLogIds,
        }, {
            where: {
                id: Number(commonUserFeeVO.id)
            },
            transaction: tx
        });
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('commonUserFeeSetting.service.update.error:' + e);
        throw e;
    }


    //管理后台操作日志
    addOptLog(currentUserId, Number(commonUserFeeVO.user_id), '修改用户手续费', ip, JSON.stringify(commonUserFeeVO), 'VIP管理');
    return 'success';
}


export async function del(id: number, currentUserId: any, ip: string | undefined) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    let status = dbInfo.status;
    let type = dbInfo.type;
    if (status == FEE_STATUS.STATUS_FEE_SETTING_EFFECTED) {
        throw ErrorCode.FEE_USED_NOT_DEL
    }


    let feeLogIdsStr = dbInfo.fee_log_ids;
    let feeLogIds = feeLogIdsStr.split(",");

    let isActive = false;
    // 根据费率设置纪录的状态，确定是否有记录被激活。
    if (type == FEE_TYPE.FEE_TYPE_SPOT) {
        let dbList = await feeRateSpotLogService.getByIdList(feeLogIds);
        let filterList = _.filter(dbList, i => i.is_check != FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK);
        if (filterList && filterList.length) {
            isActive = true
        }
    }
    else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {
        let dbList = await feeRateBaseCoinContractLogService.getByIdList(feeLogIds);
        let filterList = _.filter(dbList, i => i.is_check != FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK);
        if (filterList && filterList.length) {
            isActive = true
        }
    }
    else {
        let dbList = await feeRateContractLogService.getByIdList(feeLogIds);
        let filterList = _.filter(dbList, i => i.is_check != FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK);
        if (filterList && filterList.length) {
            isActive = true
        }
    }

    if (isActive) {
        await commonUserFeeSetting.prototype.update({
            status: FEE_STATUS.STATUS_FEE_SETTING_EFFECTED,
            update_time: new Date()
        }, {
            where: {
                id: id,
                status: status
            }
        });
        throw ErrorCode.FEE_USED_NOT_DEL
    }


    let tx;
    try {
        tx = await ormDB.transaction();
        if (type == FEE_TYPE.FEE_TYPE_SPOT) {
            await feeRateSpotLogService.uncheck2Deleted(feeLogIds, tx);

        }
        else if (type == FEE_TYPE.FEE_TYPE_BASE_COIN_CONTRACT) {
            await feeRateBaseCoinContractLogService.uncheck2Deleted(feeLogIds, tx);

        }
        else {
            await feeRateContractLogService.uncheck2Deleted(feeLogIds, tx);
        }

        await commonUserFeeSetting.prototype.update({
            status: FEE_STATUS.STATUS_FEE_SETTING_DELETED,
            update_time: new Date()
        }, {
            where: {
                id: id,
                status: status
            },
            transaction: tx
        });
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('commonUserFeeSetting.service.del.error:' + e);
        throw e;
    }

    //管理后台操作日志
    addOptLog(currentUserId, Number(dbInfo.user_id), '删除用户手续费', ip, JSON.stringify(dbInfo), 'VIP管理');

    return 'success';

}


export async function vipLevelList() {
    let resList = await vipLevelFeeSetting.prototype.findAll({
        order: [['id', 'desc']],
        raw: true
    });
    return resList;
}


async function checkAndBuildFeeSettingItem(commonUserFeeVO: CommonUserFeeVO) {
    let type = commonUserFeeVO.type;
    let pair = String(commonUserFeeVO.pair);

    if (type == FEE_TYPE.FEE_TYPE_SPOT) {
        if (pair != 'all') {
            //校验交易对是否存在
            await checkSpotPair(pair);
        }
    }
    else {
        if (pair != 'all') {
            //校验交易对是否存在
            await checkContractPair(pair);
        }

    }

    let dbFeeSetting = await getDbFeeSetting(commonUserFeeVO.user_id, type, commonUserFeeVO.user_channel, pair);
    if (dbFeeSetting) {
        throw ErrorCode.EXIST_ACTIVE_PAIR;
    }

    let feeSettingItem = buildOne(commonUserFeeVO, commonUserFeeVO.maker_fee, commonUserFeeVO.taker_fee, type, pair);
    return feeSettingItem;
}

function buildOne(commonUserFeeVO: CommonUserFeeVO, maker_fee: number | any, taker_fee: number | any, type: number | any, pair: any) {
    let one = {
        user_id: commonUserFeeVO.user_id,
        type: Number(type),
        pair: pair,
        maker_fee: Number(maker_fee),
        taker_fee: Number(taker_fee),
        begin_time: commonUserFeeVO.begin_time,
        expire_time: commonUserFeeVO.expire_time,
        remark: commonUserFeeVO.remark,
        status: FEE_STATUS.STATUS_FEE_SETTING_SUBMIT,
        fee_log_ids: "",
        user_channel: commonUserFeeVO.user_channel,
        statis_period: commonUserFeeVO.statis_period,
        amount_require_usdt: commonUserFeeVO.amount_require_usdt,
        applicant: commonUserFeeVO.applicant,
        create_time: new Date(),
        update_time: new Date(),
    }
    return one;
}

async function getDbFeeSetting(user_id: number | any, type: number | any, user_channel: any, pair: string) {
    let dbInfo = await commonUserFeeSetting.prototype.findOne({
        where: {
            user_id: Number(user_id),
            type: Number(type),
            pair: pair,
            user_channel: user_channel,
            status: { [ormDB.Op.in]: [FEE_STATUS.STATUS_FEE_SETTING_SUBMIT, FEE_STATUS.STATUS_FEE_SETTING_EFFECTED] }
        },
        raw: true
    });
    return dbInfo;
}

async function buildSubmitItems(pair: string, dbInfo: any, now: Date, user_channel: number) {
    let type = Number(dbInfo.type);
    let submitItems: any = [];
    let pairs: any = [];
    //全部交易对时需要查询交易对表
    if (pair == 'all') {
        //现货
        if (type == FEE_TYPE.FEE_TYPE_SPOT) {
            pairs = await getAllSpotPairs();
        }
        else {//合约  这里目前查询的是 contract_pairs 如果之后 币本位的交易对有单独的表 需要修改此处代码
            pairs = await getAllContractPairs();
        }
    }
    else {
        pairs.push(pair);
    }
    for (let onePair of pairs) {
        let item = {
            user_id: dbInfo.user_id,
            pair: onePair,
            fee_model: FEE_MODEL_SPOT_DEFAULT,
            maker_fee: dbInfo.maker_fee,
            taker_fee: dbInfo.taker_fee,
            beginAt: dbInfo.begin_time,
            expireAt: dbInfo.expire_time,
            is_check: FEE_RATE_LOG_STATUS.CHECK_STATUS_UNCHECK,
            comment: "",
            createdAt: now,
            updatedAt: now
        }
        //项目方
        if (user_channel == 2) {
            item.comment = COMMENT_MAKER_FEE_SUBMIT
        }
        else if (user_channel == 3) {
            item.comment = COMMENT_KOL_FEE_SUBMIT
        }
        else {
            item.comment = COMMENT_USER_FEE_SUBMIT
        }

        submitItems.push(item);
    }
    return submitItems;
}

async function checkSpotPair(pair: string) {
    let dbPair = await spotPairs.prototype.findOne(
        {
            where: {
                symbol: pair,
                status: 2
            },
            raw: true
        }
    );
    if (!dbPair) {
        logger.error('checkSpotPair.pair:' + pair);
        throw ErrorCode.PAIR_NOT_EXIST;
    }
}

async function checkContractPair(pair: string) {
    let dbPair = await contractPairs.prototype.findAll(
        {
            where: {
                symbol: pair,
                status: 2
            },
            raw: true
        }
    );
    if (!dbPair) {
        logger.error('checkSpotPair.pair:' + pair);
        throw ErrorCode.PAIR_NOT_EXIST;
    }
}

async function getProjectMakerTradeStat(user_id: number, pair: string) {
    let res = {
        avg7: 0,
        avg15: 0,
        avgUsdt: 0,
    };
    let key = user_id + "_" + pair + "_projectMaker_FeeRate";
    let redisData = await RedisClient.getSync(key);
    if (redisData) {
        let obj = JSON.parse(redisData);
        res.avg7 = obj.avg7;
        res.avg15 = obj.avg15;
        res.avgUsdt = obj.avgUsdt;
    }
    return res;
}


async function getAllSpotPairs() {
    let dbPairs = await spotPairs.prototype.findAll(
        {
            attributes: ['symbol'],
            where: {
                status: 2
            },
            raw: true
        }
    );
    return dbPairs.map(item => item.symbol);

}

async function getAllContractPairs() {
    let dbPairs = await contractPairs.prototype.findAll(
        {
            attributes: ['symbol'],
            where: {
                status: 2
            },
            raw: true
        }
    );
    return dbPairs.map(item => item.symbol);

}



