import BigNumber from "bignumber.js";

let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, financeAccountDataUtils, finAccDwDataUtils, tickerUtils } = require('@madex/ex-js-common');
import { ROBOT_KEYS, sendRobotMessage } from "../../src/utils/robotUtils";
import { siteAssetsNoPage } from "../../src/functional/mvc/service/v2/assetCheck.service";

let { logger, datetimeUtils } = require('@madex/ex-js-public');

/**
 * 余额和
 * 资金转入转出检查并告警
 */
export const balanceAndFundingInOutCheck = async function (start_time: number, end_time: number) {
    //账户
    let accountList = await financeAccountDataUtils.getAccountInfoList();
    if (!accountList.length) {
        return;
    }
    //转入转出数据
    let inOutDataList = await finAccDwDataUtils.getDepositAndWithdrawDataByAccounts(accountList, start_time, end_time);
    //余额、初始资金数据
    let balanceDataList = await financeAccountDataUtils.getFundingDataByAccounts(accountList);

    let warningValueMap: any = {};
    for (let account of accountList) {
        warningValueMap[account.id] = account;
    }

    let balanceMap: any = {};
    for (let oneData of balanceDataList) {
        let { category_id, account_id, balance_usdt } = oneData;
        let warning_value = warningValueMap[account_id] ? warningValueMap[account_id].balance_warning : 0;
        let account_name = warningValueMap[account_id] ? warningValueMap[account_id].account : "未知账户名";
        let platform_name = warningValueMap[account_id] ? getPlatformName(warningValueMap[account_id].platform) : "未知平台";
        if (Number(balance_usdt) <= Number(warning_value)) {
            let msg = `\n做市资金余额低于预设值\n时间:${new Date().toLocaleString()}\n账户ID:${account_id}\n账户名:${account_name}\n平台:${platform_name}\n当前余额:${Number(balance_usdt)}\n预设值:${Number(warning_value)}`;
            await sendRobotMessage(ROBOT_KEYS.ABK_KEY, msg);
        }
        balanceMap[account_id] = balance_usdt;
    }

    let tickerMap = {};
    for (let oneData of inOutDataList) {
        let { account_id, symbol, change, bill_type, time } = oneData;
        let usdt_rate = tickerMap[symbol] ? tickerMap[symbol] : await tickerUtils.rateCoin2USDT(symbol);
        tickerMap[symbol] = usdt_rate;
        let change_usdt = new BigNumber(change).mul(new BigNumber(usdt_rate));
        let account_name = warningValueMap[account_id] ? warningValueMap[account_id].account : "未知账户名";
        let platform_name = warningValueMap[account_id] ? getPlatformName(warningValueMap[account_id].platform) : "未知平台";
        let msg = `\n资产变动\n时间:${new Date(Number(time)).toLocaleString()}\n${bill_type == 1 ? "转入做市账户" : "从做市账户转出"}\n账户ID:${account_id}\n账户名:${account_name}\n平台:${platform_name}\n币种:${symbol}\n数量:${Number(change)}\n估值(U):${Number(change_usdt)}\n当前账户余额(U):${Number(balanceMap[account_id])}`;
        await sendRobotMessage(ROBOT_KEYS.ABK_KEY, msg);
    }


}

function getPlatformName(platform: any) {
    let name = ""
    switch (platform) {
        case 1:
            name = "KTX";
            break;
        case 2:
            name = "Binance";
            break;
        case 3:
            name = "OKEX";
            break;
        default:
            name = "未知平台";
            break;
    }
    return name;
}

/**
 * 站内资产和钱包资产对账
 */
export const walletAssetAndSiteAssetCheck = async function () {
    let dbDataList = await siteAssetsNoPage();
    for (let item of dbDataList) {
        let { asset_id, symbol, chain_type, site_assets, total_wallet_assets } = item;
        let site_assets_number = new BigNumber(site_assets).toNumber().toFixed(4);
        let total_wallet_assets_number = new BigNumber(total_wallet_assets).toNumber().toFixed(4);
        //账不平 发送lark
        if (site_assets_number != total_wallet_assets_number) {
            let msg = `\n站内钱包对账错误\n时间:${new Date().toLocaleString()}\n链名:${chain_type}\n币名:${symbol}\n站内数量:${site_assets_number}\n钱包数量:${total_wallet_assets_number}\n差值:${Number(site_assets_number) - Number(total_wallet_assets_number)}`
            await sendRobotMessage(ROBOT_KEYS.ABK_KEY, msg);
        }
    }
}