import { ormDB, userAccountChangeLog, userInfo, userProfile } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { getProfileByUserIds } from "./mUserProfile.service";
import { getRealNameByUserIds, getRealNameByUserId } from "./mUserRealName.service";
import { MUserRedisKey } from "../../../constant/mUserRedisKey";
import { getMUserInfoByEmail, getMUserInfoByUid, sendEmail, updateMUserInfo } from "../../../utils/mUserUtils";
import { existEmail, getProcessingEmailListByMUserId } from "./mUserAccountChangeLog.service";
import { AFTER_MODIFIED_MAILBOX_TIP, EMAIL_FORBID_TEMPLATE } from "../../../constant/emailTemplateType";
import { addLog, MUserLogType } from "./mUserOptLog.service";
import { addOptLog, LogType } from "./userOptLog.service";
import { checkTotp } from "../../../utils/aclUserUtils";
import { getFatherUserId, getUidsByFatherUid } from "./mUserInfoSon.service";
import { recordMUserOperateLog, TYPE } from "./mUserAccountOperateLog.service";
import { tradeAsset2USDTByUids, walletAsset2USDTByUids } from "./mUserAssets.service";
import BigNumber from "bignumber.js";

let _ = require('lodash');
let { logger, datetimeUtils } = require('@madex/ex-js-public');
let { redisUtilsCommon, authCommon } = require('@madex/ex-js-common');


export interface QueryVO {

    page?: number,

    size?: number

    user_id?: number | any;

    kyc_status?: number | any;

    lock_status?: number | any;

    totp_status?: number | any;

    start_date?: Date | any;

    end_date?: Date | any;

    email?: string | any

    totp_code?: string | any

    lock_type?: number | any//锁定类型 1 锁定当前账户 2 锁定当前账户及其所有子账户


}


export async function userList(queryVO: QueryVO) {
    let where = Object.create(null);
    if (queryVO.user_id) {
        where.user_id = queryVO.user_id;
    }

    if (!isNaN(queryVO.kyc_status)) {
        where.real_name = queryVO.kyc_status ? { [ormDB.Op.not]: null } : "";
    }

    if (!isNaN(queryVO.lock_status)) {
        where.is_lock = queryVO.lock_status;
    }

    if (!isNaN(queryVO.totp_status)) {
        where.totp_encrypt = queryVO.totp_status ? { [ormDB.Op.not]: null } : { [ormDB.Op.is]: null };
    }

    if (queryVO.start_date && queryVO.end_date) {
        let from = datetimeUtils.trim(queryVO.start_date, 's');
        let to = datetimeUtils.trim(queryVO.end_date, 's');
        where.createdAt = { [ormDB.Op.between]: [from, to] };
    }

    let resList = await userInfo.prototype.findAndCount({
        attributes: ['user_id', 'email', 'real_name', 'createdAt', 'totp_encrypt', 'is_lock'],
        where: where,
        limit: queryVO.size,
        offset: (Number(queryVO.page) - 1) * Number(queryVO.size),
        order: [["user_id", "asc"]],
        raw: true
    });
    if (resList.rows.length > 0) {
        let uids = resList.rows.map(item => item.user_id);
        //个人资料
        let profileMap = await getProfileByUserIds(uids);
        //实名信息
        let realNameMap = await getRealNameByUserIds(uids);
        //钱包余额
        let walletAssetMap = await walletAsset2USDTByUids(uids);
        //交易账户余额
        let tradeAssetMap = await tradeAsset2USDTByUids(uids);
        //替换rows的集合
        let infoList: any = [];
        for (let item of resList.rows) {
            let wallet_account_balance = new BigNumber(String(walletAssetMap[item.user_id].balance_usdt)).add(new BigNumber(String(walletAssetMap[item.user_id].holds_usdt)));
            let trade_account_balance = new BigNumber(String(tradeAssetMap[item.user_id].balance_usdt)).add(new BigNumber(String(tradeAssetMap[item.user_id].holds_usdt)));
            let oneProfile = profileMap[item.user_id];
            let oneRealName = realNameMap[item.user_id];
            let one = {
                user_id: item.user_id,
                nick_name: oneProfile ? (oneProfile.nick_name ? oneProfile.nick_name : "") : "",
                real_name: item.real_name,
                email: item.email,
                register_date: item.createdAt,
                spot_trade_total: 0,//TODO:现货交易总量
                contract_trade_total: 0,//TODO:永续合约交易总量
                spot_balance: 0,//TODO:现货余额
                contract_balance: 0,//TODO:永续合约余额
                assets_total: 0,//TODO: 总资产
                spot_order: 0,//TODO: 现货订单数量
                contract_order: 0,//TODO: 永续合约订单数量
                wallet_account_balance: wallet_account_balance,//钱包账户余额
                trade_account_balance: trade_account_balance,//TODO: 交易账户余额
                kyc_status: oneRealName ? oneRealName.status : 0,//0未提交，1审核中，2审核不通过，3审核通过
                kyc_type: oneRealName ? oneRealName.type : 0,//1身份证，2护照，3驾驶证
                kyc_identity: oneRealName ? oneRealName.identity : 0,//证件号
                is_bind_totp: item.totp_encrypt ? 1 : 0,
                lock_status: item.is_lock, //0 否 1 是
            }
            infoList.push(one);
        }
        resList.rows = infoList;

    }

    return resList;
}


export async function oneUserDetail(m_user_id: number) {

    let oneUser = await userInfo.prototype.findOne({
        attributes: ['user_id', 'email', 'real_name', 'createdAt', 'totp_encrypt', 'is_lock', 'secure_modifiedAt'],
        where: {
            user_id: m_user_id
        },
        raw: true
    });

    if (!oneUser) {
        throw ErrorCode.DATA_NOT_EXIST;
    }
    let oneRealName = await getRealNameByUserId(m_user_id);
    let isLoginLock = await redisUtilsCommon.getSync(m_user_id + MUserRedisKey.USER_LOCK_SUFFIX);
    //24小时提现限制
    let dead_line = datetimeUtils.add(oneUser.secure_modifiedAt, datetimeUtils.DAY);
    let is_withdraw = datetimeUtils.between(new Date(), dead_line);
    let res = {
        user_id: oneUser.user_id,
        real_name: oneUser.real_name,
        email: oneUser.email,
        register_date: oneUser.createdAt,
        kyc_status: oneRealName ? oneRealName.status : 0,//0未提交，1审核中，2审核不通过，3审核通过
        kyc_type: oneRealName ? oneRealName.type : 0,//1身份证，2护照，3驾驶证
        kyc_identity: oneRealName ? oneRealName.identity : 0,//证件号
        is_bind_totp: oneUser.totp_encrypt ? 1 : 0,
        lock_status: oneUser.is_lock, //0 否 1 是
        login_lock: isLoginLock ? 1 : 0,//登陆锁定 0 否 1 是
        withdraw_limit_24: is_withdraw > 0 ? 0 : 1,//24小时提现限制
    }
    return res;

}


export async function updateUserEmail(currentUserId: number, m_user_id: any, email: any, totp_code: any, ip: any) {

    //校验谷歌
    await checkTotp(currentUserId, totp_code);

    let mUserDbInfo = await getMUserInfoByEmail(email);
    if (mUserDbInfo) {
        throw ErrorCode.UPDATE_EMAIL_EXIST;
    }

    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }

    //检查邮箱是否存在于 userAccountChangeLog中，是则不可用
    let isExist = await existEmail(email);
    if (isExist) {
        throw ErrorCode.EMAIL_USED_NTO_CHANGE;
    }
    //检查是否已经有一条待修改记录 记录新/旧邮箱
    let isExistList = await getProcessingEmailListByMUserId(m_user_id);
    if (isExistList.length > 0) {
        throw ErrorCode.EMAIL_CHANGE_OFTEN;
    }

    let dbEmail = dnUidUserInfo.email;

    await modifyUserEmail(m_user_id, email, dbEmail);
    //踢出登陆
    authCommon.delSessionIdList(m_user_id);
    //新邮箱发邮件
    sendEmail(email, m_user_id, EMAIL_FORBID_TEMPLATE);
    //旧邮箱发送邮件
    sendEmail(dbEmail, m_user_id, AFTER_MODIFIED_MAILBOX_TIP);
    //记录Madex的用户自己的日志
    addLog(m_user_id, MUserLogType.UP_EMAIL, ip, currentUserId, `运营修改用户邮箱为:${email}`);
    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},修改用户:${m_user_id} 邮箱,原邮箱: ${dbEmail},新邮箱:${email}`, LogType.UPDATE_MUSER_EMAIL);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(m_user_id, currentUserId, TYPE.MAIL, `ip:${ip},修改邮箱`, dbEmail, email);
    return 'success';
}


export async function lockAccount(currentUserId: number, m_user_id: any, ip: any, lock_type: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);
    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    let comment = "";
    let fatherUserId = dnUidUserInfo.user_id;
    //锁定当前用户
    if (lock_type == 1) {
        await updateMUserInfo(m_user_id, { is_lock: 1 });
        //踢出登陆
        authCommon.delSessionIdList(m_user_id);
        comment = `ip:${ip},锁定用户:${fatherUserId}`;
    }
    else {
        if (dnUidUserInfo.user_type == userInfo.USER_TYPE.SON) {
            let temp = await getFatherUserId(m_user_id);
            if (!temp) {
                throw ErrorCode.DAD_ACCOUNT_NOT_EXIST;
            }
            fatherUserId = temp;
            let uids = await getUidsByFatherUid(fatherUserId);
            uids.push(fatherUserId);
            //更新主账户及其子账户
            await userInfo.prototype.update({
                is_lock: 1,
                updatedAt: new Date()
            }, {
                where: {
                    user_id: { [ormDB.Op.in]: uids }
                }
            });
            //踢出登陆
            for (let uid of uids) {
                authCommon.delSessionIdList(uid);
            }
            comment = `ip:${ip},锁定用户:${uids}`;
        }

    }
    //管理后台操作日志
    addOptLog(currentUserId, comment, LogType.LOCK_MUSER);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(fatherUserId, currentUserId, TYPE.ACCOUNT_STATUS, comment, "0", "1");

    return 'success';
}


export async function unlockAccount(currentUserId: number, m_user_id: any, ip: any, lock_type: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);
    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    if (Number(dnUidUserInfo.deleted_flag) > 0) {
        throw ErrorCode.DEL_USER_NO_UNLOCK;
    }
    if (!dnUidUserInfo.is_lock) {
        throw ErrorCode.USER_NOT_LOCK;
    }
    let comment = "";
    let fatherUserId = dnUidUserInfo.user_id;
    //解锁当前用户
    if (lock_type == 1) {
        await updateMUserInfo(m_user_id, {
            is_lock: 0,
            login_error_times: 0
        });
        comment = `ip:${ip},解锁用户:${fatherUserId}`;
    }
    else {
        if (dnUidUserInfo.user_type == userInfo.USER_TYPE.SON) {
            let temp = await getFatherUserId(m_user_id);
            if (!temp) {
                throw ErrorCode.DAD_ACCOUNT_NOT_EXIST;
            }
            fatherUserId = temp;
            let uids = await getUidsByFatherUid(fatherUserId);
            uids.push(fatherUserId);
            //更新主账户及其子账户
            await userInfo.prototype.update({
                is_lock: 0,
                login_error_times: 0,
                updatedAt: new Date()
            }, {
                where: {
                    user_id: { [ormDB.Op.in]: uids }
                }
            });

            comment = `ip:${ip},解锁用户:${uids}`;
        }

    }
    //管理后台操作日志
    addOptLog(currentUserId, comment, LogType.UNLOCK_MUSER);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(fatherUserId, currentUserId, TYPE.ACCOUNT_STATUS, comment, "1", "0");

    return 'success';
}


export async function clearLoginLimit(currentUserId: number, m_user_id: any, ip: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);
    await redisUtilsCommon.delSync(m_user_id + MUserRedisKey.USER_LOCK_SUFFIX);

    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},清除用户登陆限制:${m_user_id}`, LogType.CLEAR_LOGIN_LIMIT);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(m_user_id, currentUserId, TYPE.TWO_HOUR_LIMIT, `ip:${ip},清除用户登陆限制:${m_user_id}`, "", "");

    return 'success';
}


export async function clear24WithdrawLimit(currentUserId: number, m_user_id: any, ip: any, totp_code: any) {
    //校验谷歌
    await checkTotp(currentUserId, totp_code);

    let dnUidUserInfo = await getMUserInfoByUid(m_user_id);
    if (!dnUidUserInfo) {
        throw ErrorCode.USER_NOT_EXIST;
    }
    let date = datetimeUtils.add(new Date(), datetimeUtils.DAY * -1)
    await updateMUserInfo(m_user_id, {
        secure_modifiedAt: date,
    });

    //管理后台操作日志
    addOptLog(currentUserId, `ip:${ip},清除用户24小时提现限制:${m_user_id}`, LogType.WITHDRAW_24_LIMIT);
    //管理后台操作Madex 用户的日志
    recordMUserOperateLog(m_user_id, currentUserId, TYPE.TWENTY_FOUR_HOUR_LIMIT, `ip:${ip},清除用户24小时提现限制:${m_user_id}`, "", "");

    return 'success';
}

async function modifyUserEmail(m_user_id: number, email_new: string, email_old: string) {
    let tx;
    try {
        tx = await ormDB.transaction();
        await userAccountChangeLog.prototype.create({
                user_id: m_user_id,
                state: 3,
                account_old: email_old,
                account_new: email_new,
                type: 1,
                createdAt: new Date(),
                updatedAt: new Date(),
                finish_time: new Date(),
            },
            {
                transaction: tx,
            });

        await userInfo.prototype.update({
            email: email_new,
            updatedAt: new Date(),
        }, {
            where: {
                user_id: m_user_id
            },
            transaction: tx
        });
        await tx.commit();
    }
    catch (e) {
        if (tx) {
            await tx.rollback();
        }
        logger.error('modifyUserEmail.error:' + e)
        throw e;
    }

}





