import * as userOptService from "../service/userOpt.service";
import { AclUserInfoVO, AclUserInfoPageVO } from "../service/aclUser.service";

let { logger, Res3Utils, optionalUtils: Optional, apiAssertUtils: ApiAssert } = require('@madex/ex-js-public');
import { ErrorCode } from "../../../constant/errorCode";
import { getCurrentUser, getCurrentUserId, isAdminUserBySessionId } from "../../../utils/aclUserUtils";

let isIp = require('is-ip');

/**
 * 获取用户信息
 * @param req
 * @param infoVO
 */
export const getInfo = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.getInfo";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await userOptService.getInfo(currentUserId, req.cookies.session_id);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
};


/**
 * 登陆
 * @param req
 * @param infoVO
 */
export const login = async (req: any, aclUserInfoVO: AclUserInfoVO, res: any) => {
    let func_name = "userOptCtrl.login";
    let cmd = req.path;
    let result: any;
    try {
        let ip = isIp(req.ip) ? req.ip : '*.*.*.*';

        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, aclUserInfoVO.account);
        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, aclUserInfoVO.pwd);
        ApiAssert.isFalse(ErrorCode.ACCOUNT_OR_PWD_ERR, aclUserInfoVO.pwd.length < 8);
        ApiAssert.isFalse(ErrorCode.ACCOUNT_OR_PWD_ERR, aclUserInfoVO.pwd.length > 30);

        result = await userOptService.login(aclUserInfoVO.account, aclUserInfoVO.pwd, ip || "");
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }
    if (result.result && result.sessionId) {
        if (req.headers.origin){
            let iSecure = req.headers.origin.startsWith("https");

            let options = {
                expires: new Date(new Date().getTime() + 1000 * 60 * 60 * 8), // 15 days
                httpOnly: true,
                secure: iSecure
            };

            res.cookie('session_id', result.sessionId, options);
            delete result.sessionId;
        }
    }
    return Res3Utils.result(result);
};

/**
 * 登出
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const logout = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.logout";
    let cmd = req.path;
    try {
        if (req.cookies.session_id) {
            let currentUser = await getCurrentUser(req.cookies.session_id);
            if (currentUser) {
                await userOptService.deleteAllSessionByUserId(currentUser.userId);
            }
        }
        return Res3Utils.result('ok');
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};

/**
 * 登录-二次验证
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const loginConfirm = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.loginConfirm";
    let cmd = req.path;
    try {
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totpCode);

        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await userOptService.loginConfirm(req.cookies.session_id, currentUserId, aclUserInfoVO.totpCode);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};


/**
 * 修改密码
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const updatePwd = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.updatePwd";
    let cmd = req.path;
    try {
        let originPwd = aclUserInfoVO.originPwd;
        let newPwd = aclUserInfoVO.newPwd;

        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, originPwd);
        ApiAssert.isFalse(ErrorCode.PARAM_MISS, originPwd.length < 8);
        ApiAssert.isFalse(ErrorCode.PARAM_MISS, originPwd.length > 30);

        ApiAssert.isNotEmpty(ErrorCode.PARAM_MISS, newPwd);
        ApiAssert.isFalse(ErrorCode.PARAM_MISS, newPwd.length < 8);
        ApiAssert.isFalse(ErrorCode.PARAM_MISS, newPwd.length > 30);

        let res = await userOptService.updatePwd(currentUserId, originPwd, newPwd);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};

/**
 * 绑定谷歌-生成新的密钥
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const bindTotpAsk = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.bindTotpAsk";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        let res = await userOptService.bindTotpAsk(req.cookies.session_id, currentUserId);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};

/**
 * 绑定谷歌-验证新密钥
 * @param req
 * @param aclUserInfoVO
 * @param res
 */
export const bindTotpConfirm = async (req: any, aclUserInfoVO: AclUserInfoVO) => {
    let func_name = "userOptCtrl.bindTotpConfirm";
    let cmd = req.path;
    try {
        let currentUserId = await getCurrentUserId(req.cookies.session_id);
        ApiAssert.notNull(ErrorCode.PARAM_MISS, aclUserInfoVO.totpCode);
        let res = await userOptService.bindTotpConfirm(req.cookies.session_id, currentUserId, aclUserInfoVO.totpCode);
        return Res3Utils.result(res);
    }
    catch (e) {
        logger.error(`${func_name} error:${e}`);
        return Res3Utils.getErrorResult(e);
    }

};


