// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { madAdminOrmDB, aclUserInfo, aclUserAuthConfig } from "@madex/ex-ts-dao";
import { AclUserInfoConst } from "../../../constant/aclUserConstant";
import { CryptUtils } from "../../../utils/crypt-utils";
import { ErrorCode } from "../../../constant/errorCode";
import * as aclRoleAuthService from "../service/aclRoleAuth.service";
import * as aclUserService from "../service/aclUser.service";
import { getOneAclUserByAccount, getOneAclUserByUid } from "../../../utils/aclUserUtils";
import { RedisVal } from "../../../constant/redis-val";
import Config from "../../../../config";
import { AuthConfigConst } from "../../../constant/aclUserAuthConfigConstant";
import * as userOptLogService from "./userOptLog.service";


let { apiAssertUtils: ApiAssert, datetimeUtils: DatetimeUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, BigNumberUtils } = require('@madex/ex-js-common');

let _ = require('lodash');
let { logger } = require('@madex/ex-js-public');

//默认的有效期时间，3天
const LockedDeadlineDay = 3

export interface AuthConfigVO {
    id?: number;

    user_id?: number;

    totp_encrypt?: string | any;

    is_locked?: number | any;

    locked_time?: Date | any;

    force?: number | any;

    deadline?: Date | any;

    createdAt?: Date | any;

    updatedAt?: Date | any;

    user_type?: number | any;

}

export interface AuthConfigPageVO extends AuthConfigVO {
    page?: number,

    size?: number
}

export async function findByUserId(userId: number) {
    try {
        if (!userId) {
            throw ErrorCode.PARAM_MISS
        }
        let configInfo = await aclUserAuthConfig.prototype.findOne({
            where: {
                user_id: userId
            },
            raw: true
        });
        return configInfo;
    }
    catch (e) {
        logger.error('aclUserService.findByUserId.error:' + e)
        throw e;
    }
}

async function findByUserIdList(userIdList: number[]) {
    try {
        if (!userIdList || userIdList.length < 1) {
            throw ErrorCode.PARAM_MISS
        }
        let configInfoList = await aclUserAuthConfig.prototype.findAll({
            where: {
                user_id: { [madAdminOrmDB.Op.in]: userIdList }
            },
            raw: true
        });
        return configInfoList;
    }
    catch (e) {
        logger.error('aclUserService.findByUserIdList.error:' + e)
        throw e;
    }
}


export async function queryConfigList(userId: number | undefined, userType: any, page: number | undefined, size: number | undefined) {

    //查询账户信息
    let typeArr = [AclUserInfoConst.USER_TYPE.SUPPORT]

    if (userType && (await _checkUserType(userType))) {
        typeArr = [Number(userType)]
    }

    let pageList = await aclUserService.findAllForPage(Number(page), Number(size), userId, typeArr,
        [AclUserInfoConst.USER_STATUS.NORMAL, AclUserInfoConst.USER_STATUS.LOCK], null, null, null);

    if (pageList.rows.length < 1) {
        return pageList;
    }

    //查询安全项配置
    let configMap = {}
    let configList = await findByUserIdList(pageList.rows.map(item => item.user_id))
    if (configList && configList.length >= 1) {
        configMap = _.keyBy(configList, item => item.user_id)
    }

    let resultList: any[] = []
    for (let infoItem of pageList.rows) {
        let force = 0, isLocked = 0, hasTotp = 0, deadline = '', lockedTime = ''

        let configItem = configMap[infoItem.user_id]
        if (configItem) {
            force = configItem.force
            isLocked = configItem.is_locked
            hasTotp = configItem.totp_encrypt === '' ? 0 : 1
            deadline = configItem.deadline
            lockedTime = configItem.locked_time
        }

        let item = {
            userId: infoItem.user_id,
            account: infoItem.account,
            force: force,
            deadline: deadline,
            hasTotp: hasTotp,
            isLocked: isLocked,
            lockedTime: lockedTime,
        }
        resultList.push(item)
    }
    pageList.rows = resultList
    return pageList
}


export async function changeForceStatus(userId: number, forceStatus: any, session_id: any) {
    let arr = [AuthConfigConst.FORCE.FALSE, AuthConfigConst.FORCE.TRUE];
    if (!arr.includes(forceStatus)) {
        throw ErrorCode.PARAM_MISS
    }
    //已当前时间点顺延
    let deadline = DatetimeUtils.add(new Date(), LockedDeadlineDay * DatetimeUtils.DAY)

    //查询是否已有配置记录
    let configExist = await _checkAndGetAuthConfig(userId)
    if (configExist) {
        if (Number(configExist.force) === Number(forceStatus)) {
            return 'success'
        }
        if (Number(forceStatus) === AuthConfigConst.FORCE.FALSE) {
            deadline = configExist.deadline
        }
        let data2Update = {
            force: forceStatus,
            deadline: deadline,
            updatedAt: new Date()
        }
        await aclUserAuthConfig.prototype.update(data2Update, { where: { id: configExist.id } })
    }
    else {
        await getOneAclUserByUid(userId)
        let data2Add = {
            user_id: userId,
            totp_encrypt: '',
            is_locked: AuthConfigConst.IS_LOCKED.FALSE,
            force: forceStatus,
            deadline: deadline,
            createdAt: new Date(),
            updatedAt: new Date()
        }
        await aclUserAuthConfig.prototype.create(data2Add)
    }
    //如果是开启并且已绑定谷歌，则剔除登录状态
    if (Number(forceStatus) === AuthConfigConst.FORCE.TRUE && configExist && configExist.totp_encrypt !== '') {
        await _deleteAllSessionByUserId(userId)
    }
    userOptLogService.addOptLog(null, `change user: ${userId}  force status : ${forceStatus}`, userOptLogService.LogType.UPDATE, '', session_id);
    return 'success'
}


export async function changeLockedStatus(userId: number | undefined, session_id: any) {

    //查询是否已有配置记录
    let configExist = await _checkAndGetAuthConfig(userId)
    if (!configExist || configExist.is_locked === AuthConfigConst.IS_LOCKED.FALSE) {
        return 'success'
    }

    //若已开启强制绑定，则重新设置有效时间
    let deadline = configExist.deadline
    if (Number(configExist.force) === AuthConfigConst.FORCE.TRUE) {
        deadline = DatetimeUtils.add(new Date(), LockedDeadlineDay * DatetimeUtils.DAY)
    }

    //解除锁定
    let data2Update = {
        is_locked: AuthConfigConst.IS_LOCKED.FALSE,
        deadline: deadline,
        updatedAt: new Date()
    }
    await aclUserAuthConfig.prototype.update(data2Update, { where: { id: configExist.id } })
    userOptLogService.addOptLog(null, `change user:${userId} lock status : ${AuthConfigConst.IS_LOCKED.FALSE}`, userOptLogService.LogType.UPDATE, '', session_id);
    return "success"
}

export async function resetTotp(userId: number | undefined, session_id: any) {
    //查询是否已有配置记录
    let configExist = await _checkAndGetAuthConfig(userId)
    if (!configExist || configExist.totp_encrypt === '') {
        return 'success'
    }

    let data2Update = {
        totp_encrypt: '',
        is_locked: AuthConfigConst.IS_LOCKED.FALSE,
        force: AuthConfigConst.FORCE.TRUE,
        deadline: DatetimeUtils.add(new Date(), LockedDeadlineDay * DatetimeUtils.DAY),
        updatedAt: new Date()
    }
    await aclUserAuthConfig.prototype.update(data2Update, { where: { id: configExist.id } })
    //剔除登录态
    await _deleteAllSessionByUserId(Number(userId));
    userOptLogService.addOptLog(null, `reset user:${userId} totp`, userOptLogService.LogType.UPDATE, '', session_id);

    return 'success'
}

async function _checkUserType(userType: number) {
    let arr = [AclUserInfoConst.USER_TYPE.SUPPORT, AclUserInfoConst.USER_TYPE.ADMIN];
    if (!arr.includes(userType)) {
        throw ErrorCode.USER_TYPE_ILLEGAL
    }
    return true;
}


async function _checkAndGetAuthConfig(userId: number | any) {
    let userExist = await getOneAclUserByUid(userId);
    if (userExist) {
        throw ErrorCode.USER_NOT_EXIST
    }
    await _checkUserType(userExist.user_type)

    return findByUserId(userId);
}

async function _deleteAllSessionByUserId(userId: number) {

    //获取该账户使用过的所有sessionId
    let sessionListKey = RedisVal.sessionListKey(userId)
    RedisClient.lrange(sessionListKey, 0, -1, async (err, reply) => {
        //删除所有sessionId
        if (!err && reply && reply.length >= 1) {
            await RedisClient.delSync(...reply)
        }
        //删除sessionList
        await RedisClient.delSync(sessionListKey)
    })
}



