// @madex/ex-ts-dao 是 ts 的 dao, 代码在 bitbucket/ex-js-dao 的 ts 分支上
import { contractPairs, madContractOrmDB, coinType, spotPairs } from "@madex/ex-ts-dao";
import { NUMBER } from "sequelize";
import { addPairToCore } from "../../../utils/coreSystemUtils";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";
import { checkPairInCoinType } from "../../../utils/coinTypeUtils";

let { logger, datetimeUtils } = require('@madex/ex-js-public');

export interface ListParam {
    symbol?: string | any;

    page: number;

    size: number;

    status?: number

    from_time?: Date | any

    to_time?: Date | any
}

// name symbol index 都保持一致 比如BTC_USDT_SWAP
export interface AddParam {
    id?: number;

    symbol?: string;

    product_id?: number;

    base?: string;

    quote?: string;

    name?: string;

    price_scale?: number;

    quantity_scale?: number;

    maker_fee?: string;

    taker_fee?: string;

    price_increment?: number;

    quantity_increment?: number;

    default_initial_margin?: number; //"0.01",

    base_initial_margin?: number;//"0.01",

    base_maintenance_margin?: number;//"0.01",

    settlement_asset?: number;// "USDT", 结算币种

    index?: string;

    base_interest: number; //"0.0003",

    quote_interest?: number; //"0.0006",

    impact_notional?: number; // "600000",

    base_risk_limit?: number; // "1000000",

    risk_limit_step?: number; //"1000000",

    funding_times_perday: number; // 3

    status?: number

    min_order_size?: number

    max_order_size?: number

    min_order_value?: number

    max_order_value?: number

    view_precision?: number

    createdAt?: Date | any,

    updatedAt?: Date | any,
}


export async function list(param: ListParam) {
    let where = {};
    if (param.symbol) {
        where["symbol"] = { [madContractOrmDB.Op.like]: `%${param.symbol}%` };
    }
    if (param.status || param.status === 0) {
        where["status"] = param.status
    }
    if (param.from_time && param.to_time) {
        where['createdAt'] = { [madContractOrmDB.Op.between]: [param.from_time, param.to_time] }
    }
    let resList = await contractPairs.prototype.findAndCount({
        where: where,
        limit: param.size,
        offset: (Number(param.page) - 1) * Number(param.size),
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}

export const save = async (param: AddParam, currentUserId: any, ip: any) => {
    //检查交易对中的币种是否存在于coin_type 中
    await checkPairInCoinType(String(param.symbol));

    param.updatedAt = new Date();
    param.createdAt = new Date();
    let dbInfo = await contractPairs.prototype.findOne({
        where: {
            symbol: param.symbol
        },
        raw: true
    });
    if (dbInfo) {
        throw ErrorCode.DATA_EXIST;
    }

    if (!param.price_increment) {
        param.price_increment = 1;
    }
    if (!param.quantity_increment) {
        param.quantity_increment = 1;
    }
    await contractPairs.prototype.create(param);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '合约代码配置新增', ip, JSON.stringify(param), '合约代码配置');
    return 'ok';
};


export const update = async (param: AddParam, currentUserId: any, ip: any) => {
    let dbInfo = await contractPairs.prototype.findOne({
        where: {
            id: param.id
        },
        raw: true
    });
    if (!dbInfo) {
        throw ErrorCode.DATA_NOT_EXIST
    }
    let updateInfo = { updatedAt: new Date() };
    if (param.view_precision || param.view_precision === 0) {
        updateInfo['view_precision'] = param.view_precision
    }
    if (param.price_increment) {
        updateInfo['price_increment'] = param.price_increment
    }
    if (param.quantity_increment) {
        updateInfo['quantity_increment'] = param.quantity_increment
    }

    await contractPairs.prototype.update(updateInfo, {
        where: {
            id: Number(param.id)
        }
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '合约代码配置更新', ip, JSON.stringify(param), '合约代码配置');
    return 'ok';
};

export const pushToCoreSystem = async (id: any, currentUserId: any, ip: any) => {
    let where = { id: id, status: 0 };

    let cfg = await contractPairs.prototype.find({
        where: where, raw: true
    });
    if (!cfg) {
        throw ErrorCode.SUBMIT_STATUS_ERR;
    }
    let symbol = cfg.symbol;
    let sps = symbol.split("_");
    let base = sps[0];
    let quote = sps[1]
    let param = {
        "market_type": 2,
        "base": base, "quote": quote, "symbol": symbol, "name": symbol,
        "price_scale": cfg.price_scale, "quantity_scale": cfg.quantity_scale,
        "maker_fee": cfg.maker_fee, "taker_fee": cfg.taker_fee,
        "default_initial_margin": cfg.default_initial_margin,
        "base_initial_margin": cfg.base_initial_margin,
        "base_maintenance_margin": cfg.base_maintenance_margin,
        "settlement_asset": cfg.settlement_asset,
        "index": cfg.settlement_asset,
        "base_interest": cfg.base_interest,
        "quote_interest": cfg.quote_interest,
        "impact_notional": cfg.impact_notional,
        "base_risk_limit": cfg.base_risk_limit,
        "risk_limit_step": cfg.risk_limit_step,
        "funding_times_perday": cfg.funding_times_perday
    }
    if (cfg.min_order_size) {
        param["min_order_size"] = cfg.min_order_size;
    }
    if (cfg.max_order_size) {
        param["max_order_size"] = cfg.max_order_size;
    }
    if (cfg.min_order_value) {
        param["min_order_value"] = cfg.min_order_value;
    }
    if (cfg.max_order_value) {
        param["max_order_value"] = cfg.max_order_value;
    }
    let optResult = await addPairToCore(param);
    if (!optResult.is_success) {
        //管理后台操作日志
        addOptLog(currentUserId, 0, '提交合约交易对到撮合-失败', ip, `pair:${symbol},error_msg:${optResult.err_msg}`, '合约代码配置');
        throw ErrorCode.ADD_PAIR_TO_CORE_ERR;
    }

    await contractPairs.prototype.update({ status: 1 }, {
        where: where
    });
    //管理后台操作日志
    addOptLog(currentUserId, 0, '提交合约交易对到撮合-成功', ip, JSON.stringify(param), '合约代码配置');
    return 'ok';
};

export async function getAllSubmitSuccess() {
    let resList = await contractPairs.prototype.findAll({
        where: {
            status: 2
        },
        order: [["id", "desc"]],
        raw: true
    });
    return resList;
}


