import { exBusinessAreaRouter, exBusinessArea, ormDB, userApikeyStrict } from "@madex/ex-ts-dao";
import { ErrorCode } from "../../../constant/errorCode";
import { addOptLog } from "./userOptLog.service";


let _ = require('lodash');
let { logger } = require('@madex/ex-js-public');


export interface BusinessAreaVO {
    id?: number;

    is_active?: number;

    symbol?: string;

    name?: string;

    level?: number;

    father_area_id?: number;

    weight?: number;

    comment?: string;

    path?: string;

    icon_url?: string;

    createdAt?: Date | any,

    updatedAt?: Date | any,

}

export interface AreaRouterVO {
    id?: number;

    area_id?: string,

    pair?: string;

    weight?: number;

    createdAt?: Date | any,

    updatedAt?: Date | any,


}

export interface BusinessAreaPageVO extends BusinessAreaVO, AreaRouterVO {

    page?: number,

    size?: number,

}


export async function list(pageVO: BusinessAreaPageVO) {

    let where = {};

    if (pageVO.symbol) {
        where['symbol'] = pageVO.symbol
    }
    if (pageVO.is_active || pageVO.is_active === 0) {
        where['is_active'] = pageVO.is_active
    }
    if (pageVO.level) {
        where['level'] = pageVO.level
    }
    if (pageVO.father_area_id) {
        where['father_area_id'] = pageVO.father_area_id
    }

    let resList = await exBusinessArea.prototype.findAll({
        order: [["id", "asc"]],
        raw: true
    });
    return resList;
}


export async function add(businessAreaVO: BusinessAreaVO, currentUserId: any, ip: string | undefined) {

    businessAreaVO.createdAt = new Date();
    businessAreaVO.updatedAt = new Date();


    await exBusinessArea.prototype.create(businessAreaVO);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增交易区配置', ip, JSON.stringify(businessAreaVO), '交易区配置');
    return 'success';
}


export async function update(businessAreaVO: BusinessAreaVO, currentUserId: any, ip: string | undefined) {
    let exist = await exBusinessArea.prototype.findOne({
        where: {
            id: businessAreaVO.id
        },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    businessAreaVO.updatedAt = new Date();

    await exBusinessArea.prototype.update(businessAreaVO, {
        where: {
            id: Number(businessAreaVO.id)
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, 0, '修改交易区配置', ip, JSON.stringify(businessAreaVO), '交易区配置');

    return 'success';
}

export async function del(id: number, currentUserId: any, ip: string | undefined) {
    let exist = await exBusinessArea.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    await exBusinessArea.prototype.destroy({
        where: {
            id: id
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, 0, '删除交易区配置', ip, `id:${id}`, '交易区配置');

    return 'success';
}


export async function getTree() {


    let dbList = await exBusinessArea.prototype.findAll({
        where: {
            level: { [ormDB.Op.in]: [1, 2, 3] }
        },
        order: [["level", "asc"], ['father_area_id', 'asc'], ['weight', 'desc']],
        raw: true
    });
    let m = {};
    let treeList: any = [] // 需要返回给前端tree 数组
    for (let item of dbList) {
        m[item.id] = item;
        let pid = item.father_area_id;
        let level = item.level;
        if (Number(level) === 1) {
            treeList.push(item)
        }
        else {
            let pAuth = m[pid];
            if (!pAuth) {
                logger.warn("上级交易区被删除或者不存在");
                continue
            }
            if (!pAuth['childList']) {
                pAuth['childList'] = [];
            }
            pAuth['childList'].push(item);
        }
    }
    return treeList;

}


export async function routerList(pageVO: BusinessAreaPageVO) {
    let where = {};

    if (pageVO.area_id) {
        where['area_id'] = pageVO.area_id
    }
    if (pageVO.pair) {
        where['pair'] = pageVO.pair
    }


    let resList = await exBusinessAreaRouter.prototype.findAndCount({
        where: where,
        limit: pageVO.size,
        offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
        order: [["area_id", "asc"]],
        raw: true
    });
    return resList;
}


export async function routerAdd(areaRouterVO: AreaRouterVO, currentUserId: any, ip: string | undefined) {

    let areaIdsStr = String(areaRouterVO.area_id);
    let areaIdsArr = areaIdsStr.split(',');

    let dbList = await exBusinessAreaRouter.prototype.findAll({
        where: {
            area_id: { [ormDB.Op.in]: areaIdsArr }
        },
        raw: true
    });
    let dbAreaIds = dbList.map(item => item.area_id);
    let filterAreaIds = areaIdsArr.filter(item => !dbAreaIds.includes(item));
    let insertList: any = [];
    let now = new Date();
    for (let areaId of filterAreaIds) {
        let item = {
            pair: areaRouterVO.pair,
            area_id: areaId,
            weight: areaRouterVO.weight,
            createdAt: now,
            updatedAt: now,
        }
        insertList.push(item);
    }

    await exBusinessAreaRouter.prototype.bulkCreate(insertList);
    //管理后台操作日志
    addOptLog(currentUserId, 0, '新增币种所属交易区配置', ip, JSON.stringify(areaRouterVO), '币种所属交易区配置');
    return 'success';
}


export async function routerUpdate(areaRouterVO: AreaRouterVO, currentUserId: any, ip: string | undefined) {
    let exist = await exBusinessAreaRouter.prototype.findOne({
        where: {
            id: areaRouterVO.id
        },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    areaRouterVO.updatedAt = new Date();

    await exBusinessAreaRouter.prototype.update(areaRouterVO, {
        where: {
            id: Number(areaRouterVO.id)
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, 0, '修改币种所属交易区配置', ip, JSON.stringify(areaRouterVO), '币种所属交易区配置');

    return 'success';
}

export async function routerDel(id: number, currentUserId: any, ip: string | undefined) {
    let exist = await exBusinessAreaRouter.prototype.findOne({
        where: {
            id: id
        },
        raw: true
    });
    if (!exist) {
        throw ErrorCode.DATA_NOT_EXIST
    }

    await exBusinessAreaRouter.prototype.destroy({
        where: {
            id: id
        }
    })

    //管理后台操作日志
    addOptLog(currentUserId, 0, '删除币种所属交易区配置', ip, `id:${id}`, '币种所属交易区配置');

    return 'success';
}



