import {
    ormDB,
    userLog,
    userInfo,
    userRealName,
    mainUserBills
} from "@madex/ex-ts-dao";
import BigNumber from "bignumber.js";


let _ = require('lodash');
let { tickerUtils } = require('@madex/ex-js-common');


export interface QueryVO {
    page?: number,

    size?: number,

    is_export?: number,//是否导出

    from_time?: Date | any,

    to_time?: Date | any,

}


export async function increaseUserList(pageVO: QueryVO) {
    let where = {
        type: 22,//目前只有邮箱注册
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: pageVO.from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: pageVO.to_time },
            }]
    }
    let resList: any = [];
    if (!pageVO.is_export) {
        resList = await userLog.prototype.findAndCount({
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealIncreaseUserReturnData(resList.rows);
    }
    else {//导出不分页
        resList = await userLog.prototype.findAll({
            where: where,
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealIncreaseUserReturnData(resList);
    }
    return resList;
}


export async function tradeUserList(pageVO: QueryVO) {
    //TODO:
    let res = {
        count: 0,
        rows: []
    }

    return res
}

export async function activeUserList(pageVO: QueryVO) {

    if (!pageVO.is_export) {
        let res = {
            count: 0,
            rows: []
        }
        let count_sql = `SELECT COUNT(*) AS num
                         FROM user_log AS t1
                                  INNER JOIN
                              (SELECT user_id, MAX(createdAt) AS createdAt
                               FROM user_log
                               WHERE type = 0
                                 AND createdAt >= '${pageVO.from_time}'
                                 AND createdAt < '${pageVO.to_time}'
                               GROUP BY user_id) AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;

        let data_sql = `SELECT t1.*
                        FROM user_log AS t1
                                 INNER JOIN
                             (SELECT user_id, MAX(createdAt) AS createdAt
                              FROM user_log
                              WHERE type = 0
                                AND createdAt >= '${pageVO.from_time}'
                                AND createdAt < '${pageVO.to_time}'
                              GROUP BY user_id
                              ORDER BY createdAt DESC LIMIT ${(Number(pageVO.page) - 1) * Number(pageVO.size)}, ${Number(pageVO.size)})
                                 AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;
        let dbCount = await ormDB.query(count_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
        let dbData = await ormDB.query(data_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
        res.count = dbCount[0].num;
        res.rows = dbData;
        return res
    }
    else {//导出不分页
        let data_sql = `SELECT t1.*
                        FROM user_log AS t1
                                 INNER JOIN
                             (SELECT user_id, MAX(createdAt) AS createdAt
                              FROM user_log
                              WHERE type = 0
                                AND createdAt >= '${pageVO.from_time}'
                                AND createdAt < '${pageVO.to_time}'
                              GROUP BY user_id
                              ORDER BY createdAt DESC)
                                 AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;
        let dbData = await ormDB.query(data_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
        return dbData;
    }

}


export async function userDepositList(pageVO: QueryVO) {
    let where = {
        type: 100,//充值
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let dbDataList: any = [];
    if (!pageVO.is_export) {
        dbDataList = await mainUserBills.prototype.findAndCount({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList.rows);
    }
    else {//导出不分页
        dbDataList = await mainUserBills.prototype.findAll({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList);
    }

    return dbDataList;
}


export async function userWithdrawList(pageVO: QueryVO) {
    let where = {
        type: 102,//提现
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let dbDataList: any = [];
    if (!pageVO.is_export) {
        dbDataList = await mainUserBills.prototype.findAndCount({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList.rows);

    }
    else {//导出不分页
        dbDataList = await mainUserBills.prototype.findAll({
            attributes: ['user_id', 'symbol', 'change', 'createdAt'],
            where: where,
            order: [["createdAt", "desc"]],
            raw: true
        });
        await dealDepositOrWithdrawReturnData(dbDataList);

    }

    return dbDataList;
}

export async function userDepositAndWithdrawList(pageVO: QueryVO) {
    let where = {
        type: { [ormDB.Op.in]: [100, 102] },//充值、提现
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let dbDataList: any = [];
    if (!pageVO.is_export) {
        dbDataList = await mainUserBills.prototype.findAndCount({
            attributes: ['user_id', 'symbol',
                ormDB.literal('sum(case when type = 100 then ABS(`change`) else 0 end) as deposit_amount'),
                ormDB.literal('sum(case when type = 102 then ABS(`change`) else 0 end) as withdraw_amount')
            ],
            where: where,
            limit: pageVO.size,
            offset: (Number(pageVO.page) - 1) * Number(pageVO.size),
            group: ['user_id', 'symbol'],
            raw: true
        });
        //这里分组count的是每个分组的条数 count 实际 = 几个分组数
        dbDataList.count = dbDataList.count.length;
        await dealDepositAndWithdrawReturnData(dbDataList.rows);

    }
    else {//导出不分页
        dbDataList = await mainUserBills.prototype.findAll({
            attributes: ['user_id', 'symbol',
                ormDB.literal('sum(case when type = 100 then ABS(`change`) else 0 end) as deposit_amount'),
                ormDB.literal('sum(case when type = 102 then ABS(`change`) else 0 end) as withdraw_amount')
            ],
            where: where,
            group: ['user_id', 'symbol'],
            raw: true
        });
        await dealDepositAndWithdrawReturnData(dbDataList);

    }


    return dbDataList;
}


export async function gatherData(pageVO: QueryVO) {
    let [increaseCount, tradeCount, activeCount, depositAndWithdraw] =
        await Promise.all([increaseUserCount(pageVO), tradeUserCount(pageVO), activeUserCount(pageVO), depositAndWithdrawCount(pageVO)]);
    return {
        increase: increaseCount,
        trade: tradeCount,
        active: activeCount,
        deposit: depositAndWithdraw.deposit,
        withdraw: depositAndWithdraw.withdraw,
        deposit_withdraw: depositAndWithdraw.deposit_withdraw,
    }
}

async function increaseUserCount(pageVO: QueryVO) {
    let where = {
        type: 22,//目前只有邮箱注册
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: pageVO.from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: pageVO.to_time },
            }]
    }
    let count = await userLog.prototype.count({
        where: where,
        raw: true
    });
    return count;
}

async function tradeUserCount(pageVO: QueryVO) {
    //TODO:查询交易数据
    return 0;
}


async function activeUserCount(pageVO: QueryVO) {

    let count_sql = `SELECT COUNT(*) AS num
                     FROM user_log AS t1
                              INNER JOIN
                          (SELECT user_id, MAX(createdAt) AS createdAt
                           FROM user_log
                           WHERE type = 0
                             AND createdAt >= '${pageVO.from_time}'
                             AND createdAt < '${pageVO.to_time}'
                           GROUP BY user_id) AS t2 ON t1.user_id = t2.user_id AND t1.createdAt = t2.createdAt`;

    let dbCount = await ormDB.query(count_sql, { raw: true, type: ormDB.QueryTypes.SELECT });
    let count = dbCount[0].num;
    return count;
}

async function depositAndWithdrawCount(pageVO: QueryVO) {
    let res = {
        deposit: new BigNumber(0),
        withdraw: new BigNumber(0),
        deposit_withdraw: new BigNumber(0),
    }
    let where = {
        type: { [ormDB.Op.in]: [100, 102] },//充值、提现
    };
    where[ormDB.Op.and] = [
        {
            createdAt: { [ormDB.Op.gte]: pageVO.from_time }
        },
        {
            createdAt: { [ormDB.Op.lt]: pageVO.to_time },
        }];
    let amountList = await mainUserBills.prototype.findAll({
        attributes: ['symbol',
            ormDB.literal('sum(case when type = 100 then ABS(`change`) else 0 end) as deposit_amount'),
            ormDB.literal('sum(case when type = 102 then ABS(`change`) else 0 end) as withdraw_amount'),
        ],
        where: where,
        group: ['symbol'],
        raw: true
    });
    if (amountList.length) {
        let deposit_usdt = new BigNumber(0);
        let withdraw_usdt = new BigNumber(0);
        for (let item of amountList) {
            let symbol = item.symbol;
            let deposit_amount = item.deposit_amount;
            let withdraw_amount = item.withdraw_amount;
            let usdt = await tickerUtils._rateCoin2USDT(symbol);
            let deposit_amount_usdt = new BigNumber(deposit_amount).mul(new BigNumber(usdt));
            deposit_usdt = deposit_usdt.add(deposit_amount_usdt);

            let withdraw_amount_usdt = new BigNumber(withdraw_amount).mul(new BigNumber(usdt));
            withdraw_usdt = withdraw_usdt.add(withdraw_amount_usdt);
        }
        let deposit_withdraw_usdt = deposit_usdt.sub(withdraw_usdt);

        res.deposit = deposit_usdt;
        res.withdraw = withdraw_usdt;
        res.deposit_withdraw = deposit_withdraw_usdt;
    }
    return res;
}

async function dealIncreaseUserReturnData(resList: any) {
    if (resList.length) {
        let uids = resList.map(item => item.user_id);
        let dbUserListTask = userInfo.prototype.findAll({
            attributes: ['user_id', 'createdAt', 'user_from', 'invited_by_id', 'is_active'],
            where: {
                user_id: { [ormDB.Op.in]: uids }
            },
            raw: true
        });

        let dbRealNameListTask = userRealName.prototype.findAll({
            attributes: ['user_id', 'status'],
            where: {
                user_id: { [ormDB.Op.in]: uids }
            },
            raw: true
        })
        let [dbUserList, dbRealNameList] = await Promise.all([dbUserListTask, dbRealNameListTask]);
        let userMap = {};
        let realNameMap = {};
        for (let item of dbUserList) {
            userMap[item.user_id] = item;
        }

        for (let item of dbRealNameList) {
            realNameMap[item.user_id] = item;
        }
        for (let item of resList) {
            let userId = item.user_id;
            item.createdAt = userMap[userId] ? userMap[userId].createdAt : item.createdAt;
            item.user_from = userMap[userId] ? userMap[userId].user_from : 0;
            item.invited_by_id = userMap[userId] ? userMap[userId].invited_by_id : '';
            item.is_active = userMap[userId] ? userMap[userId].is_active : 0;
            item.status = realNameMap[userId] ? realNameMap[userId].status : 0;
        }
    }
}


async function dealDepositOrWithdrawReturnData(resList: any) {
    if (resList.length) {
        let usdtTickerMap = {};
        for (let item of resList) {
            let symbol = item.symbol;
            let change = item.change;
            let usdt = usdtTickerMap[symbol] ? usdtTickerMap[symbol] : await tickerUtils._rateCoin2USDT(symbol);
            usdtTickerMap[symbol] = usdt;
            item.equal_usdt = new BigNumber(usdt).mul(change);
        }
    }
}

async function dealDepositAndWithdrawReturnData(resList: any) {
    if (resList.length) {
        let usdtTickerMap = {};
        for (let item of resList) {
            let symbol = item.symbol;
            let deposit_amount = item.deposit_amount;
            let withdraw_amount = item.withdraw_amount;
            let usdt = usdtTickerMap[symbol] ? usdtTickerMap[symbol] : await tickerUtils._rateCoin2USDT(symbol);
            usdtTickerMap[symbol] = usdt;
            item.deposit_usdt = new BigNumber(usdt).mul(deposit_amount);
            item.withdraw_usdt = new BigNumber(usdt).mul(withdraw_amount);
            item.deposit_withdraw_usdt = item.deposit_usdt.sub(item.withdraw_usdt);
        }
    }
}







