import {
    madAdminOrmDB, coinAddress, coinType, abkFeeStatistics, ormDB, coinWithdraw,
    mdUserDailyFee, partnerFullUser, syncExFill, dwdExOrder
} from "@madex/ex-ts-dao";
import BigNumber from "bignumber.js";
import { any } from "async";
import { it } from "node:test";


const Otplib = require('otplib');

let { logger, apiAssertUtils: ApiAssert, BigNumberUtils, datetimeUtils } = require('@madex/ex-js-public');
let { authCommon: AuthCommon, redisUtilsCommon: RedisClient, tickerUtils } = require('@madex/ex-js-common');

let _ = require('lodash');

/**
 * 首页数据概览
 * @param from_time
 * @param to_time
 */
export async function gatherData(from_time: any, to_time: any) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }]
    }

    //手续费和返佣
    let task1 = abkFeeStatistics.prototype.findAll({
        attributes: ['user_type',
            ormDB.literal('sum(spot_fee_usdt) as spot_fee_usdt'),
            ormDB.literal('sum(lpc_fee_usdt) as lpc_fee_usdt'),
            ormDB.literal('sum(ipc_fee_usdt) as ipc_fee_usdt'),
            ormDB.literal('sum(spot_back_usdt) as spot_back_usdt'),
            ormDB.literal('sum(lpc_back_usdt) as lpc_back_usdt'),
            ormDB.literal('sum(ipc_back_usdt) as ipc_back_usdt')
        ],
        where: where,
        group: ['user_type'],
        order: [['user_type', 'asc']],
        raw: true
    });

    let [dbFeeList, {
        withdraw_total_fee,
        withdraw_common_user_fee,
        withdraw_crm_user_fee
    }] = await Promise.all([task1, getTotalWithdrawFeeByDtGroup(from_time, to_time)]);

    //普通用户数据
    let commonData = dbFeeList[0] ? dbFeeList[0] : {};
    //代理用户（crm）用户数据
    let crmData = dbFeeList[1] ? dbFeeList[1] : {};

    let spot_data = buildOneResult(commonData.spot_fee_usdt, commonData.spot_back_usdt, crmData.spot_fee_usdt, crmData.spot_back_usdt);
    let lpc_data = buildOneResult(commonData.lpc_fee_usdt, commonData.lpc_back_usdt, crmData.lpc_fee_usdt, crmData.lpc_back_usdt);
    let ipc_data = buildOneResult(commonData.ipc_fee_usdt, commonData.ipc_back_usdt, crmData.ipc_fee_usdt, crmData.ipc_back_usdt);

    let all_common_fee = new BigNumber(commonData.spot_fee_usdt).add(new BigNumber(commonData.lpc_fee_usdt)).add(new BigNumber(commonData.ipc_fee_usdt)).add(withdraw_common_user_fee);
    let all_common_back = new BigNumber(commonData.spot_back_usdt).add(new BigNumber(commonData.lpc_back_usdt)).add(new BigNumber(commonData.ipc_back_usdt));

    let all_crm_fee = new BigNumber(crmData.spot_fee_usdt).add(new BigNumber(crmData.lpc_fee_usdt)).add(new BigNumber(crmData.ipc_fee_usdt)).add(withdraw_crm_user_fee);
    let all_crm_back = new BigNumber(crmData.spot_back_usdt).add(new BigNumber(crmData.ipc_back_usdt)).add(new BigNumber(crmData.ipc_back_usdt));

    let all_data = buildOneResult(all_common_fee, all_common_back, all_crm_fee, all_crm_back);
    //总手续费+提现手续费
    all_data.total_fee = all_data.total_fee.add(withdraw_total_fee);
    return {
        all_data,
        spot_data,
        lpc_data,
        ipc_data,
        withdraw_data: withdraw_total_fee
    }

}

/**
 * 手续费数据
 * @param from_time
 * @param to_time
 * @param page
 * @param size
 * @param type  1 现货 2 U本位合约 3 币本位合约 4 全部（包含提现）
 */
export async function getFeeData(from_time: any, to_time: any, page: any, size: any, type: number) {
    let [total_obj, pageData] = await Promise.all([getTotalFeeDataOfPeriodByType(from_time, to_time, type),
        getTotalFeePageDataOfPeriodByType(page, size, from_time, to_time, type)]);
    //查询全部时要查询提现手续费
    if (type == 4) {
        let { withdrawDtFeeMap, withdraw_total_fee, withdraw_common_user_fee, withdraw_crm_user_fee } = await getTotalWithdrawFeeByDtGroup(from_time, to_time);
        //加上提现手续费
        total_obj.total_fee = total_obj.total_fee.add(withdraw_total_fee);
        total_obj.common_user.fee = total_obj.common_user.fee.add(withdraw_common_user_fee);
        total_obj.crm_user.fee = total_obj.crm_user.fee.add(withdraw_crm_user_fee);
        for (let item of pageData.rows) {
            let tradeDate = item.trade_date;
            let userType = item.user_type;
            let withdrawFeeObj = withdrawDtFeeMap[tradeDate];
            //普通用户
            if (userType == 1) {
                item.total_fee_usdt = new BigNumber(item.total_fee_usdt).add(withdrawFeeObj ? withdrawFeeObj.common_user_fee : 0);
            }
            else {
                item.total_fee_usdt = new BigNumber(item.total_fee_usdt).add(withdrawFeeObj ? withdrawFeeObj.crm_user_fee : 0);
            }
        }
    }
    pageData["total_data"] = total_obj;

    return pageData;
}

/**
 * 手续费数据详情
 * @param from_time
 * @param to_time
 * @param page
 * @param size
 * @param type  1 现货 2 U本位合约 3 币本位合约 4 全部(不含提现，提现单独接口)
 * @param user_type  用户类型:1:普通用户;2:代理用户(crm用户)
 * @param user_id
 */
export async function getFeeDataDetails(from_time: any, to_time: any, page: any, size: any, type: number, user_type: number, user_id: number) {
    if (!to_time) {
        to_time = datetimeUtils.add(from_time, datetimeUtils.DAY);
    }
    //查询crm用户
    let crmUserIds = await getCrmUserIds();
    //uid 存在 且查询的是 crm用户的数据 但是 uid 是普通用户
    if (user_id && user_type == 2 && !crmUserIds.includes(user_id)) {
        return {
            count: 0,
            rows: []
        }
    }
    //uid 存在 且查询的是 普通用户的数据 但是 uid 是crm用户
    if (user_id && user_type == 1 && crmUserIds.includes(user_id)) {
        return {
            count: 0,
            rows: []
        }
    }
    //查询当日交易的所有用户
    let tradeUserIds = await getTradeUserIds(from_time, to_time, type, user_id);
    if (!tradeUserIds.length) {
        return {
            count: 0,
            rows: []
        }
    }
    let cond_common_user_ids: any = [];
    let cond_crm_user_ids: any = [];
    for (let item of tradeUserIds) {
        if (crmUserIds.includes(item)) {
            cond_crm_user_ids.push(item);
        }
        else {
            cond_common_user_ids.push(item);
        }
    }

    let cond_user_ids = user_type == 1 ? cond_common_user_ids : cond_crm_user_ids;

    if (!cond_user_ids.length) {
        return {
            count: 0,
            rows: []
        }
    }
    //查询交易详情
    let res = await getTradeDetails(cond_user_ids, from_time, to_time, type, page, size);
    return res;

}

/**
 * 提现手续费数据
 * @param from_time
 * @param to_time
 * @param page
 * @param size
 */
export async function withdrawData(from_time: any, to_time: any, page: any, size: any) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: to_time },
            }],
        status: 3
    }


    let task2 = coinWithdraw.prototype.findAndCount({
        attributes: [ormDB.literal('DATE_FORMAT(`createdAt`,\'%Y-%m-%d 00:00:00\') as dt'), 'coin_symbol',
            ormDB.literal('sum(fee) as fee'),
            ormDB.literal('sum(amount) as amount'), 'status'],
        where: where,
        limit: Number(size),
        offset: (Number(page) - 1) * Number(size),
        order: [[ormDB.literal('DATE_FORMAT(`createdAt`,\'%Y-%m-%d 00:00:00\')'), 'desc']],
        group: [ormDB.literal('DATE_FORMAT(`createdAt`,\'%Y-%m-%d 00:00:00\')'), 'coin_symbol'],
        raw: true
    });
    let res: any;
    let [{ total_withdraw_fee, tickerMap }, pageData] = await Promise.all([getTotalWithdrawFeeByCondition(where), task2]);
    res = pageData;
    res.count = res.count.length;
    for (let item of res.rows) {
        item.amount_usdt = new BigNumber(item.amount).mul(new BigNumber(tickerMap[item.coin_symbol] || 0));
        item.fee_usdt = new BigNumber(item.fee).mul(new BigNumber(tickerMap[item.coin_symbol] || 0));
    }
    res['total_withdraw_fee'] = total_withdraw_fee;

    return res;
}


/**
 * 提现手续费数据详情
 * @param from_time
 * @param to_time
 * @param page
 * @param size
 * @param user_id
 */
export async function withdrawDataDetails(from_time: any, to_time: any, page: any, size: any, user_id: number) {
    if (!to_time) {
        to_time = datetimeUtils.add(from_time, datetimeUtils.DAY);
    }
    let where: any = {
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: to_time },
            }],
        status: 3
    }
    if (user_id) {
        where["user_id"] = user_id;
    }
    let res = await coinWithdraw.prototype.findAndCount({
        attributes: ['createdAt', 'user_id', 'coin_symbol', 'amount', 'fee', 'status'],
        where: where,
        limit: Number(size),
        offset: (Number(page) - 1) * Number(size),
        order: [['createdAt', 'desc']],
        raw: true
    });
    let tickerMap: any = {};

    for (let item of res.rows) {
        let usdt = tickerMap[item.coin_symbol] ? tickerMap[item.coin_symbol] : await tickerUtils.rateCoin2USDT(item.coin_symbol);
        tickerMap[item.coin_symbol] = usdt;
        let amount_usdt = new BigNumber(item.amount).mul(new BigNumber(usdt));
        let fee_usdt = new BigNumber(item.fee).mul(new BigNumber(usdt));
        item.amount_usdt = amount_usdt;
        item.fee_usdt = fee_usdt;
    }

    return res;

}

function buildOneResult(common_fee: any, common_back: any, crm_fee: any, crm_back: any) {
    return {
        total_fee: new BigNumber(common_fee || 0).add(new BigNumber(crm_fee || 0)),
        total_back: new BigNumber(common_back || 0).add(new BigNumber(crm_back || 0)),
        common_user: {
            fee: new BigNumber(common_fee || 0),
            back: new BigNumber(common_back || 0)
        },
        crm_user: {
            fee: new BigNumber(crm_fee || 0),
            back: new BigNumber(crm_back || 0),
        }
    }
}

/**
 * 根据类型查询一段时间内的总手续费
 * @param from_time
 * @param to_time
 * @param type 1 现货 2 U本位合约 3 币本位合约 4 全部
 */
async function getTotalFeeDataOfPeriodByType(from_time: any, to_time: any, type: number) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }]
    }
    let { fee_field, back_field } = getFields(type);

    let dbList = await abkFeeStatistics.prototype.findAll({
        attributes: ['user_type',
            ormDB.literal(`sum(\`${fee_field}\`) as total_fee_usdt`),
            ormDB.literal(`sum(\`${back_field}\`) as total_back_usdt`),
        ],
        where: where,
        group: ['user_type'],
        order: [['user_type', 'asc']],
        raw: true
    });

    let commonUserData = dbList[0] ? dbList[0] : {};
    let crmUserData = dbList[1] ? dbList[1] : {};

    return buildOneResult(commonUserData.total_fee_usdt, commonUserData.total_back_usdt, crmUserData.total_fee_usdt, crmUserData.total_back_usdt);
}


/**
 * 根据类型 查询一段时间内的手续费 分页数据
 * @param page
 * @param size
 * @param from_time
 * @param to_time
 * @param type 1 现货 2 U本位合约 3 币本位合约 4 全部
 */
async function getTotalFeePageDataOfPeriodByType(page: any, size: any, from_time: any, to_time: any, type: number) {
    let res: any;
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }]
    }
    let { fee_field, back_field } = getFields(type);


    res = await abkFeeStatistics.prototype.findAndCount({
        attributes: ['trade_date', 'user_type',
            ormDB.literal(`sum(\`${fee_field}\`) as total_fee_usdt`),
            ormDB.literal(`sum(\`${back_field}\`) as total_back_usdt`),
        ],
        where: where,
        limit: Number(size),
        offset: (Number(page) - 1) * Number(size),
        group: ['trade_date', 'user_type'],
        order: [['trade_date', 'desc'], ['user_type', 'asc']],
        raw: true
    });
    res.count = res.count.length;
    return res;
}

/**
 * 根据类型获取要查询的字段
 * @param type
 */
function getFields(type: number) {
    let fee_field = "";
    let back_field = "";
    switch (type) {
        case 1:
            fee_field = "spot_fee_usdt";
            back_field = "spot_back_usdt";
            break;
        case 2:
            fee_field = "lpc_fee_usdt";
            back_field = "lpc_back_usdt";
            break;
        case 3:
            fee_field = "ipc_fee_usdt";
            back_field = "ipc_back_usdt";
            break;
        default: // 这里拼接字符串 注意 ` 符号
            fee_field = "spot_fee_usdt` + `lpc_fee_usdt` + `ipc_fee_usdt";
            back_field = "spot_back_usdt` + `lpc_back_usdt` + `ipc_back_usdt";
            break;
    }

    return {
        fee_field, back_field
    }
}


/**
 * 根据条件查询总提现手续费
 * @param where
 */
async function getTotalWithdrawFeeByCondition(where: any) {
    //提现手续费
    let dbList = await coinWithdraw.prototype.findAll({
        attributes: ['coin_symbol',
            ormDB.literal('sum(fee) as fee')
        ],
        where: where,
        group: ['coin_symbol'],
        raw: true
    });
    let tickerMap = {};
    let total_withdraw_fee = new BigNumber(0);
    for (let item of dbList) {
        let usdt = await tickerUtils.rateCoin2USDT(item.coin_symbol);
        let fee_usdt = new BigNumber(item.fee).mul(new BigNumber(usdt));
        total_withdraw_fee = total_withdraw_fee.add(fee_usdt);
        tickerMap[item.coin_symbol] = usdt;
    }
    return {
        total_withdraw_fee, tickerMap
    }
}

/**
 * 获取某个日期交易过的用户
 * @param from_time
 * @param to_time
 * @param type 交易类型:1:现货;2:U本位合约;3:币本位合约
 * @param user_id
 */
async function getTradeUserIds(from_time: any, to_time: any, type: number, user_id: number) {
    let where: any = {
        [ormDB.Op.and]: [
            {
                trade_date: { [ormDB.Op.gte]: from_time }
            },
            {
                trade_date: { [ormDB.Op.lt]: to_time },
            }],
    }
    if (type != 4) {
        where["type"] = type;
    }
    if (user_id) {
        where["user_id"] = user_id;
    }

    let dbList = await mdUserDailyFee.prototype.findAll({
        attributes: [ormDB.literal('DISTINCT(user_id) as user_id')],
        where: where,
        raw: true
    });
    return dbList.map(item => item.user_id);
}

/**
 * 查询所有 crm 用户
 */
async function getCrmUserIds() {
    let dbList = await partnerFullUser.prototype.findAll({
        attributes: ['b_user_id'],
        where: {},
        raw: true
    });
    return dbList.map(item => item.b_user_id);
}


async function getTradeDetails(user_ids: any, from_time: any, to_time: any, type: number, page: any, size: any) {
    let market = "";
    switch (type) {
        case 1:
            market = "spot";
            break;
        case 2:
            market = "lpc";
            break;
        case 3:
            market = "ipc";
            break;
        default:
            market = "";
            break;
    }
    let where: any = {
        [ormDB.Op.and]: [
            {
                time: { [ormDB.Op.gte]: new Date(from_time).getTime() }
            },
            {
                time: { [ormDB.Op.lt]: new Date(to_time).getTime() },
            }],
        user_id: user_ids,
    }
    if (market) {
        where["market"] = market;
    }
    let res = await syncExFill.prototype.findAndCount({
        attributes: ['user_id', 'order_id', 'time', 'product', 'quantity', 'is_taker', 'fee_symbol', 'fee', 'market'],
        where: where,
        limit: Number(size),
        offset: (Number(page) - 1) * Number(size),
        order: [['time', 'desc']],
        raw: true
    });
    if (res.rows.length) {
        //查买卖、多空
        let orderIds = res.rows.map(item => item.order_id);
        let dbOrders = await dwdExOrder.prototype.findAll({
            attributes: ['order_id', 'side', 'action'],
            where: {
                order_id: orderIds
            },
            raw: true
        });
        let orderMap: any = {};
        for (let item of dbOrders) {
            orderMap[item.order_id] = item;
        }
        let tickerMap: any = {};
        for (let item of res.rows) {
            item.side = orderMap[item.order_id] ? orderMap[item.order_id].side : "";//方向
            item.action = orderMap[item.order_id] ? orderMap[item.order_id].action : "";//多空

            let coin_symbol = item.product.split('_')[0];
            let coin_symbol_usdt = tickerMap[coin_symbol] ? tickerMap[coin_symbol] : await tickerUtils.rateCoin2USDT(coin_symbol);
            let quantity_usdt = new BigNumber(item.quantity).mul(new BigNumber(coin_symbol_usdt));
            tickerMap[coin_symbol] = coin_symbol_usdt;

            let fee_symbol_usdt = tickerMap[item.fee_symbol] ? tickerMap[item.fee_symbol] : await tickerUtils.rateCoin2USDT(item.fee_symbol);
            let fee_usdt = new BigNumber(item.fee).mul(new BigNumber(fee_symbol_usdt));
            tickerMap[item.fee_symbol] = fee_symbol_usdt;

            item.quantity_usdt = quantity_usdt;
            item.fee_usdt = fee_usdt;
        }
    }


    return res;
}

/**
 * 查询总提现手续费按日期分组
 * @param from_time
 * @param to_time
 */
async function getTotalWithdrawFeeByDtGroup(from_time: any, to_time: any) {
    let condition: any = {
        [ormDB.Op.and]: [
            {
                createdAt: { [ormDB.Op.gte]: from_time }
            },
            {
                createdAt: { [ormDB.Op.lt]: to_time },
            }],
        status: 3//发币完成
    }
    //提现手续费
    let task1 = coinWithdraw.prototype.findAll({
        attributes: [ormDB.literal('DATE_FORMAT(`createdAt`,\'%Y-%m-%d 00:00:00\') as dt'), 'user_id', 'coin_symbol',
            ormDB.literal('sum(fee) as fee')
        ],
        where: condition,
        group: ['dt', 'user_id', 'coin_symbol'],
        raw: true
    });
    let tickerMap = {};

    let task2 = getCrmUserIds();
    let [dbList, crmUserIds] = await Promise.all([task1, task2]);

    let groupMap = _.groupBy(dbList, 'dt');
    let dtKeys = _.keys(groupMap);

    let dtMap: any = {};

    //普通用户提现手续费
    let total_common_user_fee = new BigNumber(0);
    //crm用户提现手续费
    let total_crm_user_fee = new BigNumber(0);

    for (let dtKey of dtKeys) {
        let oneList = groupMap[dtKey];
        //每日统计
        let dt_common_user_total_fee = new BigNumber(0);
        let dt_crm_user_total_fee = new BigNumber(0);

        for (let item of oneList) {
            let usdt = tickerMap[item.coin_symbol] ? tickerMap[item.coin_symbol] : await tickerUtils.rateCoin2USDT(item.coin_symbol);
            tickerMap[item.coin_symbol] = usdt;

            let fee_usdt = new BigNumber(item.fee).mul(new BigNumber(usdt));
            if (crmUserIds.includes(item.user_id)) {
                dt_crm_user_total_fee = dt_crm_user_total_fee.add(fee_usdt);
            }
            else {
                dt_common_user_total_fee = dt_common_user_total_fee.add(fee_usdt);
            }
        }
        dtMap[dtKey] = {
            common_user_fee: dt_common_user_total_fee,
            crm_user_fee: dt_crm_user_total_fee,
        };
        total_common_user_fee = total_common_user_fee.add(dt_common_user_total_fee);
        total_crm_user_fee = total_crm_user_fee.add(dt_crm_user_total_fee);
    }

    return {
        withdrawDtFeeMap: dtMap,
        withdraw_total_fee: total_common_user_fee.add(total_crm_user_fee),
        withdraw_common_user_fee: total_common_user_fee,
        withdraw_crm_user_fee: total_crm_user_fee
    };
}


